﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief Endpoint function class
 *
 * @details Endpoint functions are those which implement
 *          application specific functionality (such as Ethernet
 *          controllers, SATA controller, etc.).
 *          EndpointFunction class is-a-kind-of PCI function and
 *          thus inherits the more generic Function class.
 */

namespace nn { namespace pcie { namespace driver { namespace detail {

class EndpointFunction : public Function
{
public:
    struct SearchCriteria
    {
        FunctionSelectMask    selectMask;
        uint8_t               baseClassCode;
        uint8_t               subClassCode;
        uint16_t              vendorId;
        uint16_t              deviceId;
    };
    static const SearchCriteria WildcardSearchCriteria;

    EndpointFunction(Driver *pDriver, Device *pDevice, Function::Profile *pProfile);
    ~EndpointFunction();


    Result Initialize();
    Result Finalize();
    void Display();
    static EndpointFunction* GetEndpointFunction(Driver *pDriver, FunctionHandle handle);
    FunctionHandle GetHandle();
    Result SetAssignedClient(ClassDriverHandle handle);
    ClassDriverHandle GetAssignedClient();
    bool IsClientAssigned();
    int32_t GetMaxNumMsiVectors();
    void SetDMAEnable(bool busMasterEnabled);
    Result CreateDmaMapping(nn::dd::ProcessHandle procHandle, uint64_t procVa, size_t size,
                            nn::dd::MemoryPermission cpuPermission, nn::dd::MemoryPermission devicePermission,
                            BusAddress *pReturnedBusAddressBase);
    Result DestroyDmaMappingByProcVa(nn::dd::ProcessHandle procHandle, uint64_t procVa);
    Result DestroyDmaMappingByBusAddress(BusAddress busAddr);
    Result GetDmaBusAddress(nn::dd::ProcessHandle processHandle, uintptr_t procVa, BusAddress *pReturnedBusAddressBase);
    Result GetDmaBusAddressRange(BusAddress *pOutBase, BusAddress *pOutSize);
    Result AcquireIrq(IrqCallback callback, uintptr_t context,
                      IrqType irqType, int32_t numberOfVectors);
    Result ReleaseIrq();
    Result SetIrqEnable(int32_t zeroBasedIrqNumber, bool irqEnabled);
    void HandleIrq(IrqType irqType, int32_t offset);
    bool EvaluateSearchCriteria(const SearchCriteria *pCriteria, bool isAcquiredIncluded);
    void GetState(FunctionState* pState);
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;
    static int32_t FindEndpointFunctions(FunctionHandle *pOutHandles,
                                         Driver *pDriver,
                                         int32_t maxReturned,
                                         const SearchCriteria *pCriteria,
                                         bool isAcquiredIncluded);
    bool GetResetUponResumeEnable();
    void SetResetUponResumeEnable(bool isResetUponResume);

    nn::util::IntrusiveListNode m_EndpointFuncListNode;

private:
    Result AcquireMsiIrq(int32_t numberOfVectors);
    Result AcquireIntxIrq();
    int32_t GetNormalizedHandleIndex();


    Driver*             m_pDriver;
    FunctionHandle      m_Handle;
    ClassDriverHandle   m_Client;

    // IRQ Handling
    IrqType             m_IrqType;
    IrqHandle           m_IrqHandle;
    IrqCallback         m_IrqCallback;
    uintptr_t           m_IrqContext;

    // INTx
    uint8_t             m_IntPin;

    // MSI
    int32_t             m_MaxNumMsiVectors;
    int32_t             m_NumMsiVectors;
    int32_t             m_MsiCapOffset;
    int32_t             m_MsiMaskOffset;
    bool                m_Is64BitMsi;
    bool                m_IsMsiMaskable;
    int32_t             m_PowerManagementCapOffset;

    // DMA
    //DmaMapManager          *m_pDmaMapManager;
};



} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn



