﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

EndpointFunction::EndpointFunction(Driver *pDriver, Device *pDevice, Function::Profile *pProfile)
    : Function(pDriver, pDevice, pProfile)
    , m_pDriver(pDriver)
    , m_Handle(nn::pcie::InvalidFunctionHandle)
    , m_Client(nn::pcie::InvalidClassDriverHandle)
    , m_IrqType(IrqType_Invalid)
    , m_IrqHandle(-1)
    , m_IrqCallback(nullptr)
    , m_IrqContext(0)
    , m_IntPin(0)
    , m_MaxNumMsiVectors(0)
    , m_NumMsiVectors(0)
    , m_MsiCapOffset(0)
    , m_MsiMaskOffset(0)
    , m_Is64BitMsi(false)
    , m_IsMsiMaskable(false)
    , m_PowerManagementCapOffset(0)
{
    SetFunctionType(Function::FunctionType_Endpoint);
}

EndpointFunction::~EndpointFunction()
{

}

const EndpointFunction::SearchCriteria EndpointFunction::WildcardSearchCriteria =
{0, 0, 0, 0, 0};

Result EndpointFunction::Initialize()
{
    Result result = ResultSuccess();
    uint16_t cmdVal;
    Client::IrqParameters irqParameters = Client::InvalidIrqParameters;
    FunctionHandle reservedFunctionHandle = InvalidFunctionHandle;
    ClassDriverHandle reservedClient = InvalidClassDriverHandle;
    NN_PCIE_RETURN_UPON_ERROR(Function::Initialize());
    if(Client::LookupReservedEndpointFunction(&reservedClient, &reservedFunctionHandle, &irqParameters,
                                              m_pDriver, m_pBus->GetBusNum(), GetDevNum(),
                                              GetFuncNum(), GetClassCode(), GetVendorId(),
                                              GetDeviceId()).IsSuccess())
    {
        m_Handle = reservedFunctionHandle;
        m_Client = reservedClient;
        NN_PCIE_ABORT_UPON_ERROR(m_pDriver->m_EndpointFunctionHandleManager.Assign(this, m_Handle));
    }
    else
    {
        NN_PCIE_ABORT_UPON_ERROR(m_pDriver->m_EndpointFunctionHandleManager.Assign(this, &m_Handle));
    }

    // may or may not support MSI
    m_MaxNumMsiVectors = 0;
    if ((m_MsiCapOffset = FindCapability(CapabilityId_Msi)) > 0)
    {
        uint16_t msiRegVal = 0;
        m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_Flags, &msiRegVal);
        m_MaxNumMsiVectors = 1 << ((msiRegVal & MsiValue_FlagsQMask) >> 1);
        m_MaxNumMsiVectors = (m_MaxNumMsiVectors < nn::pcie::MaxIrqVectorsPerDeviceFunction) ?
            m_MaxNumMsiVectors : nn::pcie::MaxIrqVectorsPerDeviceFunction;
        m_Is64BitMsi = (msiRegVal & MsiValue_Flags64Bit) ? true : false;
        m_IsMsiMaskable = (msiRegVal & MsiValue_FlagsMaskBit) ? true : false;
        m_MsiMaskOffset = m_MsiCapOffset + ((m_Is64BitMsi) ? MsiOffset_Mask64 : MsiOffset_Mask32);
    }

    // resolve INTx pin (logical), disable for now
    m_pBus->ReadConfigByte(GetDevNum(), GetFuncNum(), Type0ConfigOffset_InterruptPin, &m_IntPin);
    m_IntPin = (m_IntPin > Type0ConfigValue_IntD) ? 0 : m_IntPin;
    m_pBus->ReadConfigWord(GetDevNum(),  GetFuncNum(), StandardConfigOffset_Command, &cmdVal);
    cmdVal |= StandardConfigValue_CommandIntxDisable;
    m_pBus->WriteConfigWord(GetDevNum(),  GetFuncNum(), StandardConfigOffset_Command, cmdVal);

    // resolver power management capability
    if ((m_PowerManagementCapOffset = FindCapability(CapabilityId_PowerManagement)) > 0)
    {
        // Force do Do state
        uint16_t pmCapability = 0;
        uint16_t pmControl = 0;
        m_pBus->ReadConfigWord(GetDevNum(),  GetFuncNum(), m_PowerManagementCapOffset + PowerManagementOffset_Control, &pmControl);
        m_pBus->WriteConfigWord(GetDevNum(),  GetFuncNum(), m_PowerManagementCapOffset + PowerManagementOffset_Control,
                (pmControl & ~PowerManagementValue_ControlStateMask) | (PmState_D0 & PowerManagementValue_ControlStateMask));

        // Display PM capability and control for informational purposes
        m_pBus->ReadConfigWord(GetDevNum(),  GetFuncNum(), m_PowerManagementCapOffset + PowerManagementOffset_Capability, &pmCapability);
        m_pBus->ReadConfigWord(GetDevNum(),  GetFuncNum(), m_PowerManagementCapOffset + PowerManagementOffset_Control, &pmControl);
        NN_PCIE_LOG_INFO("pmCapability = 0x%04x\n", pmCapability);
        NN_PCIE_LOG_INFO("pmControl    = 0x%04x\n", pmControl);
    }
    else
    {
        NN_PCIE_LOG_WARN("This endpoint function has no power management capability.");
    }

    // add this function to global list
    m_pDriver->m_EndpointFunctionList.push_back(*this);

    // was this function recovered?
    if(reservedFunctionHandle == InvalidFunctionHandle)
    {
        NN_PCIE_LOG_INFO("EndpointFunction::Initialize() handle=0x%x ",m_Handle);
    }
    else
    {
        NN_PCIE_LOG_INFO("EndpointFunction::Initialize() handle=0x%x(recovered) ",m_Handle);
        if(irqParameters.type != IrqType_Invalid)
        {
            result = EndpointFunction::AcquireIrq(irqParameters.callback, irqParameters.context,
                                                  irqParameters.type, irqParameters.numberOfVectors);
        }
    }

    if (result.IsSuccess())
    {
        // debug status
        Display();

        // inform client layer
        Client::UpdateClientFunctionState(m_pDriver, this, m_Client);
    }
    else
    {
        Function::Finalize();
        NN_PCIE_BREAK_UPON_ERROR(m_pDriver->m_EndpointFunctionHandleManager.Release(m_Handle));
        m_pDriver->m_EndpointFunctionList.erase(m_pDriver->m_EndpointFunctionList.iterator_to(*this));
    }

    return result;
}

Result EndpointFunction::Finalize()
{
    Result result;
    NN_PCIE_LOG_INFO("EndpointFunction::Finalize() - ");
    Display();

    do
    {
        // Turn off ASPM at the start of shutdown
        if(m_pRootComplex != nullptr)
        {
           m_pRootComplex->SetAspmEnable(this, false);
        }

        // Release IRQ
        ReleaseIrq();

        // Ensure DMA is off a this point
        SetDMAEnable(false);

        // Begin taking down this function
        NN_PCIE_BREAK_UPON_ERROR(Function::Finalize());
        Client::UpdateClientFunctionState(m_pDriver, this, m_Client);

        // Set device state to D3hot
        if(m_PowerManagementCapOffset > 0)
        {
            uint16_t pmControlRegVal = 0;
            m_pBus->ReadConfigWord(GetDevNum(),  GetFuncNum(), m_PowerManagementCapOffset + PowerManagementOffset_Control, &pmControlRegVal);
            m_pBus->WriteConfigWord(GetDevNum(),  GetFuncNum(), m_PowerManagementCapOffset + PowerManagementOffset_Control,
                (pmControlRegVal & ~PowerManagementValue_ControlStateMask) | (PmState_D3Hot & PowerManagementValue_ControlStateMask));
            m_pBus->ReadConfigWord(GetDevNum(),  GetFuncNum(), m_PowerManagementCapOffset + PowerManagementOffset_Control, &pmControlRegVal);
            if((pmControlRegVal & PowerManagementValue_ControlStateMask) != PmState_D3Hot)
            {
                NN_PCIE_LOG_ERROR("EndpointFunction::Finalize() - Unexpected device pmState of %s.\n",
                                  GetPmStateDescription(static_cast<PmState>(pmControlRegVal & PowerManagementValue_ControlStateMask)));

            }
        }

        NN_PCIE_BREAK_UPON_ERROR(m_pDriver->m_EndpointFunctionHandleManager.Release(m_Handle));
        m_pDriver->m_EndpointFunctionList.erase(m_pDriver->m_EndpointFunctionList.iterator_to(*this));
    }while (false);
    return result;
}

void EndpointFunction::Display()
{
    char classCodeDesc[64];
    GetClassCodeDescription(classCodeDesc, sizeof(classCodeDesc));
    NN_PCIE_LOG_NOPREFIX("BDF-%03x:%02x:%x, Class-%s, VID-0x%04x, DEVID-0x%04x, Actual/Capable Speed-%s/%s, functionHandle-0x%x\n",
                         GetBus()->GetBusNum(), GetDevNum(), GetFuncNum(),
                         classCodeDesc, GetVendorId(), GetDeviceId(),
                         GetBusSpeedDescription(GetCurrentSpeed()),
                         GetBusSpeedDescription(GetCapableSpeed()),
                         GetHandle());
}

EndpointFunction* EndpointFunction::GetEndpointFunction(Driver *pDriver, FunctionHandle handle)
{
    EndpointFunction *pFunction = pDriver->m_EndpointFunctionHandleManager.Get(handle);
    return pFunction;
}

FunctionHandle EndpointFunction::GetHandle()
{
    return m_Handle;
}

Result EndpointFunction::SetAssignedClient(ClassDriverHandle handle)
{
    Result result = ResultSuccess();
    if ((handle != nn::pcie::InvalidClassDriverHandle) &&
        (m_Client == nn::pcie::InvalidClassDriverHandle))
    {
        m_Client = handle;
    }
    else if((handle == nn::pcie::InvalidClassDriverHandle) &&
            (m_Client != nn::pcie::InvalidClassDriverHandle))
    {
        int32_t epHandleIndex = GetNormalizedHandleIndex();

        // Release IRQ when client goes away
        ReleaseIrq();

        // Teardown any remaining DMA mapping
        if(epHandleIndex >= 0)
        {
            result = m_pRootComplex->DestroyDmaMapsByTag(epHandleIndex);
        }

        m_Client = handle;
    }
    else
    {
        NN_PCIE_LOG_ERROR("EndpointFunction::SetAssignedClient() unexpected state.\n");
        result = ResultInvalidState();
    }

    return result;
}

ClassDriverHandle EndpointFunction::GetAssignedClient()
{
    return m_Client;
}

bool EndpointFunction::IsClientAssigned()
{
    return ((m_Client != nn::pcie::InvalidClassDriverHandle) ? true : false);
}

int32_t EndpointFunction::GetMaxNumMsiVectors()
{
    return m_MaxNumMsiVectors;
}

void EndpointFunction::SetDMAEnable(bool busMasterEnabled)
{
    uint16_t cmdVal = 0;
    m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), StandardConfigOffset_Command, &cmdVal);
    if (busMasterEnabled)
    {
        cmdVal |= StandardConfigValue_CommandMaster;
    }
    else
    {
        cmdVal &= ~StandardConfigValue_CommandMaster;
    }
    m_pBus->WriteConfigWord(GetDevNum(), GetFuncNum(), StandardConfigOffset_Command, cmdVal);
}

Result EndpointFunction::CreateDmaMapping(nn::dd::ProcessHandle procHandle, uint64_t procVa, size_t size,
                                          nn::dd::MemoryPermission cpuPermission, nn::dd::MemoryPermission devicePermission,
                                          BusAddress *pReturnedBusAddressBase)
{
    Result result = ResultInvalidState();
    int32_t clientHandleIndex = Client::GetNormalizedHandleIndex(m_pDriver, m_Client);
    int32_t epHandleIndex = GetNormalizedHandleIndex();
    if((clientHandleIndex >= 0) && (epHandleIndex >= 0))
    {
        result = m_pRootComplex->CreateDmaMap(pReturnedBusAddressBase, clientHandleIndex, procHandle, procVa,
                                              size, cpuPermission, devicePermission, epHandleIndex);
    }
    return result;
}

Result EndpointFunction::DestroyDmaMappingByProcVa(nn::dd::ProcessHandle procHandle, uint64_t procVa)
{
    Result result = ResultInvalidState();
    NN_UNUSED(procHandle);
    int32_t clientHandleIndex = Client::GetNormalizedHandleIndex(m_pDriver, m_Client);
    if(clientHandleIndex >= 0)
    {
        result = m_pRootComplex->DestroyDmaMapByProcAddr(clientHandleIndex, procVa);
    }
    return result;
}

Result EndpointFunction::DestroyDmaMappingByBusAddress(BusAddress busAddr)
{
    return m_pRootComplex->DestroyDmaMapByBusAddress(busAddr);
}

Result EndpointFunction::GetDmaBusAddress(nn::dd::ProcessHandle processHandle, uintptr_t procVa, BusAddress *pReturnedBusAddress)
{
    Result result = ResultInvalidState();
    NN_UNUSED(processHandle);
    int32_t clientHandleIndex = Client::GetNormalizedHandleIndex(m_pDriver, m_Client);
    if(clientHandleIndex >= 0)
    {
        result = m_pRootComplex->GetDmaBusAddress(pReturnedBusAddress, clientHandleIndex, procVa);
    }
    return result;
}

Result EndpointFunction::GetDmaBusAddressRange(BusAddress *pOutBase, BusAddress *pOutSize)
{
    return m_pRootComplex->GetDmaBusAddressRange(pOutBase, pOutSize);
}

Result EndpointFunction::AcquireIrq(IrqCallback callback, uintptr_t context,
                                    IrqType irqType, int32_t numberOfVectors)
{
    Result result = ResultSuccess();
    if ((callback != NULL) && (numberOfVectors > 0))
    {
        // is the vector already acquired?
        if (m_IrqType != IrqType_Invalid)
        {
            return ResultAlreadyAcquired();
        }

        switch (irqType)
        {
        case IrqType_Msi:
            result = AcquireMsiIrq(numberOfVectors);
            break;
        case IrqType_IntX:
            result = AcquireIntxIrq();
            break;
        default:
            result = ResultInvalidArgument();
            break;
        }
        if (result.IsSuccess())
        {
            m_IrqContext   = context;
            m_IrqCallback  = callback;
            m_IrqType      = irqType;
        }
        else
        {
            ReleaseIrq();
        }
    }
    return result;
}

Result EndpointFunction::ReleaseIrq()
{
    Result result = ResultIrqNotAcquired();
    if (m_IrqHandle >= 0)
    {
        // free root complex managed vector
        m_pRootComplex->ReleaseIrq(m_IrqHandle);

        // disable this function's MSI
        uint16_t msiRegVal = 0;
        m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_Flags, &msiRegVal);
        msiRegVal &= ~MsiValue_FlagsEnable;
        m_pBus->WriteConfigWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_Flags, msiRegVal);

        // clear local handler information
        m_IrqHandle     = -1;
        m_IrqCallback   = NULL;
        m_IrqContext    = 0;
        m_IrqType       = IrqType_Invalid;
        m_NumMsiVectors = 0;

        result = ResultSuccess();
    }
    return result;
}

Result EndpointFunction::SetIrqEnable(int32_t zeroBasedIrqNumber, bool irqEnabled)
{
    Result result = ResultInvalidArgument();

    if (m_IrqType == IrqType_Msi)
    {
        if ((zeroBasedIrqNumber < m_NumMsiVectors) && (zeroBasedIrqNumber >= 0))
        {
            bool updateMasterMsiEnable = true;

            // enable/disable this function's MSI
            if (m_IsMsiMaskable)
            {
                uint32_t mask32;
                m_pBus->ReadConfigDWord(GetDevNum(), GetFuncNum(), m_MsiMaskOffset, &mask32);
                mask32 &= ~(1 << zeroBasedIrqNumber);
                mask32 |= ((irqEnabled) ? (1 << zeroBasedIrqNumber) : 0);
                m_pBus->WriteConfigDWord(GetDevNum(), GetFuncNum(), m_MsiMaskOffset, mask32);
                if (!irqEnabled && (mask32 != 0))
                {
                    updateMasterMsiEnable = false;
                }
            }

            if (updateMasterMsiEnable)
            {
                uint16_t msiRegVal = 0;
                m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_Flags, &msiRegVal);
                msiRegVal &= ~MsiValue_FlagsEnable;
                msiRegVal |= ((irqEnabled) ? MsiValue_FlagsEnable : 0);
                m_pBus->WriteConfigWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_Flags, msiRegVal);
            }

            result = ResultSuccess();
        }
    }
    else if (m_IrqType == IrqType_IntX)
    {
        // Only one IRQ allowed for legacy INTx
        if (zeroBasedIrqNumber == 0)
        {
            uint16_t cmdVal;
            m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), StandardConfigOffset_Command, &cmdVal);
            cmdVal &= ~(StandardConfigValue_CommandIntxDisable);
            cmdVal |= ((irqEnabled) ? 0 : StandardConfigValue_CommandIntxDisable);
            m_pBus->WriteConfigWord(GetDevNum(), GetFuncNum(), StandardConfigOffset_Command, cmdVal);
            result = ResultSuccess();
        }
    }
    else
    {
        result = ResultIrqNotAcquired();
    }

    if (result.IsSuccess())
    {
        // enable/disable within root complex
        result = m_pRootComplex->SetIrqEnable(m_IrqHandle, zeroBasedIrqNumber, irqEnabled);
    }

    return result;
}

void EndpointFunction::HandleIrq(IrqType irqType, int32_t offset)
{
    bool dispatch = true;
    NN_UNUSED(offset);
    switch (irqType)
    {
    case IrqType_IntX:
        uint16_t regVal16;

        // legacy INT can have multiple functions on the same vector,
        // so we must check if this interrupt is really for this function
        m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), StandardConfigOffset_Status, &regVal16);
        if (!(regVal16 & StandardConfigValue_StatusInterrupt))
        {
            dispatch = false;
        }

        // legacy INT is level based - we must disable the vector until user re-arms it later
        m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), StandardConfigOffset_Command, &regVal16);
        regVal16 |= StandardConfigValue_CommandIntxDisable;
        m_pBus->WriteConfigWord(GetDevNum(), GetFuncNum(), StandardConfigOffset_Command, regVal16);
        break;
    case IrqType_Msi:
        break;
    default:
        break;
    }

    if ((m_IrqCallback != NULL) && dispatch)
    {
        (*m_IrqCallback)(m_IrqContext, 0);
    }
}

bool EndpointFunction::EvaluateSearchCriteria(const SearchCriteria *pCriteria, bool isAcquiredIncluded)
{
    bool match = true;

    if (pCriteria->selectMask & FunctionSelect_BaseClassCode)
    {
        if (pCriteria->baseClassCode != GetBaseClassCode())
        {
            match = false;
        }
    }
    if (pCriteria->selectMask & FunctionSelect_SubClassCode)
    {
        if (pCriteria->subClassCode != GetSubClassCode())
        {
            match = false;
        }
    }
    if (pCriteria->selectMask & FunctionSelect_VendorId)
    {
        if (pCriteria->vendorId != GetVendorId())
        {
            match = false;
        }
    }
    if (pCriteria->selectMask & FunctionSelect_DeviceId)
    {
        if (pCriteria->deviceId != GetDeviceId())
        {
            match = false;
        }
    }

    if(IsClientAssigned() && !isAcquiredIncluded)
    {
        match = false;
    }

    return match;
}

void EndpointFunction::GetState(FunctionState* pState)
{
    pState->functionHandle    = GetHandle();
    pState->busNum            = GetBus()->GetBusNum();
    pState->deviceNum         = GetDevNum();
    pState->funcNum           = GetFuncNum();
    pState->vendorId          = GetVendorId();
    pState->deviceId          = GetDeviceId();
    pState->headerType        = GetHeaderType();
    pState->classCode         = GetClassCode();
    pState->revision          = GetRevision();
    pState->speed             = GetCurrentSpeed();
    pState->maxNumMsiVectors  = GetMaxNumMsiVectors();
    pState->pmState           = GetPmState();
    pState->isAcquired        = IsClientAssigned();
}

void* EndpointFunction::operator new(size_t size) NN_NOEXCEPT
{
    return nn::pcie::detail::MemoryAllocAligned(size, MinDataAlignmentSize, "EndpointFunction");
}

void EndpointFunction::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::pcie::detail::MemoryFree(p, "EndpointFunction");
}

// Static method
int32_t EndpointFunction::FindEndpointFunctions(FunctionHandle *pOutHandles,
                                                Driver *pDriver,
                                                int32_t maxReturned,
                                                const SearchCriteria *pCriteria,
                                                bool isAcquiredIncluded)
{
    int32_t funcCount = 0;
    for (Driver::EndpointFunctionListType::iterator it = pDriver->m_EndpointFunctionList.begin();
         (it != pDriver->m_EndpointFunctionList.end()) && (funcCount < maxReturned); it++)
    {
        EndpointFunction *pEpFunc = &(*it);
        if (pEpFunc->EvaluateSearchCriteria(pCriteria, isAcquiredIncluded))
        {
            pOutHandles[funcCount] = it->GetHandle();
            funcCount++;
        }
    }
    return funcCount;
}

Result EndpointFunction::AcquireMsiIrq(int32_t numberOfVectors)
{
    Result result = ResultSuccess();
    if (numberOfVectors <= m_MaxNumMsiVectors)
    {
        RootComplex::IrqProfile irqProfile;
        RootComplex::IrqOptions irqOptions = { .irqType = IrqType_Msi };
        numberOfVectors = (numberOfVectors==0) ? 1 : numberOfVectors;
        result = m_pRootComplex->AcquireIrq(this, &irqOptions, &irqProfile);
        if (result.IsSuccess())
        {
            // setup MSI address and data
            m_pBus->WriteConfigDWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_AddressLo,
                                     (uint32_t)irqProfile.type.msi.messageAddress);
            if (m_Is64BitMsi)
            {
                m_pBus->WriteConfigDWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_AddressHi,
                                         (uint32_t)(irqProfile.type.msi.messageAddress >> 32));
                m_pBus->WriteConfigWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_Data64,
                                        irqProfile.type.msi.messageData);
            }
            else
            {
                m_pBus->WriteConfigWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_Data32,
                                        irqProfile.type.msi.messageData);
            }

            // begin with mask clear
            if (m_IsMsiMaskable)
            {
                m_pBus->WriteConfigDWord(GetDevNum(), GetFuncNum(), m_MsiMaskOffset, 0);
            }


            // disable MSI for now, setup Irq count
            uint16_t msiRegVal = 0;
            m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_Flags, &msiRegVal);
            msiRegVal &= ~MsiValue_FlagsQSize;
            msiRegVal &= ~MsiValue_FlagsEnable;
            msiRegVal |= ((FindLastSetBit(numberOfVectors) - 1) << MsiValue_FlagsQSizeShift);
            m_pBus->WriteConfigWord(GetDevNum(), GetFuncNum(), m_MsiCapOffset + MsiOffset_Flags, msiRegVal);

            // mark as acquired
            m_IrqHandle = irqProfile.irqHandle;
            m_NumMsiVectors = numberOfVectors;
        }
    }
    else
    {
        result = ResultHardwareUnsupported();
    }
    return result;
}

Result EndpointFunction::AcquireIntxIrq()
{
    Result result;
    RootComplex::IrqProfile irqProfile;
    RootComplex::IrqOptions irqOptions = { .irqType = IrqType_IntX };
    irqOptions.type.intx.intPin = m_IntPin;
    if ((result = m_pRootComplex->AcquireIrq(this, &irqOptions, &irqProfile)).IsSuccess())
    {
        m_IrqHandle = irqProfile.irqHandle;
    }
    return result;
}

int32_t EndpointFunction::GetNormalizedHandleIndex()
{
    return m_pDriver->m_EndpointFunctionHandleManager.GetNormalizedIndex(m_Handle);
}

bool EndpointFunction::GetResetUponResumeEnable()
{
    return m_pDevice->GetResetUponResumeEnable();
}

void EndpointFunction::SetResetUponResumeEnable(bool isResetUponResume)
{
    m_pDevice->SetResetUponResumeEnable(isResetUponResume);
}

} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn
