﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver { namespace detail {


Device::Device(Driver *pDriver, DeviceNumber devNum, Bus *pBus)
    : m_pDriver(pDriver)
    , m_DevNum(devNum)
    , m_pBus(pBus)
    , m_NumFunc(0)
    , m_IsResetUponResume(true)
{

}

Device::~Device()
{

}

Result Device::Initialize()
{
    Result result = ResultSuccess();
    m_pBus->AttachDevice(this);
    return result;
}

Result Device::Finalize()
{
    Result result;
    if ((result = m_pBus->DetachDevice(m_DevNum)).IsSuccess())
    {
        // destroy any remaining functions
        while ( !m_FuncList.empty())
        {
            Function *pFunc = &m_FuncList.front();
            pFunc->Finalize();
            delete pFunc;
        }
    }
    return result;
}

DeviceNumber Device::GetDevNum()
{
    return m_DevNum;
}

void Device::AddFunction(Function &func)
{
    m_FuncList.push_back(func);
    m_NumFunc++;
}

Result Device::RemoveFunction(FunctionNumber funcNum)
{
    Result result = ResultNoDevice();
    for (FuncListType::iterator it = m_FuncList.begin(); it != m_FuncList.end(); it++)
    {
        if (it->GetFuncNum() == funcNum)
        {
            m_NumFunc--;
            m_FuncList.erase(it);
            result = ResultSuccess();
            break;
        }
    }
    return result;
}

int32_t Device::GetNumFunctions()
{
    return m_NumFunc;
}

Function* Device::GetFunction(FunctionNumber funcNum)
{
    Function *pFunc = NULL;
    for (FuncListType::iterator it = m_FuncList.begin(); it != m_FuncList.end(); it++)
    {
        if (it->GetFuncNum() == funcNum)
        {
            pFunc = &(*it);
            break;
        }
    }
    return pFunc;
}

Bus* Device::GetBus()
{
    return m_pBus;
}

bool Device::GetResetUponResumeEnable()
{
    return m_IsResetUponResume;
}

void Device::SetResetUponResumeEnable(bool isResetUponResume)
{
    m_IsResetUponResume = isResetUponResume;
}

void* Device::operator new(size_t size) NN_NOEXCEPT
{
    return nn::pcie::detail::MemoryAllocAligned(size, MinDataAlignmentSize, "Device");
}

void Device::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::pcie::detail::MemoryFree(p, "Device");
}


} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn




