﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

Bus::Bus(Driver *pDriver, Bus *pUpStreamBus, BridgeFunction *pUpStreamBridgeFunction, BusNumber busNum)
    : m_pDriver(pDriver)
    , m_pUpStreamBus(pUpStreamBus)
    , m_pUpStreamBridgeFunction(pUpStreamBridgeFunction)
    , m_BusNum(busNum)
    , m_IsRoot(false)
    , m_pRootComplex(nullptr)
{

}

Bus::~Bus()
{

}

Result Bus::Initialize()
{
    Bus *pBus = m_pUpStreamBus;
    Bus *pRootBus = NULL;
    m_IsRoot = (m_pUpStreamBus == NULL) ? true : false;
    while (pBus != NULL)
    {
        pRootBus = pBus;
        pBus = pRootBus->GetUpStreamBus();
    }
    if (pRootBus != NULL)
    {
        m_pRootComplex = pRootBus->m_pRootComplex;
    }
    if ( !m_IsRoot && (m_BusNum < MaxBusCount))
    {
        m_pDriver->m_BusTable[m_BusNum] = this;
    }
    return ResultSuccess();
}

Result Bus::Finalize()
{
    // Delete any devices still on the bus
    while ( !m_DevList.empty())
    {
        Device *pDevice = &m_DevList.front();
        pDevice->Finalize();
        delete pDevice;
    }
    m_pDriver->m_BusTable[m_BusNum] = NULL;
    return ResultSuccess();
}

BusNumber Bus::GetBusNum()
{
    return m_BusNum;
}

bool Bus::IsRoot()
{
    return m_IsRoot;
}

Bus* Bus::GetUpStreamBus()
{
    return m_pUpStreamBus;
}

BridgeFunction* Bus::GetUpStreamBridgeFunction()
{
    return m_pUpStreamBridgeFunction;
}

BridgeFunction* Bus::GetRootBridgeFunction()
{
    Bus *pBus = this;
    BridgeFunction *pRootBridgeFunc = NULL;
    while (pBus != NULL)
    {
        BridgeFunction *pBf = pBus->GetUpStreamBridgeFunction();
        pRootBridgeFunc = (pBf != NULL) ? pBf : pRootBridgeFunc;
        pBus = pBus->GetUpStreamBus();
    }
    return pRootBridgeFunc;
}

void Bus::SetRootComplex(RootComplex *pRootComplex)
{
    m_pRootComplex = pRootComplex;
}

void Bus::GetRootComplex(RootComplex **ppRootComplex)
{
    *ppRootComplex = m_pRootComplex;
}

void Bus::AttachDevice(Device *pDevice)
{
    m_DevList.push_back(*pDevice);
}

Result Bus::DetachDevice(DeviceNumber devNum)
{

    NN_PCIE_LOG_INFO("Bus %d: DetachDevice(%d)\n", m_BusNum, devNum);
    Result result = ResultNoDevice();
    for (DevListType::iterator it = m_DevList.begin(); it != m_DevList.end(); it++)
    {
        if (it->GetDevNum() == devNum)
        {
            m_DevList.erase(it);
            result = ResultSuccess();
            break;
        }
    }
    return result;
}

Device* Bus::GetAttachedDevice(DeviceNumber devNum)
{
    Device *pDevice = NULL;
    for (DevListType::iterator it = m_DevList.begin(); it != m_DevList.end(); it++)
    {
        if (it->GetDevNum() == devNum)
        {
            pDevice = &(*it);
            break;
        }
    }
    return pDevice;
}


Result Bus::ReadConfigDWord(DeviceNumber devNum, FunctionNumber funcNum,
                            int32_t where, uint32_t *pDWord)
{
    Result result = ResultSuccess();
    if ((where & 0x3) == 0)
    {
        result = m_pRootComplex->ReadConfigSpace(m_BusNum, devNum, funcNum, where, sizeof(uint32_t), pDWord);
    }
    else
    {
        result = ResultInvalidRegisterOffset();
    }
    return result;
}

Result Bus::ReadConfigWord(DeviceNumber devNum, FunctionNumber funcNum,
                           int32_t where, uint16_t *pWord)
{
    Result result = ResultSuccess();
    if ((where & 0x1) == 0)
    {
        result = m_pRootComplex->ReadConfigSpace(m_BusNum, devNum, funcNum, where, sizeof(uint16_t), pWord);
    }
    else
    {
        result = ResultInvalidRegisterOffset();
    }
    return result;
}

Result Bus::ReadConfigByte(DeviceNumber devNum, FunctionNumber funcNum,
                           int32_t where, uint8_t *pByte)
{
    Result result;
    result = m_pRootComplex->ReadConfigSpace(m_BusNum, devNum, funcNum, where, sizeof(uint8_t), pByte);
    return result;
}

Result Bus::WriteConfigDWord(DeviceNumber devNum, FunctionNumber funcNum,
                             int32_t where, uint32_t dword)
{
    Result result = ResultSuccess();
    if ((where & 0x3) == 0)
    {
        result = m_pRootComplex->WriteConfigSpace(m_BusNum, devNum, funcNum, where, sizeof(uint32_t), dword);
    }
    else
    {
        result = ResultInvalidRegisterOffset();
    }
    return result;
}

Result Bus::WriteConfigWord(DeviceNumber devNum, FunctionNumber funcNum,
                            int32_t where, uint16_t word)
{
    Result result = ResultSuccess();
    if ((where & 0x1) == 0)
    {
        result = m_pRootComplex->WriteConfigSpace(m_BusNum, devNum, funcNum, where, sizeof(uint16_t), word);
    }
    else
    {
        result = ResultInvalidRegisterOffset();
    }
    return result;
}

Result Bus::WriteConfigByte(DeviceNumber devNum, FunctionNumber funcNum,
                            int32_t where, uint8_t byte)
{
    Result result;
    result = m_pRootComplex->WriteConfigSpace(m_BusNum, devNum, funcNum, where, sizeof(uint8_t), byte);
    return result;
}

void Bus::RegisterResource(Function::Resource *pResToRegister)
{
    bool added = false;
    AddrSpaceListType *pList;
    switch (pResToRegister->flags &
            (ResourceFlag_Io | ResourceFlag_Mem | ResourceFlag_Prefetch))
    {
    case ResourceFlag_Io:
        pList = &m_IoSpaceList;
        break;
    case ResourceFlag_Mem:
        pList = &m_MemNoPrefetchSpaceList;
        break;
    default:
        pList = &m_MemPrefetchSpaceList;
        break;
    }
    for (AddrSpaceListType::const_iterator it = pList->begin(); it != pList->end(); it++)
    {
        if (pResToRegister->size > it->size)
        {
            added = true;
            pList->insert(it, *pResToRegister);
            break;
        }
    }
    if ( !added)
    {
        (*pList).push_back(*pResToRegister);
    }
}

void Bus::UnRegisterResource(Function::Resource *pResToUnRegister)
{
    AddrSpaceListType *pList;
    switch (pResToUnRegister->flags &
            (ResourceFlag_Io | ResourceFlag_Mem | ResourceFlag_Prefetch))
    {
    case ResourceFlag_Io:
        pList = &m_IoSpaceList;
        break;
    case ResourceFlag_Mem:
        pList = &m_MemNoPrefetchSpaceList;
        break;
    default:
        pList = &m_MemPrefetchSpaceList;
        break;
    }
    pList->erase(pList->iterator_to(*pResToUnRegister));
}

Result Bus::AssignResourceAddresses(ResourceAddr *pIoAddrBase, ResourceAddr ioAddrLimit,
                                    ResourceAddr *pMemNoPrefAddrBase, ResourceAddr memNoPrefAddrLimit,
                                    ResourceAddr *pPrefMemAddrBase, ResourceAddr prefMemAddrLimit)
{
    ResourceAddr tempAddr;
    Result result = ResultSuccess();

    // Assign I/O resources
    for (AddrSpaceListType::iterator it = m_IoSpaceList.begin(); it != m_IoSpaceList.end(); it++)
    {
        Function *pFunc = it->pFunction;
        Function::Resource *pRes = &(*it);

        // Align starting resource virtual address with proper bus alignment!
        it->address = m_pDriver->AlignResourceAddressWithBus(*pIoAddrBase,
                                                             RoundupSize(it->size, MinIoResourceAlignmentSize));

        // Consume address space
        tempAddr = RoundupSize(it->size, MinIoResourceAlignmentSize) + it->address;
        if (tempAddr > ioAddrLimit)
        {
            result = ResultOutOfDeviceMemory();
            break;
        }
        *pIoAddrBase = tempAddr;

        // Configure the hardware
        pFunc->ConfigureResource(pRes);
    }
    // If nothing mapped, leave minimum gap
    if (m_IoSpaceList.empty())
    {
        if ((tempAddr = *pIoAddrBase + MinIoResourceAlignmentSize) <= ioAddrLimit)
        {
            *pIoAddrBase = tempAddr;
        }
        else
        {
            result = ResultOutOfDeviceMemory();
        }
    }

    // Assign non-prefetchable MEM resources
    for (AddrSpaceListType::iterator it = m_MemNoPrefetchSpaceList.begin(); it != m_MemNoPrefetchSpaceList.end(); it++)
    {
        Function *pFunc = it->pFunction;
        Function::Resource *pRes = &(*it);

        // Align starting resource virtual address with proper bus alignment!
        it->address = m_pDriver->AlignResourceAddressWithBus(*pMemNoPrefAddrBase,
                                                             RoundupSize(it->size, MinMemResourceAlignmentSize));

        // Consume address space
        tempAddr = RoundupSize(it->size, MinMemResourceAlignmentSize) + it->address;
        if (tempAddr > memNoPrefAddrLimit)
        {
            result = ResultOutOfDeviceMemory();
            break;
        }
        *pMemNoPrefAddrBase = tempAddr;

        // Configure the hardware
        pFunc->ConfigureResource(pRes);
    }
    // If nothing mapped, leave minimum gap
    if (m_MemNoPrefetchSpaceList.empty())
    {
        if ((tempAddr = *pMemNoPrefAddrBase + MinMemResourceAlignmentSize) <= memNoPrefAddrLimit)
        {
            *pMemNoPrefAddrBase = tempAddr;
        }
        else
        {
            result = ResultOutOfDeviceMemory();
        }
    }

    // Assign prefetchable MEM resources
    for (AddrSpaceListType::iterator it = m_MemPrefetchSpaceList.begin(); it != m_MemPrefetchSpaceList.end(); it++)
    {
        Function *pFunc = it->pFunction;
        Function::Resource *pRes = &(*it);

        // Align starting resource virtual address with proper bus alignment!
        it->address = m_pDriver->AlignResourceAddressWithBus(*pPrefMemAddrBase,
                                                             RoundupSize(it->size, MinMemResourceAlignmentSize));

        // Consume address space
        tempAddr = RoundupSize(it->size, MinMemResourceAlignmentSize) + it->address;
        if (tempAddr > prefMemAddrLimit)
        {
            result = ResultOutOfDeviceMemory();
            break;
        }
        *pPrefMemAddrBase = tempAddr;

        // Configure the hardware
        pFunc->ConfigureResource(pRes);
    }
    // If nothing mapped, leave minimum gap
    if (m_MemPrefetchSpaceList.empty())
    {
        if ((tempAddr = *pPrefMemAddrBase + MinMemResourceAlignmentSize) <= prefMemAddrLimit)
        {
            *pPrefMemAddrBase = tempAddr;
        }
        else
        {
            result = ResultOutOfDeviceMemory();
        }
    }

    return result;
}

void* Bus::operator new(size_t size) NN_NOEXCEPT
{
    return nn::pcie::detail::MemoryAllocAligned(size, MinDataAlignmentSize, "Bus");
}
void Bus::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::pcie::detail::MemoryFree(p, "Bus");
}

// static method
Bus* Bus::GetBus(Driver* pDriver, BusNumber busNum)
{
    Bus *pBus = NULL;
    if (busNum < MaxBusCount)
    {
        pBus = pDriver->m_BusTable[busNum];
    }
    return pBus;
}



} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn




