﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief Bridge device class
 *
 * @details A bridge device class is a container to hold one or
 * more instantiations of bridge functions.
 */

namespace nn { namespace pcie { namespace driver { namespace detail {

class BridgeDevice : public Device
{

public:
    BridgeDevice(Driver *pDriver, DeviceNumber devNum, Bus *pUpStreamBus);
    ~BridgeDevice(void);
    Result Initialize(void);
    Result Finalize(void);


    nn::util::IntrusiveListNode m_ListNode;


private:


};

/**
 * @file
 * @brief Bridge function class
 *
 * @details A bridge function logically connects multiple
 * independent buses, one upstream bus to one downstream bus.
 * BridgeFunction class is-a-kind-of PCI function and thus
 * inherits the more generic Function class.
 */

class BridgeFunction : public Function
{

public:
    enum BridgeType
    {
        BridgeType_Invalid,
        BridgeType_HostToPci,
        BridgeType_PciToPci,
        BridgeType_PcieToPci
    };


    BridgeFunction(Driver *pDriver, Device *pDevice, Function::Profile *pProfile,
                   Bus *pUpStreamBus, BusNumber implementedBusNum);
    ~BridgeFunction(void);


    Result Initialize(void);
    Result Finalize(void);
    void Display();
    Bus* GetDownStreamBus(void);
    Bus* GetUpStreamBus();
    BusNumber GetSecondaryBusNumber();
    BridgeDevice* GetDevice();
    Result  ConfigureSubordinate(BusNumber subBusNum,
                                 ResourceAddr *pIoAddrBase, ResourceAddr ioAddrLimit,
                                 ResourceAddr *pMemNoPrefAddrBase, ResourceAddr memNoPrefAddrLimit,
                                 ResourceAddr *pMemPrefAddrBase, ResourceAddr memPrefAddrLimit);
    Result SetDeviceEnable(bool enabled);
    Result SetLinkSpeed(BusSpeed speed);
    Result RetrainLink();
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;


    static bool IsBridgeFunction(Function *pFunc);
private:
    Device       *m_pDevice;
    BridgeType   m_BridgeType;
    BusNumber    m_PrimaryBusNum;
    BusNumber    m_SecondaryBusNum;
    BusNumber    m_SubordinateBusNum;
    Bus          *m_pUpStreamBus;
    Bus          *m_pDownStreamBus;

};



} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn




