﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

BridgeDevice::BridgeDevice(Driver *pDriver, DeviceNumber devNum, Bus *pUpStreamBus)
    : Device(pDriver, devNum, pUpStreamBus)
{

}

BridgeDevice::~BridgeDevice()
{

}

Result BridgeDevice::Initialize()
{
    Result result;
    result = Device::Initialize();
    return result;
}

Result BridgeDevice::Finalize()
{
    Result result;
    result = Device::Finalize();
    return result;
}

BridgeFunction::BridgeFunction(Driver *pDriver, Device *pDevice, Function::Profile *pProfile,
                               Bus *pUpStreamBus, BusNumber implementedBusNum)
    : Function(pDriver, pDevice, pProfile)
    , m_pDevice(pDevice)
    , m_BridgeType(BridgeType_Invalid)
    , m_PrimaryBusNum(0)
    , m_SecondaryBusNum(implementedBusNum)
    , m_SubordinateBusNum(0)
    , m_pUpStreamBus(pUpStreamBus)
    , m_pDownStreamBus(nullptr)
{
    SetFunctionType(Function::FunctionType_Bridge);
}

BridgeFunction::~BridgeFunction()
{
}

Result BridgeFunction::Initialize()
{
    Result result;
    uint32_t dWord;

    NN_PCIE_RETURN_UPON_ERROR(Function::Initialize());

    do
    {
        BridgeFunction *pUpStreamBridge;
        DeviceNumber devNum = m_pDevice->GetDevNum();
        FunctionNumber funcNum = GetFuncNum();
        m_PrimaryBusNum = m_pUpStreamBus->GetBusNum();
        m_SubordinateBusNum = static_cast<BusNumber>(MaxBusCount - 1);

        // resolve hierarchy
        if (m_pUpStreamBus->IsRoot())
        {
            m_BridgeType = BridgeType_HostToPci;
            pUpStreamBridge = this;
        }
        else
        {
            m_BridgeType = BridgeType_PciToPci;
            pUpStreamBridge = m_pUpStreamBus->GetUpStreamBridgeFunction();
        }

        if ((m_pDownStreamBus = new Bus(m_pDriver, m_pUpStreamBus, pUpStreamBridge, m_SecondaryBusNum)) == NULL)
        {
            result = ResultMemAllocFailure();
            break;
        }
        NN_PCIE_BREAK_UPON_ERROR(m_pDownStreamBus->Initialize());

        NN_PCIE_BREAK_UPON_ERROR(SetDeviceEnable(false));

        // update bus numbers, it's best to do it in a single write
        if (m_pUpStreamBus->ReadConfigDWord(devNum, funcNum, Type1ConfigOffset_PrimaryBusNumber, &dWord).IsSuccess())
        {
            dWord = (dWord & 0xff000000) | (m_SubordinateBusNum << 16) | (m_SecondaryBusNum << 8) | (m_PrimaryBusNum);
            m_pUpStreamBus->WriteConfigDWord(devNum, funcNum, Type1ConfigOffset_PrimaryBusNumber, dWord);
        }

    }while (false);

    if (result.IsSuccess())
    {
        NN_PCIE_LOG_INFO("BridgeFunction::Initialize() "); Display();
    }
    else
    {
        Function::Finalize();
    }

    return result;
}

Result BridgeFunction::Finalize()
{
    uint32_t dWord;
    DeviceNumber devNum = m_pDevice->GetDevNum();
    FunctionNumber funcNum = GetFuncNum();

    NN_PCIE_LOG_INFO("BridgeFunction::Finalize() "); Display();

    // tear down devices
    m_pDownStreamBus->Finalize();

    // set bus limits back to zero (no more bus accesses after this!)
    if (m_pUpStreamBus->ReadConfigDWord(devNum, funcNum, Type1ConfigOffset_PrimaryBusNumber, &dWord).IsSuccess())
    {
        dWord = (dWord & 0xff000000);
        m_pUpStreamBus->WriteConfigDWord(devNum, funcNum, Type1ConfigOffset_PrimaryBusNumber, dWord);
    }

    // free the bus object
    delete m_pDownStreamBus;
    m_pDownStreamBus = nullptr;

    return Function::Finalize();
}

void BridgeFunction::Display()
{
    char classCodeDesc[64];
    GetClassCodeDescription(classCodeDesc, sizeof(classCodeDesc));
    NN_PCIE_LOG_NOPREFIX("BDF-%03x:%02x:%x, Class-%s, VID-0x%04x, DEVID-0x%04x, CapableSpeed-%s\n",
                         GetBus()->GetBusNum(), GetDevNum(), GetFuncNum(),
                         classCodeDesc, GetVendorId(), GetDeviceId(),
                         GetBusSpeedDescription(GetCapableSpeed()));
}

Bus* BridgeFunction::GetDownStreamBus()
{
    return m_pDownStreamBus;
}

Bus* BridgeFunction::GetUpStreamBus()
{
    return m_pUpStreamBus;
}

BusNumber BridgeFunction::GetSecondaryBusNumber()
{
    return m_SecondaryBusNum;
}

BridgeDevice* BridgeFunction::GetDevice()
{
    return reinterpret_cast<BridgeDevice*>(Function::GetDevice());
}

Result  BridgeFunction::ConfigureSubordinate(BusNumber subBusNum,
                                             ResourceAddr *pIoAddrBase, ResourceAddr ioAddrLimit,
                                             ResourceAddr *pMemNoPrefAddrBase, ResourceAddr memNoPrefAddrLimit,
                                             ResourceAddr *pMemPrefAddrBase, ResourceAddr memPrefAddrLimit)
{
    Result result;
    uint32_t baseRegVal, limitRegVal;
    DeviceNumber devNum = m_pDevice->GetDevNum();
    FunctionNumber funcNum = GetFuncNum();

    // stop upon first error
    do
    {
        BusAddress ioBusAddressBase, memNoPrefBusAddressBase, memPrefBusAddressBase;
        BusAddress ioBusAddressLimit, memNoPrefBusAddressLimit, memPrefBusAddressLimit;

        /*
         * Get base bus addresses
         */
        NN_PCIE_ABORT_UPON_ERROR(m_pDriver->GetBusAddressess(*pIoAddrBase, &ioBusAddressBase));
        NN_PCIE_ABORT_UPON_ERROR(m_pDriver->GetBusAddressess(*pMemNoPrefAddrBase, &memNoPrefBusAddressBase));
        NN_PCIE_ABORT_UPON_ERROR(m_pDriver->GetBusAddressess(*pMemPrefAddrBase, &memPrefBusAddressBase));

        /*
         * Configure subordinate bus number
         */
        m_SubordinateBusNum = subBusNum;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigByte(devNum, funcNum, Type1ConfigOffset_SubordinateBusNumber, m_SubordinateBusNum));

        /*
         * Assign resource addresses of all subordinate functions
         */
        NN_PCIE_BREAK_UPON_ERROR(m_pDownStreamBus->AssignResourceAddresses(pIoAddrBase, ioAddrLimit,
                                                                           pMemNoPrefAddrBase, memNoPrefAddrLimit,
                                                                           pMemPrefAddrBase, memPrefAddrLimit));

        /*
         * Get limits bus addresses
         */
        NN_PCIE_ABORT_UPON_ERROR(m_pDriver->GetBusAddressess(*pIoAddrBase - 1, &ioBusAddressLimit));
        NN_PCIE_ABORT_UPON_ERROR(m_pDriver->GetBusAddressess(*pMemNoPrefAddrBase - 1, &memNoPrefBusAddressLimit));
        NN_PCIE_ABORT_UPON_ERROR(m_pDriver->GetBusAddressess(*pMemPrefAddrBase - 1, &memPrefBusAddressLimit));

        /*
         * Configure address bases and limits
         */
        // setup bridge IO bases
        baseRegVal = (ioBusAddressBase & 0x0000F000) >> 8;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigByte(devNum, funcNum, Type1ConfigOffset_IoBase, (uint8_t)baseRegVal));
        baseRegVal = ioBusAddressBase >> 16;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, Type1ConfigOffset_IoBaseUpper16, (uint16_t)baseRegVal));
        // setup bridge IO limits
        limitRegVal = (ioBusAddressLimit & 0x0000F000) >> 8;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigByte(devNum, funcNum, Type1ConfigOffset_IoLimit, (uint8_t)limitRegVal));
        limitRegVal = ioBusAddressLimit >> 16;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, Type1ConfigOffset_IoLimitUpper16, (uint16_t)limitRegVal));

        // setup bridge non-prefetchable MEM bases
        baseRegVal = (memNoPrefBusAddressBase & 0xFFF00000) >> 16;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, Type1ConfigOffset_MemoryBase, (uint16_t)baseRegVal));
        // setup bridge non-prefetchable MEM limits
        limitRegVal = (memNoPrefBusAddressLimit & 0xFFF00000) >> 16;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, Type1ConfigOffset_MemoryLimit, (uint16_t)limitRegVal));

        // setup bridge prefetchable MEM bases
        baseRegVal = (memPrefBusAddressBase & 0xFFF00000) >> 16;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, Type1ConfigOffset_PrefetchMemoryBase, (uint16_t)baseRegVal));
        baseRegVal = ((uint64_t)memPrefBusAddressBase) >> 32;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigDWord(devNum, funcNum, Type1ConfigOffset_PrefetchBaseUpper32, baseRegVal));
        // setup bridge prefetchable MEM limits
        limitRegVal = (memPrefBusAddressLimit & 0xFFF00000) >> 16;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, Type1ConfigOffset_PrefetchMemoryLimit, (uint16_t)limitRegVal));
        limitRegVal = ((uint64_t)memPrefBusAddressLimit) >> 32;
        NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigDWord(devNum, funcNum, Type1ConfigOffset_PrefetchLimitUpper32, limitRegVal));

        // helpful debug
        NN_PCIE_LOG_VERBOSE("BridgeFunction BDF-%03x:%02x:%x base/limit settings:\n",
                            m_pUpStreamBus->GetBusNum(), devNum, funcNum);
        NN_PCIE_LOG_VERBOSE("  IO:          0x%llx/0x%llx\n", ioBusAddressBase, ioBusAddressLimit);
        NN_PCIE_LOG_VERBOSE("  no pref MEM: 0x%llx/0x%llx\n", memNoPrefBusAddressBase, memNoPrefBusAddressLimit);
        NN_PCIE_LOG_VERBOSE("  pref MEM:    0x%llx/0x%llx\n", memPrefBusAddressBase, memPrefBusAddressLimit);
    }while (false);

    return result;
}

Result BridgeFunction::SetDeviceEnable(bool enabled)
{
    Result result = ResultSuccess();
    DeviceNumber devNum = m_pDevice->GetDevNum();
    FunctionNumber funcNum = GetFuncNum();

    do
    {
        if (enabled)
        {
            /* Enable IO, MEM and bus mastering */
            uint16_t cmdVal;
            NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->ReadConfigWord(devNum, funcNum, StandardConfigOffset_Command, &cmdVal));
            cmdVal |= (StandardConfigValue_CommandIo | StandardConfigValue_CommandMemory | StandardConfigValue_CommandMaster | StandardConfigValue_CommandSerr);
            NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, StandardConfigOffset_Command, cmdVal));

            /* Enable master aborts now */
            uint16_t bCtlWord;
            NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->ReadConfigWord(devNum, funcNum, Type1ConfigOffset_BridgeControl, &bCtlWord));
            bCtlWord |= Type1ConfigValue_BridgeControlMasterAbort;
            NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, Type1ConfigOffset_BridgeControl, bCtlWord));
        }
        else
        {
            /* Disable IO, MEM and bus mastering, errors */
            uint16_t cmdVal;
            NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->ReadConfigWord(devNum, funcNum, StandardConfigOffset_Command, &cmdVal));
            cmdVal &= ~(StandardConfigValue_CommandIo | StandardConfigValue_CommandMemory | StandardConfigValue_CommandMaster | StandardConfigValue_CommandSerr);
            NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, StandardConfigOffset_Command, cmdVal));

            /* Disable master aborts */
            uint16_t bCtlWord;
            NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->ReadConfigWord(devNum, funcNum, Type1ConfigOffset_BridgeControl, &bCtlWord));
            bCtlWord &= ~Type1ConfigValue_BridgeControlMasterAbort;
            NN_PCIE_BREAK_UPON_ERROR(m_pUpStreamBus->WriteConfigWord(devNum, funcNum, Type1ConfigOffset_BridgeControl, bCtlWord));
        }
    } while (false);

    return result;
}

Result BridgeFunction::SetLinkSpeed(BusSpeed speed)
{
    uint16_t regVal16;
    Result result = ResultSuccess();

    // Set link speed
    NN_PCIE_RETURN_UPON_ERROR(ReadPCIeCapabilityWord(PcieOffset_LinkControl2, &regVal16));

    regVal16 &= ~PcieValue_LinkStatusCurrentLinkSpeed;
    regVal16 |= ((speed == BusSpeed_EGen2) ? PcieValue_LinkStatusCurrentLinkSpeed_5_0GB : PcieValue_LinkStatusCurrentLinkSpeed_2_5GB);
    NN_PCIE_RETURN_UPON_ERROR(WritePCIeCapabilityWord(PcieOffset_LinkControl2, regVal16));

    result = RetrainLink();

    return result;
}

Result BridgeFunction::RetrainLink()
{
    uint16_t regVal16 = 0;
    Result result = ResultSuccess();

    NN_PCIE_RETURN_UPON_ERROR(ReadPCIeCapabilityWord(PcieOffset_LinkControl, &regVal16));
    regVal16 |= PcieValue_LinkControlRetrainLink;
    NN_PCIE_RETURN_UPON_ERROR(WritePCIeCapabilityWord(PcieOffset_LinkControl, regVal16));

    // Wait for link training to finish
    bool isLinkTraining = false;
    uint32_t elapsedTime = 0;
    do
    {
        NN_PCIE_RETURN_UPON_ERROR(ReadPCIeCapabilityWord(PcieOffset_LinkStatus, &regVal16));
        isLinkTraining = (regVal16 & PcieValue_LinkStatusLinkTraining) ? true : false;
        if (isLinkTraining)
        {
            int32_t waitTimeInUs = 10000;
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(waitTimeInUs));
            elapsedTime += waitTimeInUs;
            result = ResultPortLinkDown();
        }
        else
        {
            result = ResultSuccess();
            break;
        }
    }while (isLinkTraining && (elapsedTime < MaxPortLinkWaitTimeUs));

    return result;
}

void* BridgeFunction::operator new(size_t size) NN_NOEXCEPT
{
    return nn::pcie::detail::MemoryAllocAligned(size, MinDataAlignmentSize, "BridgeFunction");
}

void BridgeFunction::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::pcie::detail::MemoryFree(p, "BridgeFunction");
}

// static method
bool BridgeFunction::IsBridgeFunction(Function *pFunc)
{
    return ((pFunc->GetHeaderType() == StandardConfigValue_HeaderTypeBridge) &&
            (pFunc->Get16BitClassCode() == PciBridgeClassCode));
}

} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn
