﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include "pcie_Memory.h"

namespace nn { namespace pcie { namespace detail {

static lmem::HeapHandle s_HeapHandle = NULL;


// Set to enable debug allocation tracing
#if 0
#define NN_PCIE_TRACE_ALLOC(pMem, objectName, size)   \
    NN_SDK_LOG("%p | %s | Alloc | size %d\n",pMem, objectName, size)
#define NN_PCIE_TRACE_FREE(pMem, objectName)          \
    NN_SDK_LOG("%p | %s | Free\n",pMem, objectName)
#else
#define NN_PCIE_TRACE_ALLOC(pMem, objectName, size)   \
    NN_UNUSED(objectName)
#define NN_PCIE_TRACE_FREE(pMem, objectName)          \
    NN_UNUSED(objectName)
#endif


void MemoryInit(void *pBase, size_t size)
{
    s_HeapHandle = lmem::CreateExpHeap(pBase, size, nn::lmem::CreationOption_ThreadSafe);
    lmem::SetExpHeapAllocationMode(s_HeapHandle, lmem::AllocationMode_BestFit);
}

void MemoryFinalize()
{
    lmem::DestroyExpHeap(s_HeapHandle);
    s_HeapHandle = NULL;
}

void* MemoryAllocAligned(size_t size, size_t alignment)
{
    void *pMem;
    pMem = lmem::AllocateFromExpHeap(s_HeapHandle, size, alignment);
    return pMem;
}

void* MemoryAllocAligned(size_t size, size_t alignment, const char *objectName)
{
    void* pMem = MemoryAllocAligned(size, alignment);
    NN_PCIE_TRACE_ALLOC(pMem, objectName, size);
    return pMem;
}

void* MemoryAlloc(size_t size)
{
    return lmem::AllocateFromExpHeap(s_HeapHandle, size);
}

void* MemoryAlloc(size_t size, const char *objectName)
{
    void* pMem = MemoryAlloc(size);
    NN_PCIE_TRACE_ALLOC(pMem, objectName, size);
    return pMem;
}

void* MemoryCalloc(size_t size, const char *objectName)
{
    void *pMem = MemoryAlloc(size, objectName);
    if (pMem)
    {
        std::memset(pMem, 0, size);
    }
    return pMem;
}

void* MemoryCallocAligned(size_t size, size_t alignment, const char *objectName)
{
    void *pMem = MemoryAllocAligned(size, alignment, objectName);
    if (pMem)
    {
        std::memset(pMem, 0, size);
    }
    return pMem;
}

void MemoryFree(void *pMem)
{
    lmem::FreeToExpHeap(s_HeapHandle, pMem);
}

void MemoryFree(void *pMem, const char *objectName)
{
    NN_PCIE_TRACE_FREE(pMem, objectName);
    MemoryFree(pMem);
}

size_t MemoryGetTotalFreeSize()
{
    size_t totalFreeSize = 0;
    if (s_HeapHandle != NULL)
    {
        totalFreeSize = lmem::GetExpHeapTotalFreeSize(s_HeapHandle);
    }
    return totalFreeSize;
}

} // end of namespace detail
} // end of namespace usb0
} // end of namespace nn
