﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @file
 * @brief Macros
 *
 * @details
 */

#pragma once

namespace nn { namespace pcie { namespace detail {


#define NN_PCIE_ROUNDUP_SIZE(size, width) \
    (((size) + (width) - 1) & ~size_t((width) - 1))
#define NN_PCIE_MIN(a, b)   (((a) < (b)) ? (a) : (b))
#define NN_PCIE_MAX(a, b)   (((a) > (b)) ? (a) : (b))

enum LogLevel {
    LogLevel_Error,
    LogLevel_Warn,
    LogLevel_Info,
    LogLevel_Verbose,
};

static const LogLevel s_LogLevel = LogLevel_Info;

#define NN_PCIE_LOG_COMMAND(level, tag, fmt, ...)               \
    do                                                          \
    {                                                           \
        if( nn::pcie::detail::s_LogLevel >= level )             \
        {                                                       \
            NN_SDK_LOG("[pcie]%s " fmt, tag, ##__VA_ARGS__);    \
        }                                                       \
    } while(0)


// Log macros
#define NN_PCIE_ABORT(format,...)            NN_ABORT("[pcie] abort: " format, ##__VA_ARGS__)
#define NN_PCIE_LOG_VERBOSE(format,...)      NN_PCIE_LOG_COMMAND(nn::pcie::detail::LogLevel_Verbose, "",          format, ##__VA_ARGS__)
#define NN_PCIE_LOG_INFO(format,...)         NN_PCIE_LOG_COMMAND(nn::pcie::detail::LogLevel_Info,    "[info]",    format, ##__VA_ARGS__)
#define NN_PCIE_LOG_WARN(format,...)         NN_PCIE_LOG_COMMAND(nn::pcie::detail::LogLevel_Warn,    "[warning]", format, ##__VA_ARGS__)
#define NN_PCIE_LOG_ERROR(format,...)        NN_PCIE_LOG_COMMAND(nn::pcie::detail::LogLevel_Error,   "[error]",   format, ##__VA_ARGS__)
#define NN_PCIE_LOG_NOPREFIX(format,...)     NN_SDK_LOG(format, ##__VA_ARGS__)

// Error handling macros
#define NN_PCIE_ABORT_IF_ZERO(uintval)                                    \
    if(!(uintval))                                                        \
    {                                                                     \
        NN_PCIE_ABORT("%s is zero in func-%s @ line-%d.\n",               \
                      #uintval,__FUNCTION__,__LINE__);                    \
    }

#define NN_PCIE_ABORT_IF_NULL(pointerVal)                                 \
    if(!(pointerVal))                                                     \
    {                                                                     \
        NN_PCIE_ABORT("%s is NULL in func-%s @ line-%d.\n",               \
                      #pointerVal,__FUNCTION__,__LINE__);                 \
}


#define NN_PCIE_BREAK_UPON_ERROR(attemptedMethodCall)                     \
    if (!((result=(attemptedMethodCall)).IsSuccess()))                    \
    {                                                                     \
        NN_PCIE_LOG_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n", \
                         #attemptedMethodCall,result.GetModule(),         \
                         result.GetDescription(),__FUNCTION__,__LINE__);  \
    }

#define NN_PCIE_ABORT_UPON_ERROR(attemptedMethodCall)                     \
    if (!((result=(attemptedMethodCall)).IsSuccess()))                    \
    {                                                                     \
        NN_PCIE_ABORT("%s unsuccessful %d:%d in func-%s @ line-%d.\n",    \
                         #attemptedMethodCall,result.GetModule(),         \
                         result.GetDescription(),__FUNCTION__,__LINE__);  \
    }


#define NN_PCIE_BREAK_UNLESS(condition, failureResult)                    \
    if (!(condition))                                                     \
    {                                                                     \
        NN_PCIE_LOG_WARN("condition %s false in func-%s @ line-%d.\n",    \
                         #condition,__FUNCTION__,__LINE__); break;        \
    }

#define NN_PCIE_BREAK_UPON_MEM_ALLOC_FAIL(attemptedAlloc)                 \
    if (!((attemptedAlloc)!=NULL))                                        \
    {                                                                     \
        result = ResultMemAllocFailure();                                 \
        NN_PCIE_LOG_WARN("%s unsuccessful alloc in func-%s @ line-%d.\n", \
                         #attemptedAlloc,__FUNCTION__,__LINE__); break;   \
    }

#define NN_PCIE_ABORT_UNLESS(condition)                                   \
    if (!(condition))                                                     \
    {                                                                     \
        NN_PCIE_ABORT("condition %s false in func-%s @ line-%d.\n",       \
                      #condition,__FUNCTION__,__LINE__);                  \
    }

#define NN_PCIE_RETURN_UPON_ERROR(attemptedMethodCall)                    \
    if (!((result=(attemptedMethodCall)).IsSuccess()))                    \
    {                                                                     \
        NN_PCIE_LOG_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n", \
                         #attemptedMethodCall,result.GetModule(),         \
                         result.GetDescription(),__FUNCTION__,__LINE__);  \
        return result;                                                    \
    }

#define NN_PCIE_ABORT_UPON_ERROR(attemptedMethodCall)                     \
    if (!((result=(attemptedMethodCall)).IsSuccess()))                    \
    {                                                                     \
        NN_PCIE_ABORT("%s unsuccessful %d:%d in func-%s @ line-%d.\n",    \
                         #attemptedMethodCall,result.GetModule(),         \
                         result.GetDescription(),__FUNCTION__,__LINE__);  \
    }

#define NN_PCIE_WARN_UPON_ERROR(attemptedMethodCall)                      \
    if (!((result=(attemptedMethodCall)).IsSuccess()))                    \
    {                                                                     \
        NN_PCIE_LOG_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n", \
                         #attemptedMethodCall,result.GetModule(),         \
                         result.GetDescription(),__FUNCTION__,__LINE__);  \
    }

#define NN_PCIE_COMPARE_RESULT(a, b)                                    \
    ((((a).GetInnerValueForDebug()) == ((b).GetInnerValueForDebug())) ? true : false)

#define NN_PCIE_ARRAY_SIZE(array) (sizeof(array) / sizeof((array)[0]))

#define NN_PCIE_ARRAY_COUNT32(array) (int32_t)(sizeof(array) / sizeof((array)[0]))


} // end of namespace detail
} // end of namespace pcie
} // end of namespace nn


