﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/audioctrl/audioctrl_AudioControllerOutputDeviceControllerShim.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/hid/system/hid_Handheld.h>
#include <nn/lbl/lbl_Switch.h>
#include <nn/omm/omm_Result.h>
#include <nn/omm/detail/omm_Log.h>
#include <nn/omm/srv/omm_Common.private.h>
#include <nn/omm/srv/omm_OperationModeManager.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/psc/psc_PmModule.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/spl/spl_Api.h>
#include <nn/usb/pd/usb_PdCradle.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Exchange.h>
#include <nn/vi/vi_Display.h>
#include <nn/vi/vi_DisplayMode.private.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/vi_DisplayEvents.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/prepo/prepo_ApiAdmin.h>
#include <nn/pm/pm_BootModeApi.h>

#include "detail/omm_Utility.h"
#include "omm_ImageBinary.h"

namespace nn { namespace omm { namespace srv {

    bool IsHandheldSupported(ConsoleStyle consoleStyle) NN_NOEXCEPT
    {
        return consoleStyle == ConsoleStyle::Handheld || consoleStyle == ConsoleStyle::ConsoleAndHandheld;
    }

    bool IsConsoleSupported(ConsoleStyle consoleStyle) NN_NOEXCEPT
    {
        return consoleStyle == ConsoleStyle::Console || consoleStyle == ConsoleStyle::ConsoleAndHandheld;
    }

    namespace {
        const TimeSpan NoizeReductionTimeSpan = TimeSpan::FromMilliSeconds(10);

        OperationModePolicy GetPolicy(const char* policy) NN_NOEXCEPT
        {
            if (std::strncmp(policy, "auto", sizeof("auto")) == 0) return OperationModePolicy::Auto;
            if (std::strncmp(policy, "handheld", sizeof("handheld")) == 0) return OperationModePolicy::Handheld;
            if (std::strncmp(policy, "console", sizeof("console")) == 0) return OperationModePolicy::Console;

            return OperationModePolicy::Auto;
        }

        ConsoleStyle GetConsoleStyle(const char* style) NN_NOEXCEPT
        {
            if (std::strncmp(style, "console_and_handheld", sizeof("console_and_handheld")) == 0) return ConsoleStyle::ConsoleAndHandheld;
            if (std::strncmp(style, "console", sizeof("console")) == 0) return ConsoleStyle::Console;
            if (std::strncmp(style, "handheld", sizeof("handheld")) == 0) return ConsoleStyle::Handheld;

            return ConsoleStyle::ConsoleAndHandheld;
        }

        bool IsAcOkBoot() NN_NOEXCEPT
        {
            spl::Initialize();
            NN_UTIL_SCOPE_EXIT{ spl::Finalize(); };
            return spl::GetBootReason().bootReason == spl::BootReason_AcOk;
        }

        void SubmitErrorReportVideoOutputSettingContext(const char* setting)
        {
            erpt::Context context(erpt::CategoryId::VideoOutputInfo);
            context.Add(erpt::FieldId::VideoOutputSetting, setting, static_cast<uint32_t>(std::strlen(setting)));
            context.SubmitContext();
        }

        bool CanSleepOnAcOkBoot()
        {
            settings::factory::ConfigurationId1 configurationId1;
            settings::factory::GetConfigurationId1(&configurationId1);
            auto isSdev = strncmp( "SDEV", configurationId1.string, 4 ) == 0;

            bool isProdMode;
            auto keySize = settings::fwdbg::GetSettingsItemValue(&isProdMode, sizeof(isProdMode), "omm", "sleep_on_ac_ok_boot");
            NN_ABORT_UNLESS(keySize == sizeof(isProdMode));

            return !(isSdev && !isProdMode);
        }
    }

    OperationModeManager::OperationModeManager() NN_NOEXCEPT :
        m_CradleObserver(this), m_Current(OperationMode::Handheld), m_ChangeEvent(os::EventClearMode_ManualClear, true), m_PrecedingLcdProcessingEvent(os::EventClearMode_ManualClear),
        m_IsAutoSwitchEnabled(), m_IsImmediatelyAfterAwake(), m_OperationModePolicy(OperationModePolicy::Auto), m_ConsoleStyle(ConsoleStyle::ConsoleAndHandheld)
    {
        m_PrecedingLcdProcessingEvent.Signal();
    }

    void OperationModeManager::InitializeForShowStartupLogo() NN_NOEXCEPT
    {
        InitializePolicy();
        m_DisplayManager.Initialize(this, m_ConsoleStyle);
    }

    void OperationModeManager::Initialize() NN_NOEXCEPT
    {
        usb::pd::OpenCradleSession(&m_PdCradleSession);
        // クレードル復旧制御の権限を獲得しておく
        usb::pd::EnableCradleRecovery(nullptr, &m_PdCradleSession);
        m_CradleLedController.Initialize(m_PdCradleSession, m_ConsoleStyle);

        if (m_ConsoleStyle == ConsoleStyle::ConsoleAndHandheld)
        {
            switch (m_OperationModePolicy)
            {
            case OperationModePolicy::Auto:
                {
                    m_IsAutoSwitchEnabled = true;
                }
                break;
            case OperationModePolicy::Console:
                {
                    SwitchToConsoleMode(true);
                }
                break;
            case OperationModePolicy::Handheld:
                {
                    SwitchToHandheldMode();
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }

        }
        else if (m_ConsoleStyle == ConsoleStyle::Console)
        {
            SwitchToConsoleMode(true);
        }
        else if (m_ConsoleStyle == ConsoleStyle::Handheld)
        {
            SwitchToHandheldMode();
        }
        else
        {
            NN_UNEXPECTED_DEFAULT;
        }

        m_CradleObserver.Initialize(m_PdCradleSession, m_ConsoleStyle);
        m_SkipsToTriggerNextOneTouchPlay = IsAcOkBoot() && m_CradleObserver.HasCradle() && m_CradleObserver.IsGoodCradleMounted();

        // workaround for SIGLO-63830
        // SDEV ではクレードル接続状態での AC 接続や Target Manager からの電源 ON でも
        // AcOkBoot が true になってしまい、m_ShouldSleepOnBoot が true となりスリープに入ってしまう
        // SDEV では元々クレードル起動ができないこと、デバッグ用途などで不便なことを考慮して
        // SDEV でかつ DEV_MODE のときは m_ShouldSleepOnBoot を false にする
        m_ShouldSleepOnBoot = CanSleepOnAcOkBoot() && IsAcOkBoot() && m_CradleObserver.HasCradle() && m_CradleObserver.IsGoodCradleMounted();

        m_CradleObserver.Start();
        m_AwakeModeManager.Initialize(&m_DisplayManager, &m_CradleObserver);

        m_DisplayManager.StartHotplugHandler();
        if (IsHandheldSupported(m_ConsoleStyle))
        {
            StartPowerStateHandler();
        }
    }

    void OperationModeManager::ShowStartupLogo() NN_NOEXCEPT
    {
        if (!IsHandheldSupported(m_ConsoleStyle))
        {
            return;
        }
        NN_DETAIL_OMM_TRACE("[OperationModeManager] Show startup logo: %lld ms\n", os::GetSystemTick().ToTimeSpan().GetMilliSeconds());

        bool isFieldTesting;
        settings::fwdbg::GetSettingsItemValue(&isFieldTesting, sizeof(isFieldTesting), "systemconfig", "field_testing");

        if (!isFieldTesting)
        {
            // SIGLO-74913
            // 起動ロゴの画像は vi が持っているので出力開始の指示だけすれば表示される。
            m_DisplayManager.StartDrawingLayer();
            m_DisplayManager.SetStartupLogoLayerVisible(true);

            int fadeInMs;
            settings::fwdbg::GetSettingsItemValue(&fadeInMs, sizeof(fadeInMs), "omm", "startup_fade_in_ms");

            m_DisplayManager.FadeLcd(false, 0);
            EnableHandheldDisplay();
            m_DisplayManager.FadeLcdAndBacklight(true, TimeSpan::FromMilliSeconds(fadeInMs));
            this->m_IsStartupLogoShown = true;
        }
    }

    void OperationModeManager::HideStartupLogo() NN_NOEXCEPT
    {
        if (!m_IsStartupLogoShown)
        {
            return;
        }
        NN_DETAIL_OMM_TRACE("[OperationModeManager] Hide startup logo: %lld ms\n", os::GetSystemTick().ToTimeSpan().GetMilliSeconds());

        int fadeOutMs;
        settings::fwdbg::GetSettingsItemValue(&fadeOutMs, sizeof(fadeOutMs), "omm", "startup_fade_out_ms");

        // バックライトはつけたまま
        // BG レイヤを無効にしたら黒塗してグローバルフェードは戻しておく
        m_DisplayManager.FadeLcd(false, TimeSpan::FromMilliSeconds(fadeOutMs));
        m_DisplayManager.SetDrawingLayerVisible(false);
        m_DisplayManager.SetDrawingLayerTexture(0, 0, 0, 0, nullptr, 0);
        m_DisplayManager.FadeLcd(true, TimeSpan::FromMilliSeconds(0));

        m_DisplayManager.StopDrawingLayer();
        this->m_IsStartupLogoShown = false;
    }

    void OperationModeManager::InitializePolicy() NN_NOEXCEPT
    {
        char consoleStyleString[32];
        settings::fwdbg::GetSettingsItemValue( consoleStyleString, sizeof(consoleStyleString), "omm", "console_style" );
        NN_DETAIL_OMM_TRACE("[OperationModeManager] console style: %s\n", consoleStyleString);
        m_ConsoleStyle = GetConsoleStyle(consoleStyleString);

        char policyString[16];
        settings::fwdbg::GetSettingsItemValue(policyString, sizeof(policyString), "omm", "operation_mode_policy");
        NN_DETAIL_OMM_TRACE("[OperationModeManager] mode policy: %s\n", policyString);
        m_OperationModePolicy = GetPolicy(policyString);

        int sleepFadeInMilliseconds;
        settings::fwdbg::GetSettingsItemValue(&sleepFadeInMilliseconds, sizeof(sleepFadeInMilliseconds), "omm", "sleep_fade_in_ms");
        int sleepFadeOutMilliseconds;
        settings::fwdbg::GetSettingsItemValue(&sleepFadeOutMilliseconds, sizeof(sleepFadeOutMilliseconds), "omm", "sleep_fade_out_ms");
        int backlightOffMilliseconds;
        settings::fwdbg::GetSettingsItemValue(&backlightOffMilliseconds, sizeof(backlightOffMilliseconds), "omm", "backlight_off_ms_on_handheld_switch");

        m_SleepFadeInSpan = TimeSpan::FromMilliSeconds(sleepFadeInMilliseconds);
        m_SleepFadeOutSpan = TimeSpan::FromMilliSeconds(sleepFadeOutMilliseconds);
        m_BacklightOffSpanOnHandheldSwitch = TimeSpan::FromMilliSeconds(backlightOffMilliseconds);
    }

    Result OperationModeManager::GetOperationMode(sf::Out<OperationMode> outValue) NN_NOEXCEPT
    {
        *outValue = m_Current;
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::GetOperationModeChangeEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
    {
        *outValue = sf::NativeHandle(m_ChangeEvent.GetReadableHandle(), false);
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::EnableAudioVisual() NN_NOEXCEPT
    {
        NN_DETAIL_OMM_TRACE("[OperationModeManager] EnableAudioVisual\n");

        if (IsHandheldSupported(m_ConsoleStyle))
        {
            m_PrecedingLcdProcessingEvent.Wait();
        }

        if (m_Current == OperationMode::Console)
        {
            // workaround for SIGLO-49622
            // クレードル接続 + スリープ状態のときドックアウト -> ドックインした時、
            // 先に m_CradleLedController.On(); すると usb:pd が非アクティブで成功しないことがある。
            // ここで先に EnableConsoleDisplay(); することで usd:pd がアクティブになるための時間を稼いでいる。
            EnableConsoleDisplay();
            m_CradleLedController.On();
        }
        else
        {
            EnableHandheldDisplay();
        }

        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::DisableAudioVisual() NN_NOEXCEPT
    {
        NN_DETAIL_OMM_TRACE("[OperationModeManager] DisableAudioVisual\n");

        if (m_Current == OperationMode::Console)
        {
            DisableAllDisplay();
            m_CradleLedController.Off();
        }
        else
        {
            DisableAllDisplay();
        }

        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::FadeInDisplay() NN_NOEXCEPT
    {
        NN_DETAIL_OMM_TRACE("[OperationModeManager] FadeInDisplay\n");

        if (m_Current == OperationMode::Handheld)
        {
            m_DisplayManager.FadeLcdAndBacklight(true, m_SleepFadeInSpan);
        }
        // 据置モードでは hotplug 時に Fade 設定される

        if (!m_CradleObserver.IsStarted())
        {
            m_CradleObserver.Start();
        }

        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::FadeOutDisplay() NN_NOEXCEPT
    {
        NN_DETAIL_OMM_TRACE("[OperationModeManager] FadeOutDisplay\n");

        if (m_CradleObserver.IsStarted())
        {
            m_CradleObserver.Stop();
        }

        if (m_Current == OperationMode::Console)
        {
            m_DisplayManager.FadeTv(false, m_SleepFadeOutSpan);
        }
        else
        {
            m_DisplayManager.FadeLcdAndBacklight(false, m_SleepFadeOutSpan);
        }

        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::NotifyCecSettingsChanged() NN_NOEXCEPT
    {
        m_DisplayManager.GetCecController().NotifyCecSettingsChanged();
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::SetApplicationCecSettingsAndNotifyChanged(int mode) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS( IsValidTvPowerStateMatchingMode(mode), ResultInvalidParameter() );
        m_DisplayManager.GetCecController().SetApplicationCecSettingsAndNotifyChanged(static_cast<omm::TvPowerStateMatchingMode>(mode));
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::SetOperationModePolicy(OperationModePolicy policy) NN_NOEXCEPT
    {
        m_CradleObserver.Stop();
        NN_UTIL_SCOPE_EXIT{ m_CradleObserver.Start(); };

        if (policy == OperationModePolicy::Auto)
        {
            m_IsAutoSwitchEnabled = true;
            // 据置モード固定から auto に戻したとき、クレードルが接続されていない場合
            if (m_Current == OperationMode::Console && !m_CradleObserver.IsGoodCradleMounted())
            {
                SwitchToHandheldMode();
            }
            // 携帯モード固定から auto に戻したとき、クレードルが接続されている場合
            else if (m_Current == OperationMode::Handheld && m_CradleObserver.IsGoodCradleMounted())
            {
                SwitchToConsoleMode(true);
            }
        }
        else if (policy == OperationModePolicy::Console)
        {
            m_IsAutoSwitchEnabled = false;
            if (m_Current == OperationMode::Handheld)
            {
                SwitchToConsoleMode(true);
            }
        }
        else if (policy == OperationModePolicy::Handheld)
        {
            m_IsAutoSwitchEnabled = false;
            if (m_Current == OperationMode::Console)
            {
                SwitchToHandheldMode();
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::EnterSleepAndWait(sf::NativeHandle handle) NN_NOEXCEPT
    {
        NN_DETAIL_OMM_TRACE("[OperationModeManager] EnterSleepAndWait enter\n");

        m_DisplayManager.StopHotplugHandler();
        m_DisplayManager.GetCecController().BeforeSleep();

        // クレードルとの接続異常発生からの復旧処理による再接続で DP2HDMI が電源 ON になるのを抑制
        usb::pd::DisableCradleRecovery( nullptr, &m_PdCradleSession );

        os::SystemEvent sleepCancelEvent;
        sleepCancelEvent.AttachReadableHandle( handle.GetOsHandle(), handle.IsManaged(), os::EventClearMode_ManualClear );
        handle.Detach();
        m_AwakeModeManager.EnterSleepAndWait( &sleepCancelEvent );

        // クレードルとの接続異常発生からの復旧処理による再接続で DP2HDMI が電源 ON になるのを許可
        bool isCradleRecoverySuspended;
        usb::pd::EnableCradleRecovery( &isCradleRecoverySuspended, &m_PdCradleSession );

        // workaround for SIGLO-38576
        if (m_Current == OperationMode::Handheld &&
            m_CradleObserver.HasCradle() && m_CradleObserver.IsGoodCradleMounted())
        {
            m_IsImmediatelyAfterAwake = true;
            NN_UTIL_SCOPE_EXIT{ m_IsImmediatelyAfterAwake = false; };

            m_PrecedingLcdProcessingEvent.Wait();
            m_CradleObserver.HandleStateExplicitly();
        }

        m_DisplayManager.StartHotplugHandler();
        m_DisplayManager.GetCecController().AfterSleep();

        NN_DETAIL_OMM_TRACE("[OperationModeManager] EnterSleepAndWait exit\n");

        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::GetCradleStatus(sf::Out<CradleStatus> outValue) NN_NOEXCEPT
    {
        *outValue = m_CradleObserver.GetCradleStatus();
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::GetCradleFwVersion(sf::Out<uint32_t> outPdcH, sf::Out<uint32_t> outPdcA, sf::Out<uint32_t> outMcu, sf::Out<uint32_t> outDp2Hdmi) NN_NOEXCEPT
    {
        return m_CradleObserver.GetCradleFwVersion(outPdcH.GetPointer(), outPdcA.GetPointer(), outMcu.GetPointer(), outDp2Hdmi.GetPointer());
    }

    void OperationModeManager::OnMounted() NN_NOEXCEPT
    {
        if (m_IsAutoSwitchEnabled)
        {
            auto handleDisplay = !m_IsImmediatelyAfterAwake;
            SwitchToConsoleMode(handleDisplay);
        }
    }

    Result OperationModeManager::GetDefaultDisplayResolution(sf::Out<int32_t> outWidth, sf::Out<int32_t> outHeight) NN_NOEXCEPT
    {
        m_DisplayManager.GetDefaultDisplayResolution(outWidth.GetPointer(), outHeight.GetPointer());
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::GetDefaultDisplayResolutionChangeEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
    {
        auto pEvent = m_DisplayManager.GetDefaultDisplayResolutionChangeEvent();
        *outValue = sf::NativeHandle(pEvent->GetReadableHandle(), false);
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::UpdateDefaultDisplayResolution() NN_NOEXCEPT
    {
        if (m_Current == OperationMode::Console)
        {
            m_DisplayManager.UpdateDefaultDisplayResolution( false );
        }
        else
        {
            m_DisplayManager.UpdateDefaultDisplayResolution( true );
        }
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::ShouldSleepOnBoot(sf::Out<bool> pOut) NN_NOEXCEPT
    {
        *pOut = m_ShouldSleepOnBoot;
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::NotifyHdcpApplicationExecutionStarted() NN_NOEXCEPT
    {
        m_DisplayManager.GetHdcpManager().NotifyHdcpApplicationExecutionStarted();
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::NotifyHdcpApplicationExecutionFinished() NN_NOEXCEPT
    {
        m_DisplayManager.GetHdcpManager().NotifyHdcpApplicationExecutionFinished();
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::NotifyHdcpApplicationDrawingStarted() NN_NOEXCEPT
    {
        m_DisplayManager.GetHdcpManager().NotifyHdcpApplicationDrawingStarted();
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::NotifyHdcpApplicationDrawingFinished() NN_NOEXCEPT
    {
        m_DisplayManager.GetHdcpManager().NotifyHdcpApplicationDrawingFinished();
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::GetHdcpAuthenticationFailedEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
    {
        auto pEvent = m_DisplayManager.GetHdcpManager().GetHdcpAuthenticationFailedEvent();
        *outValue = sf::NativeHandle(pEvent->GetReadableHandle(), false);
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::GetHdcpAuthenticationFailedEmulationEnabled(sf::Out<bool> pOut) NN_NOEXCEPT
    {
        *pOut = m_DisplayManager.GetHdcpManager().GetHdcpAuthenticationFailedEmulationEnabled();
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::SetHdcpAuthenticationFailedEmulation(bool enable) NN_NOEXCEPT
    {
        return m_DisplayManager.GetHdcpManager().SetHdcpAuthenticationFailedEmulation(enable);
    }

    Result OperationModeManager::GetHdcpState(sf::Out<HdcpState> outValue) NN_NOEXCEPT
    {
        *outValue = m_DisplayManager.GetHdcpManager().GetHdcpState();
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::GetHdcpStateChangeEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
    {
        auto pEvent = m_DisplayManager.GetHdcpManager().GetHdcpStateChangeEvent();
        *outValue = sf::NativeHandle(pEvent->GetReadableHandle(), false);
        NN_RESULT_SUCCESS;
    }

    Result OperationModeManager::ShowCardUpdateProcessing() NN_NOEXCEPT
    {
        const int pointX = CardUpdateProcessingPointX;
        const int pointY = CardUpdateProcessingPointY;
        const int width = CardUpdateProcessingWidth;
        const int height = CardUpdateProcessingHeight;
        const size_t dataSize = CardUpdateProcessingDataSize;
        const Bit8* pData = CardUpdateProcessingData;

        NN_SDK_ASSERT(width * height * sizeof(Bit32) <= dataSize);

        m_DisplayManager.StartDrawingLayer();
        m_DisplayManager.SetDrawingLayerTexture(pointX, pointY, width, height, pData, dataSize);
        m_DisplayManager.SetDrawingLayerVisible(true);

        m_DisplayManager.EnableHandheldDisplay();

        // 描画する時間
        os::SleepThread(TimeSpan::FromMilliSeconds(500));

        // 同じ絵を描画するので黒塗りは省略
        m_DisplayManager.SetDrawingLayerVisible(false);
        m_DisplayManager.StopDrawingLayer();

        NN_RESULT_SUCCESS;
    }

    void OperationModeManager::OnUnmounted() NN_NOEXCEPT
    {
        if (m_IsAutoSwitchEnabled)
        {
            SwitchToHandheldMode();
        }
    }

    OperationMode OperationModeManager::GetOperationMode() const NN_NOEXCEPT
    {
        return m_Current;
    }

    void OperationModeManager::EnableHandheldDisplay() NN_NOEXCEPT
    {
        m_DisplayManager.EnableHandheldDisplay();
    }

    void OperationModeManager::EnableConsoleDisplay() NN_NOEXCEPT
    {
        m_DisplayManager.GetCecController().Enable();
        if (!util::Exchange(&m_SkipsToTriggerNextOneTouchPlay, false))
        {
            m_DisplayManager.GetCecController().TriggerOneTouchPlay();
        }
        m_DisplayManager.EnableConsoleDisplay();
    }

    void OperationModeManager::DisableAllDisplay() NN_NOEXCEPT
    {
        m_DisplayManager.DisableAllDisplay();
    }

    void OperationModeManager::NotifyOperationModeChanged() NN_NOEXCEPT
    {
        m_ChangeEvent.Signal();
        m_DisplayManager.GetHdcpManager().NotifyOperationModeChanged();
        if(nn::pm::GetBootMode() != nn::pm::BootMode_Maintenance)
        {
            nn::prepo::SetOperationMode(static_cast<int64_t>(m_Current.load()));
        }
    }

    void OperationModeManager::SwitchToHandheldMode() NN_NOEXCEPT
    {
        audioctrl::SetDefaultTarget(audioctrl::AudioTarget_Speaker, NoizeReductionTimeSpan, NoizeReductionTimeSpan);
        m_CradleLedController.Off();
        m_DisplayManager.GetCecController().Disable();
        EnableHandheldDisplay();
        m_DisplayManager.FadeLcd(true, 0);

        hid::system::EnableHandheldHids();
        detail::SetTcOperatingMode(OperationMode::Handheld);
        detail::SetApmPerformanceMode(OperationMode::Handheld);

        m_Current = OperationMode::Handheld;
        NotifyOperationModeChanged();

        os::SleepThread(m_BacklightOffSpanOnHandheldSwitch);
        m_DisplayManager.SwitchBacklightOn();

        SubmitErrorReportVideoOutputSettingContext("LCD");

        NN_DETAIL_OMM_TRACE("[OperationModeManager] Switched to handheld mode\n");
    }

    void OperationModeManager::SwitchToConsoleMode(bool handleDisplay) NN_NOEXCEPT
    {
        audioctrl::SetDefaultTarget(audioctrl::AudioTarget_Tv, NoizeReductionTimeSpan, NoizeReductionTimeSpan);

        if (handleDisplay)
        {
            m_CradleLedController.On();
            m_DisplayManager.SwitchBacklightOff();
            EnableConsoleDisplay();
        }

        hid::system::DisableHandheldHids();
        detail::SetTcOperatingMode(OperationMode::Console);
        detail::SetApmPerformanceMode(OperationMode::Console);
        detail::EnableBtmRadio();

        m_Current = OperationMode::Console;
        NotifyOperationModeChanged();

        SubmitErrorReportVideoOutputSettingContext("TV");

        NN_DETAIL_OMM_TRACE("[OperationModeManager] Switched to console mode\n");
    }

    // 携帯モードスリープ復帰時に LCD にできるだけ早く火を入れるための最適化
    void OperationModeManager::PowerStateHandlerImpl() NN_NOEXCEPT
    {
        const psc::PmModuleId Dependencies[] = { psc::PmModuleId_Display };

        psc::PmModule module;
        NN_ABORT_UNLESS_RESULT_SUCCESS(module.Initialize(
            psc::PmModuleId::PmModuleId_Omm,
            Dependencies,
            sizeof(Dependencies) / sizeof(Dependencies[0]),
            os::EventClearMode_AutoClear
        ));

        psc::PmState priorState = psc::PmState_FullAwake;
        while (NN_STATIC_CONDITION(true))
        {
            psc::PmState state;
            psc::PmFlagSet flags;
            module.GetEventPointer()->Wait();
            NN_ABORT_UNLESS_RESULT_SUCCESS(module.GetRequest(&state, &flags));
            NN_DETAIL_OMM_TRACE("[OperationModeManager] Power state request %u\n", state);

            if (m_Current == OperationMode::Handheld &&
                state == psc::PmState_MinimumAwake && priorState != psc::PmState_FullAwake)
            {
                NN_DETAIL_OMM_TRACE("[OperationModeManager] MinimumAwake on handheld wake sequence. Set LCD power state on asynchronously\n");
                SetLcdPowerOnAsync();
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(module.Acknowledge(state, ResultSuccess()));
            priorState = state;
        }
    }

    void OperationModeManager::StartPowerStateHandler() NN_NOEXCEPT
    {
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[8192];
        static os::ThreadType s_Thread;

        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&s_Thread, [](void* this_)
        {
            reinterpret_cast<OperationModeManager*>(this_)->PowerStateHandlerImpl();
        }, this, s_Stack, sizeof(s_Stack), NN_SYSTEM_THREAD_PRIORITY(omm, PowerStateHandler)));
        os::SetThreadNamePointer(&s_Thread, NN_SYSTEM_THREAD_NAME(omm, PowerStateHandler));
        os::StartThread(&s_Thread);
    }

    void OperationModeManager::SetLcdPowerOnAsync() NN_NOEXCEPT
    {
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[8192];
        static os::ThreadType s_Thread;
        static bool s_ThreadIsStarted = false;

        if (s_ThreadIsStarted)
        {
            os::WaitThread(&s_Thread);
            os::DestroyThread(&s_Thread);
        }

        m_PrecedingLcdProcessingEvent.Clear();
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&s_Thread, [](void* p)
        {
            auto self = reinterpret_cast<OperationModeManager*>(p);
            self->EnableHandheldDisplay();
            self->m_PrecedingLcdProcessingEvent.Signal();
        }, this, s_Stack, sizeof(s_Stack), NN_SYSTEM_THREAD_PRIORITY(omm, SetLcdPowerOnAsync)));
        os::SetThreadNamePointer(&s_Thread, NN_SYSTEM_THREAD_NAME(omm, SetLcdPowerOnAsync));
        os::StartThread(&s_Thread);

        s_ThreadIsStarted = true;
    }

}}}
