﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/psm/psm_System.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/omm/srv/omm_CradleObserver.h>
#include <nn/omm/detail/omm_Log.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/erpt/common/erpt_Ids.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/usb/pd/usb_PdCradle.h>

namespace nn { namespace omm { namespace srv {
    namespace {
        bool IsCradleTypeStatus(const usb::pd::Status& status) NN_NOEXCEPT
        {
            return status.IsActive() && (status.GetDeviceType() == usb::pd::StatusDeviceType_Cradle || status.GetDeviceType() == usb::pd::StatusDeviceType_RelayBox);
        }

        bool IsGoodCradleMountedStatus(const usb::pd::Status& status) NN_NOEXCEPT
        {
            return IsCradleTypeStatus(status) && !status.IsCradlePowerShortage();
        }

        bool IsAnyCradleMountedStatus(const usb::pd::Status& status) NN_NOEXCEPT
        {
            return IsCradleTypeStatus(status);
        }
    }

    MultiWaitEvent::MultiWaitEvent() NN_NOEXCEPT : m_Count()
    {
        os::InitializeMultiWait(&m_MultiWait);
    }

    int MultiWaitEvent::WaitAny() NN_NOEXCEPT
    {
        auto signaled = os::WaitAny(&m_MultiWait);
        for (int i = 0; i < m_Count; i++)
        {
            if (&m_MultiWaitHolder[i] == signaled)
            {
                return i;
            }
        }

        NN_ABORT("must not come here");
    }

    CradleObserver::CradleObserver(CradleObserver::IHandler* handler) NN_NOEXCEPT : m_Handler(handler), m_IsStarted(), m_State(State::Boot), m_StopEvent(os::EventClearMode_ManualClear) {}

    CradleObserver::~CradleObserver() NN_NOEXCEPT
    {
        if (m_IsStarted)
        {
            Stop();
        }
    }

    void CradleObserver::Initialize(const usb::pd::CradleSession& cradleSession, ConsoleStyle consoleStyle) NN_NOEXCEPT
    {
        m_ConsoleStyle = consoleStyle;
        usb::pd::OpenSession(&m_PdSession);
        NN_ABORT_UNLESS_RESULT_SUCCESS(usb::pd::BindNoticeEvent(&m_CradleStatusEvent, &m_PdSession));
        m_StopIndex = m_MultiWait.Link(m_StopEvent.GetBase());
        m_StatusIndex = m_MultiWait.Link(&m_CradleStatusEvent);
        m_PdCradleSession = cradleSession;
    }

    void CradleObserver::Start() NN_NOEXCEPT
    {
        if (HasCradle())
        {
            static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
            NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_Thread, [](void* p) {
                auto observer = reinterpret_cast<CradleObserver*>(p);
                observer->ExecuteObservationLoop();
            }, this, s_Stack, sizeof(s_Stack), NN_SYSTEM_THREAD_PRIORITY(omm, CradleObserver)));
            os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(omm, CradleObserver));
            os::StartThread(&m_Thread);
            m_IsStarted = true;
        }
    }

    void CradleObserver::Stop() NN_NOEXCEPT
    {
        if (HasCradle())
        {
            m_StopEvent.Signal();
            os::WaitThread(&m_Thread);
            os::DestroyThread(&m_Thread);
            m_IsStarted = false;
        }
    }

    CradleObserver::Signal CradleObserver::Wait() NN_NOEXCEPT
    {
        if (HasCradle())
        {
            auto index = m_MultiWait.WaitAny();
            if (index == m_StopIndex)
            {
                m_StopEvent.Clear();
                return Signal::Stop;
            }
            if (index == m_StatusIndex)
            {
                os::ClearSystemEvent(&m_CradleStatusEvent);
                return Signal::Status;
            }

            NN_ABORT("must not come here.");
        }

        return Signal::Stop;
    }

    void CradleObserver::HandleState() NN_NOEXCEPT
    {
        if (HasCradle())
        {
            switch (m_State)
            {
            case State::Boot: m_State = HandleBootState(); break;
            case State::NotMounted: m_State = HandleNotMountedState(); break;
            case State::Mounted: m_State = HandleMountedState(); break;
            case State::Stop: break;;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    void CradleObserver::ExecuteObservationLoop() NN_NOEXCEPT
    {
        if (HasCradle())
        {
            while (NN_STATIC_CONDITION(true))
            {
                HandleState();

                auto signal = Wait();
                if (signal == Signal::Stop)
                {
                    break;
                }
            }
        }
    };

    CradleObserver::State CradleObserver::HandleBootState() NN_NOEXCEPT
    {
        if (HasCradle() && IsGoodCradleMounted())
        {
            NN_DETAIL_OMM_TRACE("[CradleObserver] Cradle is mounted at boot time.\n");

            m_Handler->OnMounted();
            auto submitContextResult = SubmitErrorReportCradleFirmwareContext();
            if( submitContextResult.IsFailure() )
            {
                NN_DETAIL_OMM_WARN("[CradleObserver] Failed to submit error report cradle firmware context : 0x%08x.\n", submitContextResult.GetInnerValueForDebug());
            }
            return State::Mounted;
        }
        else
        {
            NN_DETAIL_OMM_TRACE("[CradleObserver] Cradle is not mounted at boot time.\n");

            m_Handler->OnUnmounted();
            return State::NotMounted;
        }
    }

    CradleObserver::State CradleObserver::HandleNotMountedState() NN_NOEXCEPT
    {
        if (HasCradle() && IsGoodCradleMounted())
        {
            NN_DETAIL_OMM_TRACE("[CradleObserver] Cradle is mounted.\n");

            m_Handler->OnMounted();
            auto submitContextResult = SubmitErrorReportCradleFirmwareContext();
            if( submitContextResult.IsFailure() )
            {
                NN_DETAIL_OMM_WARN("[CradleObserver] Failed to submit error report cradle firmware context : 0x%08x.\n", submitContextResult.GetInnerValueForDebug());
            }
            return State::Mounted;
        }

        return State::NotMounted;
    }

    CradleObserver::State CradleObserver::HandleMountedState() NN_NOEXCEPT
    {
        if (!HasCradle() || !IsGoodCradleMounted())
        {
            NN_DETAIL_OMM_TRACE("[CradleObserver] Cradle is unmounted.\n");

            m_Handler->OnUnmounted();
            return State::NotMounted;
        }

        return State::Mounted;
    }

    bool CradleObserver::HasCradle() NN_NOEXCEPT
    {
        if (m_ConsoleStyle == ConsoleStyle::ConsoleAndHandheld)
        {
            return true;
        }

        return false;
    }

    bool CradleObserver::IsGoodCradleMounted() NN_NOEXCEPT
    {
        if (HasCradle())
        {
            usb::pd::Status status;
            NN_ABORT_UNLESS_RESULT_SUCCESS(usb::pd::GetStatus(&status, &m_PdSession));

            return IsGoodCradleMountedStatus(status);
        }

        return false;
    }

    bool CradleObserver::IsAnyCradleMounted() NN_NOEXCEPT
    {
        if (HasCradle())
        {
            usb::pd::Status status;
            NN_ABORT_UNLESS_RESULT_SUCCESS(usb::pd::GetStatus(&status, &m_PdSession));

            return IsAnyCradleMountedStatus(status);
        }

        return false;
    }

    CradleStatus CradleObserver::GetCradleStatus() NN_NOEXCEPT
    {
        usb::pd::Status status;
        NN_ABORT_UNLESS_RESULT_SUCCESS(usb::pd::GetStatus(&status, &m_PdSession));

        if (!IsCradleTypeStatus(status))
        {
            return CradleStatus::None;
        }

        return status.IsCradlePowerShortage() ? CradleStatus::NotEnoughPower : CradleStatus::Mountable;
    }

    void CradleObserver::HandleStateExplicitly() NN_NOEXCEPT
    {
        if (HasCradle())
        {
            NN_ABORT_UNLESS(!m_IsStarted);

            HandleState();
        }
    }

    Result CradleObserver::GetCradleFwVersion(
        usb::pd::VdmPdcHFwVersion*      pOutPdcHFwVersion,
        usb::pd::VdmPdcAFwVersion*      pOutPdcAFwVersion,
        usb::pd::VdmMcuFwVersion*       pOutMcuFwVersion,
        usb::pd::VdmDp2HdmiFwVersion*   pOutDp2HdmiVersion) NN_NOEXCEPT
    {
        NN_RESULT_DO(usb::pd::GetCradleVdo(reinterpret_cast<usb::pd::Vdo*>(pOutPdcHFwVersion), &m_PdCradleSession, usb::pd::CradleVdmCommand_PdcHFwVersion));
        NN_RESULT_DO(usb::pd::GetCradleVdo(reinterpret_cast<usb::pd::Vdo*>(pOutPdcAFwVersion), &m_PdCradleSession, usb::pd::CradleVdmCommand_PdcAFwVersion));
        NN_RESULT_DO(usb::pd::GetCradleVdo(reinterpret_cast<usb::pd::Vdo*>(pOutMcuFwVersion), &m_PdCradleSession, usb::pd::CradleVdmCommand_McuFwVersion));
        NN_RESULT_DO(usb::pd::GetCradleVdo(reinterpret_cast<usb::pd::Vdo*>(pOutDp2HdmiVersion), &m_PdCradleSession, usb::pd::CradleVdmCommand_Dp2HdmiFwVersion));
        NN_RESULT_SUCCESS;
    }

    Result CradleObserver::SubmitErrorReportCradleFirmwareContext() NN_NOEXCEPT
    {
        usb::pd::VdmPdcHFwVersion       pdcHFwVersion;
        usb::pd::VdmPdcAFwVersion       pdcAFwVersion;
        usb::pd::VdmMcuFwVersion        mcuFwVersion;
        usb::pd::VdmDp2HdmiFwVersion    dp2HdmiFwVersion;

        NN_RESULT_DO(GetCradleFwVersion(&pdcHFwVersion, &pdcAFwVersion, &mcuFwVersion, &dp2HdmiFwVersion));

        erpt::Context context(erpt::CategoryId::CradleFirmwareInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::CradlePdcHFwVersion, static_cast<uint32_t>(pdcHFwVersion)));
        NN_RESULT_DO(context.Add(erpt::FieldId::CradlePdcAFwVersion, static_cast<uint32_t>(pdcAFwVersion)));
        NN_RESULT_DO(context.Add(erpt::FieldId::CradleMcuFwVersion, static_cast<uint32_t>(mcuFwVersion)));
        NN_RESULT_DO(context.Add(erpt::FieldId::CradleDp2HdmiFwVersion, static_cast<uint32_t>(dp2HdmiFwVersion)));
        NN_RESULT_DO(context.SubmitContext());
        NN_RESULT_SUCCESS;
    }
}}}
