﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Optional.h>
#include <nn/omm/omm_Api.h>
#include <nn/omm/detail/omm_IOperationModeManager.sfdl.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/omm/srv/omm_OperationModeManager.h>
#else
#include <nn/sf/sf_ProxyObjectAllocator.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/omm/detail/omm_PortName.h>
#endif

namespace nn { namespace omm {
    namespace {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        util::optional<sf::UnmanagedServiceObject<detail::IOperationModeManager, srv::OperationModeManager>> g_Server;
#else
        sf::ProxyObjectAllocator<16> g_ServiceObjectAllocator = NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER;
#endif
        sf::SharedPointer<detail::IOperationModeManager> g_Interface;
    }

    void Initialize() NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        g_Server.emplace();
        g_Server->GetImpl().Initialize();
        g_Interface = g_Server->GetShared();
#else
        g_ServiceObjectAllocator.Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS((sf::CreateHipcProxyByName<detail::IOperationModeManager>(&g_Interface, g_ServiceObjectAllocator.GetMemoryResource(), detail::PortNameForOperationModeManager)));
#endif
    }

    void Finalize() NN_NOEXCEPT
    {
        g_Interface = nullptr;
#if defined( NN_BUILD_CONFIG_OS_WIN )
        g_Server = util::nullopt;
#endif
    }


    OperationMode GetOperationMode() NN_NOEXCEPT
    {
        OperationMode mode;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->GetOperationMode(&mode));
        return mode;
    }

    void GetOperationModeChangeEvent(os::SystemEvent* outValue) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->GetOperationModeChangeEvent(&nativeHandle));
        outValue->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), os::EventClearMode_AutoClear);
        nativeHandle.Detach();
    }

    CradleStatus GetCradleStatus() NN_NOEXCEPT
    {
        CradleStatus status;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->GetCradleStatus(&status));
        return status;
    }

    Result GetCradleFwVersion(uint32_t* pOutPdcH, uint32_t* pOutPdcA, uint32_t* pOutMcu, uint32_t* pOutDp2Hdmi) NN_NOEXCEPT
    {
        return g_Interface->GetCradleFwVersion(pOutPdcH, pOutPdcA, pOutMcu, pOutDp2Hdmi);
    }

    void EnableAudioVisual() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->EnableAudioVisual());
    }

    void DisableAudioVisual() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->DisableAudioVisual());
    }

    void FadeInDisplay() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->FadeInDisplay());
    }

    void FadeOutDisplay() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->FadeOutDisplay());
    }

    void EnterSleepAndWait(os::SystemEvent* pEvent) NN_NOEXCEPT
    {
        sf::NativeHandle handle(pEvent->GetReadableHandle(), false);
        NN_ABORT_UNLESS_RESULT_SUCCESS( g_Interface->EnterSleepAndWait(std::move(handle)) );
    }

    void NotifyCecSettingsChanged() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->NotifyCecSettingsChanged());
    }

    void SetApplicationCecSettingsAndNotifyChanged(TvPowerStateMatchingMode mode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( IsValidTvPowerStateMatchingMode(mode) );
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->SetApplicationCecSettingsAndNotifyChanged(mode));
    }

    Result SetOperationModePolicy(OperationModePolicy policy) NN_NOEXCEPT
    {
        return g_Interface->SetOperationModePolicy(policy);
    }

    void GetDefaultDisplayResolution(int32_t* pWidth, int32_t* pHeight) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->GetDefaultDisplayResolution(pWidth, pHeight));
    }

    void GetDefaultDisplayResolutionChangeEvent(os::SystemEvent* outValue) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->GetDefaultDisplayResolutionChangeEvent(&nativeHandle));
        outValue->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), os::EventClearMode_AutoClear);
        nativeHandle.Detach();

        // 初回取得時は必ず signal されているため、クリアして渡す
        outValue->Clear();
    }

    void UpdateDefaultDisplayResolution() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->UpdateDefaultDisplayResolution());
    }

    bool ShouldSleepOnBoot() NN_NOEXCEPT
    {
        bool isShouldSleepOnBoot;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ShouldSleepOnBoot(&isShouldSleepOnBoot));
        return isShouldSleepOnBoot;
    }

    void NotifyHdcpApplicationExecutionStarted() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->NotifyHdcpApplicationExecutionStarted());
    }

    void NotifyHdcpApplicationExecutionFinished() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->NotifyHdcpApplicationExecutionFinished());
    }

    void NotifyHdcpApplicationDrawingStarted() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->NotifyHdcpApplicationDrawingStarted());
    }

    void NotifyHdcpApplicationDrawingFinished() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->NotifyHdcpApplicationDrawingFinished());
    }

    void GetHdcpAuthenticationFailedEvent(os::SystemEvent* outValue) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->GetHdcpAuthenticationFailedEvent(&nativeHandle));
        outValue->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), os::EventClearMode_ManualClear);
        nativeHandle.Detach();
    }

    bool GetHdcpAuthenticationFailedEmulationEnabled() NN_NOEXCEPT
    {
        bool isEnabled;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->GetHdcpAuthenticationFailedEmulationEnabled(&isEnabled));
        return isEnabled;
    }

    Result SetHdcpAuthenticationFailedEmulation(bool enable) NN_NOEXCEPT
    {
        return g_Interface->SetHdcpAuthenticationFailedEmulation(enable);
    }

    HdcpState GetHdcpState() NN_NOEXCEPT
    {
        HdcpState hdcpState;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->GetHdcpState(&hdcpState));
        return hdcpState;
    }

    void GetHdcpStateChangeEvent(os::SystemEvent* outValue) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->GetHdcpStateChangeEvent(&nativeHandle));
        outValue->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), os::EventClearMode_AutoClear);
        nativeHandle.Detach();
    }

    void ShowCardUpdateProcessing() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Interface->ShowCardUpdateProcessing());
    }
}}
