﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "olsc_Uploader.h"

#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/os/os_ThreadApi.h>

namespace nn { namespace olsc { namespace srv { namespace transfer {

/* --------------------------------------------------------------------------------------------
    Callback
 */
size_t UploaderBase::ReadFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT
{
    const auto OutputBytes = unitBytes * count;
    auto& obj = *reinterpret_cast<UploaderBase*>(pContext);
    // Obj が受信可能であることの検査
    if (true
        && obj.m_BufferInfo.stored.load() // 送信したいバッファにデータが入っている
        && obj.m_BufferInfo.total > 0)// 送信したいバッファのサイズが 0 より大きい
    {
        // 送信したいデータを ptr にコピー
        auto copyHead = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(obj.m_BufferInfo.address) + obj.m_OutputOffset);
        auto copyBytes = std::min(obj.m_BufferInfo.total - obj.m_OutputOffset, OutputBytes);
        std::memcpy(ptr, copyHead, copyBytes);
        obj.m_BufferInfo.used += copyBytes;

        obj.m_OutputOffset += copyBytes;
        // 末尾かどうか
        if (obj.m_OutputOffset >= obj.m_BufferInfo.total)
        {
            NN_SDK_ASSERT(obj.m_OutputOffset == obj.m_BufferInfo.total);
            obj.m_OutputOffset = 0;
            obj.m_BufferInfo.stored.store(false);
        }
        return copyBytes;
    }

    return CURL_READFUNC_PAUSE;
}

UploaderBase::UploaderBase(CURL* curlHandle, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
    : WebApiAccessorBase(curlHandle, pCancelable)
{
}

UploaderBase::~UploaderBase() NN_NOEXCEPT
{
    if (m_CurlMulti != nullptr)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(FinalizeImpl());
        WebApiAccessorBase::Reset();
    }
}

Result UploaderBase::Initialize(size_t uploadSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CurlMulti == nullptr);

    NN_RESULT_DO(WebApiAccessorBase::SetDefault());
    NN_RESULT_DO(WebApiAccessorBase::SetReadFunction(ReadFunction, uploadSize, this));

    auto* curlm = curl_multi_init();
    if (!curlm)
    {
        NN_DETAIL_OLSC_WARN("Failed to create multi handle.\n");
        return ResultCurlMultiApiError();
    }
    m_CurlMulti = curlm;
    m_UploadSize = uploadSize;

    NN_RESULT_SUCCESS;
}

Result UploaderBase::FinalizeImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CurlMulti != nullptr);

    auto mr = curl_multi_cleanup(m_CurlMulti);
    NN_RESULT_THROW_UNLESS(mr == CURLM_OK, ResultCurlMultiApiError());
    m_CurlMulti = nullptr;
    NN_RESULT_SUCCESS;
}

size_t UploaderBase::Send(const void* buffer, size_t sendSize) NN_NOEXCEPT
{
#define NN_OLSC_IO_FAIL_WITH_RESULT_UNLESS(condition, result) \
    do \
    { \
        if (!(condition)) \
        { \
            m_IoResult = (result); \
            return 0u; \
        } \
    } while (NN_STATIC_CONDITION(false))

    if (!(!m_BufferInfo.stored.load() && m_IoResult.IsSuccess()))
    {
        // 次のすべての条件を満たさない場合は即時返る
        // 1. 最後の書き込みが終了していない
        // 2. 以前に IO でエラーがない
        return 0u;
    }

    m_BufferInfo.address = buffer;
    m_BufferInfo.total = sendSize;
    m_BufferInfo.used = 0u;
    m_BufferInfo.stored.store(true); // コールバックでのバッファの上書き OK

    int numRunning = 0;
    do
    {
        int numfds = 0;
        // timeout = 1000msec を指定しているが、 libcURL 内部で 1msec に置き換えられる
        auto mr = curl_multi_wait(m_CurlMulti, NULL, 0, 1000, &numfds);
        NN_RESULT_THROW_UNLESS(mr == CURLM_OK, ResultCurlMultiApiError());

        // キャンセルされているかの検査
        NN_OLSC_IO_FAIL_WITH_RESULT_UNLESS(!IsCanceled(), ResultCanceledDuringDataAccess());

        if(numfds <= 0)
        {
            // 通信が始まっていない場合は待つ。
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }

        mr = curl_multi_perform(m_CurlMulti, &numRunning);
        NN_RESULT_THROW_UNLESS(mr == CURLM_OK, ResultCurlMultiApiError());
        if (numRunning <= 0)
        {
            NN_SDK_ASSERT(numRunning == 0);
            m_IoResult = ExtractMultiPerformError(m_CurlMulti);
            break;
        }
    } while (m_BufferInfo.stored.load());
    return m_BufferInfo.used;

#undef NN_OLSC_IO_FAIL_WITH_RESULT_UNLESS
}

Result UploaderBase::WaitForSendComplete() NN_NOEXCEPT
{
    int numRunning = 0;
    do
    {
        int numfds = 0;
        // timeout = 1000msec を指定しているが、 libcURL 内部で 1msec に置き換えられる
        auto mr = curl_multi_wait(m_CurlMulti, NULL, 0, 1000, &numfds);
        NN_RESULT_THROW_UNLESS(mr == CURLM_OK, ResultCurlMultiApiError());

        // キャンセルされているかの検査
        NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceledDuringDataAccess());

        if(numfds <= 0)
        {
            // 通信が始まっていない場合は待つ。
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }

        mr = curl_multi_perform(m_CurlMulti, &numRunning);
        NN_RESULT_THROW_UNLESS(mr == CURLM_OK, ResultCurlMultiApiError());
    } while (numRunning > 0);
    return ExtractMultiPerformError(m_CurlMulti);
}

Result UploaderBase::Open() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CurlMulti != nullptr);

    // multi-interface に登録
    NN_RESULT_DO(WebApiAccessorBase::Attach(m_CurlMulti));

    // いったんここで Open 済みにする
    bool success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            Close();
        }
    };

    // Fillbuffer で即時 wait 解除されるように pause しておく
    NN_RESULT_DO(WebApiAccessorBase::Pause(CURLPAUSE_ALL));

    // 通信の開始 (この時点で m_BufferInfo にバッファは設定されていない)
    int numRunning;
    auto mr = curl_multi_perform(m_CurlMulti, &numRunning);
    NN_RESULT_THROW_UNLESS(mr == CURLM_OK, ResultCurlMultiApiError());

    if (numRunning <= 0)
    {
        NN_SDK_ASSERT(numRunning == 0);
        NN_RESULT_DO(ExtractMultiPerformError(m_CurlMulti));
    }
    success = true;
    NN_RESULT_SUCCESS;
}

void UploaderBase::Close() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(WebApiAccessorBase::Detach(m_CurlMulti));
}

Result UploaderBase::ExtractMultiPerformError(CURLM* curlMulti) NN_NOEXCEPT
{
    CURLcode r = CURLE_OK;
    int numMessages;
    do
    {
        auto* pInfo = curl_multi_info_read(curlMulti, &numMessages);
        if (true
            && r == CURLE_OK
            && pInfo != nullptr
            && pInfo->msg == CURLMSG_DONE
            && pInfo->data.result != CURLE_OK)
        {
            r = pInfo->data.result;
        }
    } while (numMessages > 0);
    NN_RESULT_DO(WebApiAccessorBase::HandleCurlErrorWithContext(r));
    NN_RESULT_DO(WebApiAccessorBase::GetResult());
    NN_RESULT_SUCCESS;
}

Result UploaderBase::GetResult() const NN_NOEXCEPT
{
    NN_RESULT_DO(m_IoResult);
    NN_RESULT_DO(WebApiAccessorBase::GetResult());
    NN_RESULT_SUCCESS;
}

FsExporterUploader::FsExporterUploader(CURL* curlHandle, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
    : UploaderBase(curlHandle, pCancelable)
{
}

Result FsExporterUploader::Upload(fs::ISaveDataChunkExporter* exporter, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(buffer != nullptr);
    NN_SDK_ASSERT(bufferSize > 0);
    NN_SDK_ASSERT(m_UploadSize > 0);

    auto l_Buffer = reinterpret_cast<char*>(buffer);

    NN_RESULT_DO(Open());
    NN_UTIL_SCOPE_EXIT
    {
        Close();
    };

    size_t totalSentSize = 0;
    while (totalSentSize < m_UploadSize)
    {
        size_t pulledSize = 0;
        NN_RESULT_DO(exporter->Pull(&pulledSize, l_Buffer, bufferSize));

        NN_ABORT_UNLESS(pulledSize != 0); // m_UploadSize に誤りがあることになる

        auto sentSize = Send(l_Buffer, pulledSize);
        if(sentSize != pulledSize)
        {
            // 何かしらの通信エラー
            return GetResult();
        }
        totalSentSize += sentSize;
    }

    NN_RESULT_DO(WaitForSendComplete());
    NN_RESULT_DO(GetResult());
    NN_RESULT_SUCCESS;
}

}}}} //namespace nn::olsc::srv::transfer

