﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/olsc/srv/olsc_TransferTaskAgent.h>

namespace nn { namespace olsc { namespace srv {

TransferTaskAgent::TransferTaskAgent(TransferTaskDatabaseManager& ttdm, ThreadResource& threadResource, TransferTaskFactoryBase& taskFactory, TransferTaskStartEventManager& startEventManager, TransferTaskCompleteEventManager& completeEventManager,
    SeriesInfoDatabaseManager& seriesInfoDatabaseManager, SaveDataArchiveInfoCacheManager& sdaInfoCacheManager, database::ErrorHistoryDatabase& errorHistoryDatabase,
    TransferTaskBase::TransferTaskExecutionResource& executionResource) NN_NOEXCEPT
    :   m_Ttdm(ttdm), m_ThreadResource(threadResource), m_Started(false), m_CurrentTask(nullptr),
        m_TaskFactory(taskFactory), m_TaskExecutionThread({}),
        m_ExecutionStopCount(0), m_TaskStartEventManager(startEventManager), m_TaskCompleteEventManager(completeEventManager),
        m_ExecutionResource(executionResource), m_RetryWaiting(false), m_SeriesInfoDatabaseManager(seriesInfoDatabaseManager), m_SdaInfoCacheManager(sdaInfoCacheManager),
        m_ErrorHistoryDatabase(errorHistoryDatabase)
{
}

TransferTaskAgent::~TransferTaskAgent() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ExecutionLock)> lock(m_ExecutionLock);
    if (m_Started)
    {
        Stop();
    }
}

void TransferTaskAgent::CancelCurrentTask() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_TaskLock)> taskLock(m_TaskLock);
    if (m_CurrentTask)
    {
        m_CurrentTask->Cancel();
    }
}

void TransferTaskAgent::Start() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ExecutionLock)> lock(m_ExecutionLock);
    NN_SDK_REQUIRES(!m_Started);
    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_TaskExecutionThread, [](void* p) {
        auto t = reinterpret_cast<TransferTaskAgent*>(p);
        t->Execute();
    }, this, m_ThreadResource.stack.data(), m_ThreadResource.stack.size(), m_ThreadResource.priority));
    os::SetThreadName(&m_TaskExecutionThread, m_ThreadResource.name);
    os::StartThread(&m_TaskExecutionThread);
    m_Started = true;
}

void TransferTaskAgent::Stop() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ExecutionLock)> lock(m_ExecutionLock);
    NN_SDK_REQUIRES(m_Started);

    m_Started = false;
    m_TerminateEvent.Signal();
    os::DestroyThread(&m_TaskExecutionThread);
    m_TerminateEvent.Clear();
}

nn::util::optional<TransferTaskDetailInfo> TransferTaskAgent::GetExecutableTaskDetailInfo() NN_NOEXCEPT
{
    return m_Ttdm.FindDetailInfo([&](const TransferTaskDetailInfo& detailInfo) -> bool
    {
        // TODO: 実行禁止リスト(対象アプリ起動中等)に入っているなら executable ではないので false を返す。

        return detailInfo.state == TransferTaskDetailInfo::State::NotFinished;
    });
}

void TransferTaskAgent::Execute() NN_NOEXCEPT
{
    auto agentState = AgentState::TaskExecutable;

    m_ExecutionResource.curlHandle = curl_easy_init();
    NN_ABORT_UNLESS_NOT_NULL(m_ExecutionResource.curlHandle);
    NN_UTIL_SCOPE_EXIT{
        curl_easy_cleanup(m_ExecutionResource.curlHandle);
        m_ExecutionResource.curlHandle = nullptr;
    };

    while (agentState != AgentState::Terminated)
    {
        if (agentState == AgentState::TaskExecutable)
        {
            agentState = PerformExecutableState();
        }
        else if (agentState == AgentState::TaskNonExecutable)
        {
            agentState = PerformNonExecutableState();
        }
        else
        {
            NN_ABORT("Not come here.\n");
        }
    }

    ClearCurrentTask();
}

TransferTaskBase* TransferTaskAgent::GetCurrentTask() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_TaskLock.IsLockedByCurrentThread());
    return m_CurrentTask;
}

typename TransferTaskAgent::TaskLocker  TransferTaskAgent::LockTask() NN_NOEXCEPT
{
    return TaskLocker(m_TaskLock);
}

void  TransferTaskAgent::IncrementStopExecutionCount() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_TaskLock)> lock(m_TaskLock);
    if (m_CurrentTask)
    {
        NN_DETAIL_OLSC_TRACE("IncrementStopExecutionCount() is called. TransferTask is running.\n");
    }

    m_ExecutionStopCount++;
    if (m_ExecutionStopCount == 1)
    {
        m_ExecutionStopEvent.Signal();
    }
}

void  TransferTaskAgent::DecrementStopExecutionCount() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_TaskLock)> lock(m_TaskLock);
    m_ExecutionStopCount--;
    if (m_ExecutionStopCount == 0)
    {
        m_ExecutionStopEvent.Signal();
    }
}

typename TransferTaskAgent::AgentState TransferTaskAgent::PerformExecutableState() NN_NOEXCEPT
{
    const int TerminateEventIndex = 0;
    const int PostProcessEventIndex = 1;
    const int ExecutionStopEventIndex = 2;
    const int CurlTimerEventIndex = 3;
    const int ExecutionListUpdatedEventIndex = 4;
    const int TaskRetryEventIndex = 5;

    auto& executionListUpdatedEvent = m_Ttdm.GetExecutionListUpdatedEvent();

    NN_DETAIL_OLSC_TRACE("TransferTaskAgent TaskExecutableState. WaitingEvent.\n");
    auto signaled = os::WaitAny(
        m_TerminateEvent.GetBase(),
        m_PostProcessEvent.GetBase(),
        m_ExecutionStopEvent.GetBase(),
        m_CurlTimerEvent.GetBase(),
        executionListUpdatedEvent.GetBase(),
        m_TaskRetryEvent.GetBase());

    switch (signaled)
    {
        case TerminateEventIndex:
            {
                m_TerminateEvent.Clear();

                NN_DETAIL_OLSC_TRACE("TerminateEvent signaled.\n");
                return AgentState::Terminated;
            }

        case PostProcessEventIndex:
            {
                m_PostProcessEvent.Clear();
                NN_DETAIL_OLSC_TRACE("PostProcessEvent signaled.\n");

                ProcessPostProcessEvent();

                return CanExecute() ? AgentState::TaskExecutable : AgentState::TaskNonExecutable;
            }

        case ExecutionStopEventIndex:
            {
                m_ExecutionStopEvent.Clear();
                NN_DETAIL_OLSC_TRACE("ExecutionStopEvent signaled.\n");

                return CanExecute() ? AgentState::TaskExecutable : AgentState::TaskNonExecutable;
            }

        case CurlTimerEventIndex:
            {
                m_CurlTimerEvent.Clear();
                NN_DETAIL_OLSC_TRACE("curlTimer signaled.\n");

                ResetCurlHandle();

                return CanExecute() ? AgentState::TaskExecutable : AgentState::TaskNonExecutable;
            }

        case ExecutionListUpdatedEventIndex:
        case TaskRetryEventIndex:
            {
                executionListUpdatedEvent.Clear();
                m_TaskRetryEvent.Clear();

                NN_DETAIL_OLSC_TRACE("ExecutionListUpdatedEvent or TaskRetryEvent signaled.\n");

                if (ProcessTaskList())
                {
                    executionListUpdatedEvent.Signal();
                }
                // curlHandle のキャッシュを、連続するリクエストの 10 秒後に破棄する。
                m_CurlTimerEvent.StartOneShot(TimeSpan::FromSeconds(10));

                return CanExecute() ? AgentState::TaskExecutable : AgentState::TaskNonExecutable;
            }

        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

typename TransferTaskAgent::AgentState TransferTaskAgent::PerformNonExecutableState() NN_NOEXCEPT
{
    const int TerminateEventIndex = 0;
    const int ExecutionStopEventIndex = 1;
    const int RetryWaitingEventIndex = 2;
    const int CurlTimerEventIndex = 3;

    NN_DETAIL_OLSC_TRACE("TransferTaskAgent TaskNotExecutableState. WaitingEvent.\n");

    auto signaled = os::WaitAny(
        m_TerminateEvent.GetBase(),
        m_ExecutionStopEvent.GetBase(),
        m_RetryWaitingEvent.GetBase(),
        m_CurlTimerEvent.GetBase());

    switch (signaled)
    {
        case TerminateEventIndex:
            {
                m_TerminateEvent.Clear();

                return AgentState::Terminated;
            }

        case ExecutionStopEventIndex:
            {
                m_ExecutionStopEvent.Clear();

                return CanExecute() ? AgentState::TaskExecutable : AgentState::TaskNonExecutable;
            }

        case RetryWaitingEventIndex:
            {
                m_RetryWaitingEvent.Clear();
                m_RetryWaiting = false;

                return CanExecute() ? AgentState::TaskExecutable : AgentState::TaskNonExecutable;
            }

        case CurlTimerEventIndex:
            {
                m_CurlTimerEvent.Clear();
                NN_DETAIL_OLSC_TRACE("curlTimer signaled.\n");

                ResetCurlHandle();

                return CanExecute() ? AgentState::TaskExecutable : AgentState::TaskNonExecutable;
            }

        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

bool TransferTaskAgent::ProcessTaskList() NN_NOEXCEPT
{
    auto executable = GetExecutableTaskDetailInfo();
    if (!executable)
    {
        NN_DETAIL_OLSC_TRACE("Not found executable task\n");
        return false;
    }


    {
        std::lock_guard<decltype(m_TaskLock)> taskLock(m_TaskLock);
        if (m_ExecutionStopCount > 0)
        {
            NN_DETAIL_OLSC_TRACE("Skipping task execution because stop count is over 0.(count: %d)\n", m_ExecutionStopCount);
            return true;
        }

        auto lastSi = m_SeriesInfoDatabaseManager.Acquire(executable->uid)->Get(executable->appId);
        // TODO : suspendedInfo を context に統合後にコメントアウトを有効化
        // m_CurrentTask = m_TaskFactory.Create(*executable, lastSi, m_ExecutionResource);
        m_CurrentTask = m_TaskFactory.Create(*executable, lastSi, m_ExecutionResource, std::move(m_SuspendedInfo));
    }

    m_TaskStartEventManager.Signal();
    m_CurrentTask->Execute();

    auto& context = m_CurrentTask->GetContext();

    auto& latestSda = m_CurrentTask->TryGetLatestSaveDataArchive();
    auto& detailInfo = m_CurrentTask->GetDetailInfo();

    if (m_CurrentTask->IsSucceeded())
    {
        NN_DETAIL_OLSC_TRACE("Task succeeded\n");
        NN_DETAIL_OLSC_TRACE("Tid: %016llx\n", context.id);
        NN_ABORT_UNLESS(latestSda, "latestSda must be not null.\n");

        // 成功してるのでエラー履歴は消す
        m_ErrorHistoryDatabase.RemoveLastErrorInfo(detailInfo.uid, detailInfo.appId);

        // TODO: UL タスクの成功の場合、自動 UL 全体設定が ON なら、このアプリに対して 自動 UL 設定を ON にする

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ttdm.SetCompleted(context.id));

        {
            auto writeMount = m_SeriesInfoDatabaseManager.AcquireWriteMount(executable->uid);
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_SeriesInfoDatabaseManager.Acquire(executable->uid)->Update(executable->appId, latestSda->seriesInfo));
            NN_ABORT_UNLESS_RESULT_SUCCESS(writeMount.Commit());
        }
    }

    // TODO : 各ケースに入る result 整理とリトライまでの時間の設定
    else if (m_CurrentTask->IsCanceled())
    {
        NN_DETAIL_OLSC_TRACE("Task suspended\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ttdm.SetSuspended(context.id, context));

        m_SuspendedInfo = m_CurrentTask->DetachTransferTaskSuspendedInfo();
    }
    else if (m_CurrentTask->IsFatal())
    {
        m_ErrorHistoryDatabase.SetLastError(detailInfo.uid, detailInfo.appId, detailInfo.config.kind, m_CurrentTask->IsRetryable(), context.lastResult);

        // TODO: UL タスクの失敗の場合、自動 UL 全体設定が ON なら、このアプリに対して 自動 UL 設定を OFF にする
        NN_DETAIL_OLSC_TRACE("Task failed(NOT retryable): %08x\n", context.lastResult.GetInnerValueForDebug());
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ttdm.SetFailed(context.id, context.lastResult));
    }
    else if (m_CurrentTask->IsRetryable())
    {
        m_ErrorHistoryDatabase.SetLastError(detailInfo.uid, detailInfo.appId, detailInfo.config.kind, m_CurrentTask->IsRetryable(), context.lastResult);

        NN_DETAIL_OLSC_TRACE("Task failed(retryable): %08x\n", context.lastResult.GetInnerValueForDebug());
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ttdm.SetFailed(context.id, context.lastResult));
    }

    if (latestSda)
    {
        auto writeMount = m_SdaInfoCacheManager.AcquireWriteMount(executable->uid);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_SdaInfoCacheManager.Acquire(executable->uid)->Add(*latestSda));
        NN_ABORT_UNLESS_RESULT_SUCCESS(writeMount.Commit());
    }

    m_TaskCompleteEventManager.Signal();
    m_PostProcessEvent.Signal();

    return true;
}

void TransferTaskAgent::ClearCurrentTask() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_TaskLock)> taskLock(m_TaskLock);

    if (m_CurrentTask)
    {
        m_TaskFactory.Destroy(m_CurrentTask);
        m_CurrentTask = nullptr;
    }
}

void TransferTaskAgent::ProcessPostProcessEvent() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_TaskLock)> taskLock(m_TaskLock);

    if (m_ExecutionStopCount > 0)
    {
        NN_DETAIL_OLSC_TRACE("Skipping post process because stop count is over 0.(count: %d)\n", m_ExecutionStopCount);
        m_PostProcessEvent.Signal();
        return;
    }

    auto& context = m_CurrentTask->GetContext();


    if (m_CurrentTask->IsCanceled())
    {
        NN_DETAIL_OLSC_TRACE("Resume cancelled task.\n");
        m_TaskRetryEvent.Signal();
    }
    else if (m_CurrentTask->IsSucceeded() || m_CurrentTask->IsFatal())
    {
        m_Ttdm.RemoveTransferTask(context.id);
    }
    else if (m_CurrentTask->IsRetryable())
    {
        NN_DETAIL_OLSC_TRACE("Remove task context for retry\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ttdm.SetRetry(context.id));

        StartRetryWaiting();
    }

    ClearCurrentTask();
}

bool TransferTaskAgent::CanExecute() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_TaskLock)> lock(m_TaskLock);

    return m_ExecutionStopCount == 0 && !m_RetryWaiting;
}

void TransferTaskAgent::StartRetryWaiting() NN_NOEXCEPT
{
    m_RetryWaiting = true;
    // TODO: サーバエラーが連続する時はリトライ間隔を延ばす必要がある
    //       （サーバが死んでるにも関わらず、10秒間隔で全NXがアクセスするのはまずい）
    m_RetryWaitingEvent.StartOneShot(TimeSpan::FromSeconds(10));

    m_TaskRetryEvent.Signal();
}

void TransferTaskAgent::ResetCurlHandle() NN_NOEXCEPT
{
    curl_easy_cleanup(m_ExecutionResource.curlHandle);
    m_ExecutionResource.curlHandle = curl_easy_init();
    NN_ABORT_UNLESS_NOT_NULL(m_ExecutionResource.curlHandle);
}

}}} //namespace nn::olsc::srv

