﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/olsc/srv/olsc_DaemonControllerImpl.h>

#include <nn/account/account_Api.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_AutoUploadSettingDatabase.h>
#include <nn/olsc/srv/olsc_ForbiddenSaveDataIndicationImpl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace olsc { namespace srv {

Result DaemonControllerImpl::GetApplicationAutoUploadSetting(sf::Out<bool> outValue, const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT
{
    outValue.Set(GetApplicationAutoUploadEnabledImpl(uid, appId));
    NN_RESULT_SUCCESS;
}

bool DaemonControllerImpl::GetApplicationAutoUploadEnabledImpl(const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT
{
    auto setting = m_AutoUploadSettingManager.GetApplicationSetting(uid, appId);
    switch (setting)
    {
        case database::AutoUploadSettingDatabase::ApplicationAutoUploadSetting::NotSet:
            return m_AutoUploadSettingManager.GetGlobalSetting(uid);
        case database::AutoUploadSettingDatabase::ApplicationAutoUploadSetting::Disabled:
            return false;
        case database::AutoUploadSettingDatabase::ApplicationAutoUploadSetting::Enabled:
            return true;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

Result DaemonControllerImpl::SetApplicationAutoUploadSetting(const account::Uid& uid, ApplicationId appId, bool isEnabled) NN_NOEXCEPT
{
    auto setting = isEnabled
        ? database::AutoUploadSettingDatabase::ApplicationAutoUploadSetting::Enabled
        : database::AutoUploadSettingDatabase::ApplicationAutoUploadSetting::Disabled;

    m_AutoUploadSettingManager.SetApplicationSetting(uid, appId, setting);
    NN_RESULT_SUCCESS;
}

Result DaemonControllerImpl::SetGlobalAutoUploadSetting(const account::Uid& uid, bool isEnabled) NN_NOEXCEPT
{
    auto previous = m_AutoUploadSettingManager.SetGlobalSetting(uid, isEnabled);

    // TRUE から FALSE に遷移した場合は登録されている自動ULタスクを削除
    if (!isEnabled && previous)
    {
        // 既存の自動 UL タスクを削除
        m_Ttdm.RemoveAutoUploadTaskByUser(uid);
    }

    NN_RESULT_SUCCESS;
}

Result DaemonControllerImpl::GetGlobalAutoUploadSetting(sf::Out<bool> outValue, const account::Uid& uid) const NN_NOEXCEPT
{
    outValue.Set(m_AutoUploadSettingManager.GetGlobalSetting(uid));
    NN_RESULT_SUCCESS;
}

void DaemonControllerImpl::DeleteAllApplicationSetting(const account::Uid& uid) NN_NOEXCEPT
{
    m_AutoUploadSettingManager.CleanupApplicationSetting(uid);
}

Result DaemonControllerImpl::CreateForbiddenSaveDataInidication(sf::Out<sf::SharedPointer<IForbiddenSaveDataIndication>> pOut, ApplicationId appId) NN_NOEXCEPT
{
    auto p = sf::CreateSharedObjectEmplaced<IForbiddenSaveDataIndication, ForbiddenSaveDataIndicationImpl>(&m_ObjectMemoryResource, appId);
    NN_RESULT_THROW_UNLESS(p, olsc::ResultOutOfObjectMemoryResource());
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

}}} //namespace nn::olsc::srv

