﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_SeriesInfoDatabase.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/util/util_TFormatString.h>

namespace nn { namespace olsc { namespace srv { namespace database {
namespace{
}

const char* SeriesInfoDatabase::DatabaseImpl::GetEntryFileRelativePath() const NN_NOEXCEPT
{
    return "si_entry";
}

const char* SeriesInfoDatabase::DatabaseImpl::GetMetadataFileRelativePath() const NN_NOEXCEPT
{
    return "si_meta";
}

util::ReadMount SeriesInfoDatabase::DatabaseImpl::AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT
{
    return mountManager.AcquireUserSeriesInfoSaveForRead(m_Uid);
}

util::WriteMount SeriesInfoDatabase::DatabaseImpl::AcquireWriteMount(util::DefaultMountManager& mountManager) NN_NOEXCEPT
{
    return mountManager.AcquireUserSeriesInfoSaveForWrite(m_Uid);
}

nn::util::optional<SeriesInfo> SeriesInfoDatabase::Find(ApplicationId appId) const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Db)> dbLock(m_Db);

    auto entry = m_Db.FindIf([&](const Entry& e) -> bool {
        return e.appId == appId;
    });

    if (entry)
    {
        return entry->seriesInfo;
    }

    return nn::util::nullopt;
}

SeriesInfo SeriesInfoDatabase::Get(ApplicationId appId) const NN_NOEXCEPT
{
    auto entry = Find(appId);
    return (entry) ? *entry : InvalidSeriesInfo;
}

Result SeriesInfoDatabase::Update(ApplicationId appId, const SeriesInfo& seriesInfo) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Db)> dbLock(m_Db);

    Entry entry = { appId, seriesInfo };

    auto pred = [&](const Entry& e) -> bool {
        return e.appId == appId;
    };

    auto replaceResult = m_Db.ReplaceIf(pred, entry);
    NN_RESULT_THROW_UNLESS(replaceResult.IsSuccess() || olsc::ResultDatabaseEntryNotFound::Includes(replaceResult), replaceResult);
    if (!replaceResult.IsSuccess())
    {
        // 未登録だったので新規登録
        NN_RESULT_DO(m_Db.PushBack(entry));
    }

    NN_RESULT_SUCCESS;
}

Result SeriesInfoDatabase::Remove(ApplicationId appId) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Db)> dbLock(m_Db);

    auto pred = [&](const Entry& e) -> bool {
        return e.appId == appId;
    };

    NN_RESULT_TRY(m_Db.RemoveIf(pred))
        NN_RESULT_CATCH(olsc::ResultDatabaseEntryNotFound) {}
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

Result SeriesInfoDatabase::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Db)> dbLock(m_Db);

    NN_RESULT_DO(m_Db.Cleanup());

    NN_RESULT_SUCCESS;
}

}}}} //namespace nn::olsc::srv::database

