﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

#include <nn/nn_Result.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_SaveDataArchiveInfoCache.h>

namespace nn { namespace olsc { namespace srv { namespace adaptor {

class SaveDataArchiveInfoOutputStreamBase
{
public:
    void SetExtractStatus(SaveDataArchiveStatus status) NN_NOEXCEPT
    {
        m_ExtractStatus = status;
    }

    int GetCount() const NN_NOEXCEPT
    {
        return m_CurrentIndex;
    }

    virtual Result Output(const SaveDataArchiveInfo data[], int count) NN_NOEXCEPT = 0;

protected:
    nn::util::optional<SaveDataArchiveStatus> m_ExtractStatus = nn::util::nullopt;
    int m_CurrentIndex = 0;

    SaveDataArchiveInfoOutputStreamBase() NN_NOEXCEPT = default;
    ~SaveDataArchiveInfoOutputStreamBase() NN_NOEXCEPT = default;

    template <typename OutputFunc>
    Result OutputForEach(OutputFunc output, const SaveDataArchiveInfo data[], int count) NN_NOEXCEPT
    {
        for(int i = 0; i < count; i++)
        {
            // m_ExtractStatus が設定されていない場合、全部 output
            // m_ExtractStatus が設定されている場合は m_ExtractStatus と一致した status の Sda だけ output
            if (!m_ExtractStatus || m_ExtractStatus == data[i].status)
            {
                NN_RESULT_DO(output(data[i]));
                m_CurrentIndex++;
            }
        }
        NN_RESULT_SUCCESS;
    }
};

// ストレージ上の SdaInfoCache に直接書き出す OutputStream
class SaveDataArchiveInfoStorageOutputStream
    : public SaveDataArchiveInfoOutputStreamBase
{
public:
    NN_IMPLICIT SaveDataArchiveInfoStorageOutputStream(database::SaveDataArchiveInfoCache& sdaInfoCache) NN_NOEXCEPT
        : m_SdaInfoCache(sdaInfoCache)
    {
    }

    virtual Result Output(const SaveDataArchiveInfo data[], int count) NN_NOEXCEPT NN_OVERRIDE
    {
        auto output = [this](const SaveDataArchiveInfo& r) -> Result {
            return this->m_SdaInfoCache.Add(r);
        };

        return OutputForEach(output, data, count);
    }

private:
    database::SaveDataArchiveInfoCache& m_SdaInfoCache;
};

class SaveDataArchiveInfoMemoryOutputStream
    : public SaveDataArchiveInfoOutputStreamBase
{
public:
    SaveDataArchiveInfoMemoryOutputStream(SaveDataArchiveInfo* outBuffer, int outBufferLen) NN_NOEXCEPT
        : m_OutBuffer(outBuffer)
        , m_OutBufferLen(outBufferLen)
    {
    }

    virtual Result Output(const SaveDataArchiveInfo data[], int count) NN_NOEXCEPT NN_OVERRIDE
    {
        auto output = [this](const SaveDataArchiveInfo& r) -> Result {
            NN_RESULT_THROW_UNLESS(this->m_CurrentIndex < this->m_OutBufferLen, nn::olsc::ResultSaveDataArchiveUnacceptableObjectCount());
            m_OutBuffer[this->m_CurrentIndex] = r;
            NN_RESULT_SUCCESS;
        };

        return OutputForEach(output, data, count);
    }

private:
    SaveDataArchiveInfo* const m_OutBuffer;
    const int m_OutBufferLen;
};

class ComponentFileInfoMemoryOutputStream
{
public:
    ComponentFileInfoMemoryOutputStream(ComponentFileInfo* outBuffer, int outBufferLen) NN_NOEXCEPT
        : m_OutBuffer(outBuffer)
        , m_OutBufferLen(outBufferLen)
    {
    }

    int GetCount() const NN_NOEXCEPT
    {
        return m_CurrentIndex;
    }

    virtual nn::Result Output(const ComponentFileInfo data[], int count) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_CurrentIndex + count <= m_OutBufferLen, nn::olsc::ResultComponentFileUnacceptableObjectCount());
        std::memcpy(&m_OutBuffer[m_CurrentIndex], data, sizeof(ComponentFileInfo) * count);
        m_CurrentIndex += count;
        NN_RESULT_SUCCESS;
    }

private:
    ComponentFileInfo* const m_OutBuffer;
    const int m_OutBufferLen;
    int m_CurrentIndex = 0;
};

}}}} // namespace nn::olsc::srv::adaptor
