﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "olsc_KeySeedPackageAdaptor.h"

#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/srv/olsc_Util.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_StringUtil.h>


namespace nn { namespace olsc { namespace srv { namespace adaptor {

namespace {
    const int MaxPropertyCount = KeySeedPackageBuilder::MaxPropertyCount;
    enum FlagIndex : int
    {
        FlagIndex_KeySeedPackage = 0,
        FlagIndex_PropertyCount // Property 数
    };

    const char FieldString[][KeySeedPackageAdaptor::MaxPathLength] =
    {
        ".encoded_key_seed_package"
    };

    struct Flags
    {
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_KeySeedPackage> KeySeedPackage;
    };
} // namespace

KeySeedPackageBuilder::KeySeedPackageBuilder() NN_NOEXCEPT
{
    Reset();
}

void KeySeedPackageBuilder::Reset() NN_NOEXCEPT
{
    m_Imported.Reset();
}

bool KeySeedPackageBuilder::Validate() NN_NOEXCEPT
{
    // 必要な要素が全て設定されているかどうかの判定
    if(!m_Imported[FlagIndex_KeySeedPackage])
    {
        NN_DETAIL_OLSC_WARN("[Error] Ksp Validate Error\n");
        return false;
    }
    return true;
}

const fs::SaveDataTransferManagerVersion2::KeySeedPackage& KeySeedPackageBuilder::GetKeySeedPackage() const NN_NOEXCEPT
{
    return m_KeySeedPackage;
}

template<typename FlagType>
void KeySeedPackageBuilder::SetFlag() NN_NOEXCEPT
{
    m_Imported.template Set<FlagType>();
}

template<typename FlagType>
bool KeySeedPackageBuilder::IsUnset() const NN_NOEXCEPT
{
    return !m_Imported.template Test<FlagType>();
}

void KeySeedPackageBuilder::SetKeySeedPackage(const char ksp[], size_t kspSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(kspSize == fs::SaveDataTransferManagerVersion2::KeySeedPackage::Size, "%s %s SizeError", __FILE__, __FUNCTION__);
    NN_UNUSED(kspSize);
    std::memcpy(&m_KeySeedPackage, ksp, sizeof(m_KeySeedPackage));
    SetFlag<Flags::KeySeedPackage>();
}

// ----------------------

bool KeySeedPackageAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    NN_UNUSED(valueLength);
    if(!m_IsAcceptable)
    {
        return true;
    }

    size_t fullPathLength = 0;
    const char* FullPath = jsonPath.ToString(&fullPathLength);
    NN_SDK_ASSERT(fullPathLength > static_cast<size_t>(m_PathBufferLength), "%s %s pathLength Error", __FILE__, __FUNCTION__);

    if(m_Builder.IsUnset<Flags::KeySeedPackage>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_KeySeedPackage], MaxPathLength - m_PathBufferLength) == 0)
        {
            char hash[fs::SaveDataTransferManagerVersion2::KeySeedPackage::Size];
            size_t decodedLength;
            auto b64Error = nn::util::Base64::FromBase64String(&decodedLength, hash, sizeof(hash), value, nn::util::Base64::Mode_UrlSafe);

            // digest チェックは Validate でできないのでここで落とす
            m_IsAcceptable = (b64Error == nn::util::Base64::Status_Success) && (decodedLength == fs::SaveDataTransferManagerVersion2::KeySeedPackage::Size);

            if(m_IsAcceptable)
            {
                m_Builder.SetKeySeedPackage(hash, sizeof(hash));
            }
            else
            {
                NN_DETAIL_OLSC_WARN("Ksp parse error. '%s' is invalid value\n", FieldString[FlagIndex_KeySeedPackage]);
            }
            return true;
        }
    }
    return true;
} // NOLINT(impl/function_size)

bool KeySeedPackageAdaptor::NotifyObjectBeginImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    if(MakeAndCheckObjectPath(m_PathBuffer, sizeof(m_PathBuffer), jsonPath))
    {
        m_Builder.Reset();
        m_PathBufferLength = nn::util::Strnlen(m_PathBuffer, MaxPathLength);
        // パス長さが許容範囲外なら、Validate できないだけで処理は続行
        m_IsAcceptable = m_PathBufferLength < MaxPathLength;
        if(!m_IsAcceptable)
        {
            NN_DETAIL_OLSC_WARN("Sda parse error. pathLength is invalid\n");
        }
    }
    return true;
}

bool KeySeedPackageAdaptor::NotifyObjectEndImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match(m_PathBuffer))
    {
        m_PathBufferLength = 0;
        if(!(m_IsAcceptable && m_Builder.Validate()))
        {
            SetResult(ResultKeySeedPackageUnacceptableContent());
            return false;
        }
        m_IsAcceptable = false;
    }
    return true;
}

bool KeySeedPackageAdaptor::MakeAndCheckObjectPath(char* buffer, size_t bufferSize, const JsonPathType& jsonPath) NN_NOEXCEPT
{
    nn::util::TSNPrintf(buffer, bufferSize, "$");
    return jsonPath.Match(buffer);
}

Result KeySeedPackageAdaptor::GetKeySeedPackage(fs::SaveDataTransferManagerVersion2::KeySeedPackage* pOutKsp) const NN_NOEXCEPT
{
    NN_RESULT_DO(GetResult());
    *pOutKsp = m_Builder.GetKeySeedPackage();
    NN_RESULT_SUCCESS;
}

}}}} //namespace nn::olsc::srv
