﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "detail/olsc_Initialization.h"

#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_DaemonController.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace olsc {

void swap(DaemonController& lhs, DaemonController& rhs) NN_NOEXCEPT
{
    std::swap(lhs.m_Interface, rhs.m_Interface);
}

DaemonController::DaemonController() NN_NOEXCEPT : m_Interface(nullptr)
{
}

DaemonController::~DaemonController() NN_NOEXCEPT
{
    if (m_Interface)
    {
        sf::ReleaseSharedObject(m_Interface);
        m_Interface = nullptr;
    }
}

DaemonController::DaemonController(DaemonController&& rhs) NN_NOEXCEPT : m_Interface(rhs.m_Interface)
{
    rhs.m_Interface = nullptr;
}

DaemonController::DaemonController(srv::IDaemonController* interface) NN_NOEXCEPT : m_Interface(interface)
{
}

DaemonController& DaemonController::operator=(DaemonController&& rhs) NN_NOEXCEPT
{
    DaemonController t(std::move(rhs));
    swap(*this, t);
    return *this;
}

bool DaemonController::IsAutoUploadEnabled(const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT
{
    return GetApplicationAutoUploadSetting(uid, appId);
}

void DaemonController::SetAutoUploadEnabled(const account::Uid& uid, ApplicationId appId, bool enabled) NN_NOEXCEPT
{
    return SetApplicationAutoUploadSetting(uid, appId, enabled);
}

bool DaemonController::GetApplicationAutoUploadSetting(const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    bool isEnabled;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetApplicationAutoUploadSetting(&isEnabled, uid, appId));
    return isEnabled;
}

void DaemonController::SetApplicationAutoUploadSetting(const account::Uid& uid, ApplicationId appId, bool enabled) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->SetApplicationAutoUploadSetting(uid, appId, enabled));
}

void DaemonController::SetGlobalAutoUploadSetting(const account::Uid& uid, bool enabled) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->SetGlobalAutoUploadSetting(uid, enabled));
}

bool DaemonController::GetGlobalAutoUploadSetting(const account::Uid& uid) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    bool isEnabled;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->GetGlobalAutoUploadSetting(&isEnabled, uid));
    return isEnabled;
}

Stopper DaemonController::StopAutonomyTaskExecution() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    NN_DETAIL_OLSC_WARN("Currently, StopAutonomyTaskExecution() is not implemented. \n");
    return Stopper();
}

ForbiddenSaveDataIndication DaemonController::CreateForbiddenSaveDataInidication(ncm::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_Interface);
    sf::SharedPointer<srv::IForbiddenSaveDataIndication> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->CreateForbiddenSaveDataInidication(&p, {appId.value}));
    return ForbiddenSaveDataIndication(p.Detach());
}

}} // namespace nn::olsc
