﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "detail/olsc_Initialization.h"

#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_ApiForPrivate.h>
#include <nn/olsc/olsc_ApiForSystemService.h>
#include <nn/olsc/olsc_DaemonController.h>
#include <nn/olsc/olsc_DataTransferPolicyApi.h>
#include <nn/olsc/olsc_RemoteStorageController.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/olsc_TransferTaskListController.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace olsc {

namespace {

detail::InitializationManager g_InitializationManager = NN_OLSC_INITIALIZATION_MANAGER_INITIALIZER;
detail::ObjectHolder g_ObjectHolder;

void EnsureInitializationForSystemService() NN_NOEXCEPT
{
    if (!g_InitializationManager)
    {
        InitializeForSystemService();
    }
}

}

void InitializeForSystemService() NN_NOEXCEPT
{
    g_InitializationManager.Initialize([&]()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.AcquireForSystemService());
    });
}

void FinalizeForSystemService() NN_NOEXCEPT
{
    g_InitializationManager.Finalize([&]
    {
        g_ObjectHolder.Release();
    });
}

void InitializeWithServiceObject(sf::SharedPointer<srv::IOlscServiceForSystemService> serviceObject) NN_NOEXCEPT
{
    g_InitializationManager.Initialize([&]()
    {
        g_ObjectHolder.InitializeWithServiceObject(serviceObject);
    });
}

void OpenTransferTaskListController(TransferTaskListController* out) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(out);
    EnsureInitializationForSystemService();

    nn::sf::SharedPointer<nn::olsc::srv::ITransferTaskListController> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().GetTransferTaskListController(&p));
    TransferTaskListController t(p.Detach());

    *out = std::move(t);
}

void OpenRemoteStorageController(RemoteStorageController* out, const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(out);
    EnsureInitializationForSystemService();

    nn::sf::SharedPointer<nn::olsc::srv::IRemoteStorageController> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().GetRemoteStorageController(&p));
    RemoteStorageController t(p.Detach(), uid);

    *out = std::move(t);
}

RemoteStorageController OpenRemoteStorageController(const account::Uid& uid) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();

    nn::sf::SharedPointer<nn::olsc::srv::IRemoteStorageController> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().GetRemoteStorageController(&p));
    return RemoteStorageController(p.Detach(), uid);
}

TransferTaskListController OpenTransferTaskListController() NN_NOEXCEPT
{
    EnsureInitializationForSystemService();

    nn::sf::SharedPointer<nn::olsc::srv::ITransferTaskListController> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().GetTransferTaskListController(&p));
    return TransferTaskListController(p.Detach());
}

DaemonController OpenDaemonController() NN_NOEXCEPT
{
    EnsureInitializationForSystemService();

    nn::sf::SharedPointer<nn::olsc::srv::IDaemonController> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().GetDaemonController(&p));

    return DaemonController(p.Detach());
}

void DeleteAllTransferTask() NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().DeleteAllTransferTask());
}

void DeleteAllSeriesInfo(const account::Uid& uid) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().DeleteAllSeriesInfo(uid));
}

void DeleteAllSaveDataArchiveInfoCache(const account::Uid& uid) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().DeleteAllSdaInfoCache(uid));
}

void DeleteAllApplicationSetting(const account::Uid& uid) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().DeleteAllApplicationSetting(uid));
}

void DeleteAllTransferTaskErrorInfo() NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().DeleteAllTransferTaskErrorInfo());
}

void RegisterTransferTaskErrorInfo(const TransferTaskErrorInfo& errorInfo) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().RegisterTransferTaskErrorInfo(errorInfo));
}

int ListTransferTaskErrorInfo(TransferTaskErrorInfo out[], int maxOutCount, int offset) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();

    int listedCount;
    sf::OutArray<TransferTaskErrorInfo> outBuffer(out, maxOutCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().ListTransferTaskErrorInfo(outBuffer, &listedCount, offset));

    return listedCount;
}

void RemoveTransferTaskErrorInfo(const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().RemoveTransferTaskErrorInfo(uid, appId));
}

int GetTransferTaskErrorInfoCount() NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    int count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().GetTransferTaskErrorInfoCount(&count));

    return count;
}

Result AddSaveDataArchiveInfoCache(const account::Uid& uid, srv::SaveDataArchiveInfo& sda) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().AddSaveDataArchiveInfoCache(uid, sda));
    NN_RESULT_SUCCESS;
}

void DeleteSeriesInfo(const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().DeleteSeriesInfo(uid, appId));
}

void GetSeriesInfo(srv::SeriesInfo* out, const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().GetSeriesInfo(out, uid, appId));
}

bool GetDataTransferPolicy(DataTransferPolicy* out, ApplicationId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);

    EnsureInitializationForSystemService();
    NN_RESULT_TRY(g_ObjectHolder.GetForSystemService().GetDataTransferPolicy(out, id))
        NN_RESULT_CATCH_ALL
        {
            return false;
        }
    NN_RESULT_END_TRY;
    return true;
}

void DeleteDataTransferPolicy(ApplicationId appId) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().DeleteDataTransferPolicy(appId));
}

Result RequestUpdateDataTransferPolicy(AsyncRequest* out, ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);
    EnsureInitializationForSystemService();

    sf::SharedPointer<srv::IAsyncResult> p;
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().RequestUpdateDataTransferPolicyAsync(&p, appId));
    AsyncRequest t(p.Detach());
    *out = std::move(t);

    NN_RESULT_SUCCESS;
}

void RunAutonomyUploadTransferTaskRegistration(const account::Uid& uid) NN_NOEXCEPT
{
    EnsureInitializationForSystemService();
    auto result = g_ObjectHolder.GetForSystemService().RunAutonomyUploadTransferTaskRegistration(uid);
    if (result.IsFailure())
    {
        if (result <= olsc::ResultNotImplemented())
        {
            NN_DETAIL_OLSC_WARN("ScanAutoUpload() is not implemented.\n");
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
}

}} // namespace nn::olsc
