﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/am/am_Shim.h>
#include <nn/am/am_Result.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/oe/oe_GamePlayRecordingApi.private.h>
#include <nn/oe/oe_GamePlayRecordingApiForDebug.private.h>
#include <nn/oe/oe_SelfControlApis.h>
#include <nn/grc/grc_Result.h>

#include <cstring>

namespace nn { namespace oe {

NN_STATIC_ASSERT( WorkBufferSizeForGamePlayRecording % os::MemoryPageSize == 0 );

namespace {
    bool g_IsGamePlayRecordingInitialized{false};
}

//-----------------------------------------------------------------------------
//  常時動画撮影機能
//
void EnableGamePlayRecording(void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT( !g_IsGamePlayRecordingInitialized );
    NN_SDK_REQUIRES( reinterpret_cast<uintptr_t>(buffer) % os::MemoryPageSize == 0 );
    NN_SDK_REQUIRES( WorkBufferSizeForGamePlayRecording == size );

    // TransferMemory 化して am に渡す
    {
        os::TransferMemory transMem(buffer, size, os::MemoryPermission_None);
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->InitializeGamePlayRecording( sf::NativeHandle(transMem.Detach(), true), size ));
    }
    g_IsGamePlayRecordingInitialized = true;

    // 録画開始
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetApplicationFunctions()->SetGamePlayRecordingState( static_cast<int>(am::service::GamePlayRecordingState::Recording) ) );
}

bool IsGamePlayRecordingSupported() NN_NOEXCEPT
{
    bool isSupported;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->IsGamePlayRecordingSupported(&isSupported));
    return isSupported;
}

void RestartGamePlayRecording() NN_NOEXCEPT
{
    oe::EnableRecording();
}

void PauseGamePlayRecording() NN_NOEXCEPT
{
    oe::DisableRecording();
}

void RequestToSaveRecordingForDebug() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->RequestFlushGamePlayingMovieForDebug());
}

Result SaveGamePlayRecordingForDebug(TimeSpan maxTime) NN_NOEXCEPT
{
retry:
    sf::NativeHandle handle;
    NN_RESULT_TRY(am::GetApplicationFunctions()->StartContinuousRecordingFlushForDebug(&handle, maxTime.GetNanoSeconds()))
        NN_RESULT_CATCH(grc::ResultFlushIsInProcess)
        {
            os::SleepThread(TimeSpan::FromMilliSeconds(100));
            goto retry;
        }
        NN_RESULT_CATCH(am::ResultDevelopmentFunctionCalled)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY
    os::SystemEvent e;
    e.AttachReadableHandle(handle.GetOsHandle(), false, os::EventClearMode_ManualClear);
    e.Wait();
    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
// アプリ動画撮影の許可
//
void EnableRecording() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetApplicationFunctions()->SetGamePlayRecordingState( static_cast<int>(am::service::GamePlayRecordingState::Recording) ) );
}


//-----------------------------------------------------------------------------
// アプリ動画撮影の禁止
//
void DisableRecording() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetApplicationFunctions()->SetGamePlayRecordingState( static_cast<int>(am::service::GamePlayRecordingState::Stopped) ) );
}


//-----------------------------------------------------------------------------
// スクリーンショット撮影の禁止
//
void DisableScreenShot() NN_NOEXCEPT
{
    auto result = am::GetSelfController()->SetScreenShotPermission(applet::ScreenShotPermission_Forbid);
    if (!result.IsSuccess())
    {
        if (result <= am::ResultLackOfCapability())
        {
            // nmeta ファイルで許可されていない場合はスルーする
            return;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}


//-----------------------------------------------------------------------------
// スクリーンショット撮影の許可
//
void EnableScreenShot() NN_NOEXCEPT
{
    auto result = am::GetSelfController()->SetScreenShotPermission(applet::ScreenShotPermission_Permit);
    if (!result.IsSuccess())
    {
        if (result <= am::ResultLackOfCapability())
        {
            // nmeta ファイルで許可されていない場合はスルーする
            return;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}


//-----------------------------------------------------------------------------
// アルバム静止画やアルバム動画ファイルの画像の回転方向の設定
//
void SetAlbumImageOrientation(album::ImageOrientation orientation) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(    orientation == album::ImageOrientation_None
                     || orientation == album::ImageOrientation_Rotate90
                     || orientation == album::ImageOrientation_Rotate180
                     || orientation == album::ImageOrientation_Rotate270 );

    // 静止画と動画の両方に影響する
    auto result = am::GetSelfController()->SetAlbumImageOrientation(static_cast<int>(orientation));
    if (!result.IsSuccess())
    {
        if (result <= am::ResultLackOfCapability())
        {
            // nmeta ファイルで許可されていない場合はスルーする
            return;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}

void SetScreenShotImageOrientation(album::ImageOrientation orientation) NN_NOEXCEPT
{
    SetAlbumImageOrientation(orientation);
}


//-----------------------------------------------------------------------------
//  権利表記関連
//
void InitializeCopyrightFrameBuffer(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    // フレームバッファのサイズは 256KiB の倍数でないといけない模様
    NN_SDK_REQUIRES( bufferSize == CopyrightFrameBufferSize );

    std::memset(buffer, 0, bufferSize);
    os::TransferMemory trmem(buffer, bufferSize, os::MemoryPermission_None);
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetApplicationFunctions()->InitializeApplicationCopyrightFrameBuffer(sf::NativeHandle(trmem.Detach(), true), bufferSize, CopyrightFrameBufferWidth, CopyrightFrameBufferHeight) );
}

void SetCopyrightImage(const void* buffer, size_t bufferSize, int x, int y, int width, int height, WindowOriginMode originMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(bufferSize == width * height * sizeof(Bit32));
    NN_SDK_REQUIRES(x >= 0 && y >= 0 && (x + width <= CopyrightFrameBufferWidth) && (y + height <= CopyrightFrameBufferHeight));

    sf::InBuffer inBuffer(static_cast<const char*>(buffer), bufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetApplicationFunctions()->SetApplicationCopyrightImage(inBuffer, x, y, width, height, originMode) );
}

void SetCopyrightVisibility(bool isVisible) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetApplicationFunctions()->SetApplicationCopyrightVisibility(isVisible) );
}


//-----------------------------------------------------------------------------

}}  // namespace nn::oe

