﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/oe/oe_InStoreDemoApis.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/applet/applet_Types.h>
#include <nn/applet/applet_Apis.h>
#include <nn/applet/applet_LibraryAppletSelf.h>
#include <nn/applet/applet_ApplicationSelf.h>
#include <nn/applet/applet_Storage.h>
#include <nn/oe/oe_OperationModeApis.h>
#include <nn/oe/oe_RidSpecificApis.private.h>
#include <nn/oe/oe_LibraryAppletControlApis.h>
#include <nn/oe/oe_LibraryAppletControlTypes.h>
#include <nn/oe/oe_SelfControlApis.h>
#include <nn/oe/oe_SpecificToOceanApis.private.h>
#include <nn/audio/audio_Applet.h>

#include <nn/am/am_Shim.h>
#include <nn/am/service/am_CommonTypes.h>
#include <nn/am/am_Result.h>

#include <mutex>
#include <algorithm>

namespace nn { namespace oe {

static_assert( static_cast<Bit32>(    oe::MessageExitRequest) ==
               static_cast<Bit32>(applet::Message_Exit), "");

static_assert( static_cast<Bit32>(    oe::MessageFocusStateChanged) ==
               static_cast<Bit32>(applet::Message_FocusStateChanged), "");
static_assert( static_cast<Bit32>(    oe::MessageResume) ==
               static_cast<Bit32>(applet::Message_RestartFromSuspend), "");

static_assert( static_cast<Bit32>(    oe::MessageOperationModeChanged) ==
               static_cast<Bit32>(applet::Message_OperationModeChanged), "");
static_assert( static_cast<Bit32>(    oe::MessagePerformanceModeChanged) ==
               static_cast<Bit32>(applet::Message_PerformanceModeChanged), "");

static_assert( static_cast<Bit32>(         oe::FocusState_InFocus) ==
               static_cast<Bit32>(am::service::FocusState_InFocus), "");
static_assert( static_cast<Bit32>(         oe::FocusState_OutOfFocus) ==
               static_cast<Bit32>(am::service::FocusState_OutOfFocus), "");
static_assert( static_cast<Bit32>(         oe::FocusState_Background) ==
               static_cast<Bit32>(am::service::FocusState_Background), "");

//-----------------------------------------------------------------------------

namespace {

os::SystemEventType g_MessageEvent;

struct MutexForInitialization
{
    os::MutexType    m_Mutex;

    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex( &m_Mutex );
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex( &m_Mutex );
    }
};

MutexForInitialization g_AppletMutex = { NN_OS_MUTEX_INITIALIZER(false) };

ThemeColorType g_ThemeColorType = ThemeColorType_Default;

os::Mutex g_EnableHandlingForExitRequestCountMutex{false};
int g_EnableHandlingForExitRequestCount = 0;

void WaitForFirstExpectedMessage() NN_NOEXCEPT
{
    for (;;)
    {
        auto msg = nn::oe::PopNotificationMessage();
        if (msg == nn::oe::MessageFocusStateChanged)
        {
            auto current = nn::oe::GetCurrentFocusState();
            if (current == nn::oe::FocusState_InFocus)
            {
                // この状態になるまで待つ
                return;
            }
        }
    }
}

MutexForInitialization g_OutOfFocusSuspendingMutex = { NN_OS_MUTEX_INITIALIZER(false) };

}   // namespace anonymous

//-----------------------------------------------------------------------------
//  InitializeApplet()
//
//  静的オブジェクトのコンストラクタはまだ実行されていない段階で呼ばれる。
//
void InitializeApplet() NN_NOEXCEPT
{
    std::lock_guard<MutexForInitialization> lock(g_AppletMutex);

    // ２回目以降の呼出時は何もしない
    static bool alreadyCalled = false;
    if (alreadyCalled)
    {
        return;
    }
    alreadyCalled = true;

    // アプリケーションとしての初期化を行ない、AM 傘下に入る
    if (!am::InitializeApplicationInterfaces())
    {
        // 非アプリや、"appletOE" 権限のないテストなどで初期化に失敗した場合は
        // アボートせずにリターンするようにする。
        return;
    }

    // メッセージ用の SystemEvent を取得

    sf::NativeHandle sfHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetCommonStateGetter()->GetEventHandle(&sfHandle) );
    os::AttachReadableHandleToSystemEvent(&g_MessageEvent,
                                          sfHandle.GetOsHandle(),
                                          sfHandle.IsManaged(),
                                          os::EventClearMode_ManualClear);
    sfHandle.Detach();

    // 最初のメッセージの待機と取得
    WaitForFirstExpectedMessage();

    // 以下の呼出は不要だが、AcquireForegroundRights() 削除時に一緒に削除する
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetWindowController()->AcquireForegroundRights() );
    SetFocusHandlingMode(FocusHandlingMode_Suspend);
}


//-----------------------------------------------------------------------------
//  通知メッセージの取得（内部関数）
//
Message GetNotificationMessage() NN_NOEXCEPT
{
    am::AppletMessage message;
    NN_RESULT_TRY( am::GetCommonStateGetter()->ReceiveMessage(&message) )
        NN_RESULT_CATCH(am::ResultNoMessage)
        {
            return 0;
        }
    NN_RESULT_END_TRY
    return static_cast<Message>(message.message[0]);
}


//-----------------------------------------------------------------------------
//  マスターボリューム期待値の設定
//
void SetExpectedVolumeBalance(float appVolume, float laVolume) NN_NOEXCEPT
{
    NN_SDK_ASSERT(VolumeValueMin <= appVolume && appVolume <= VolumeValueMax,
        "第 1 引数で指定された volume 値が異常です（volume=%f）", appVolume);
    NN_SDK_ASSERT(VolumeValueMin <= laVolume && laVolume <= VolumeValueMax,
        "第 2 引数で指定された volume 値が異常です（volume=%f）", laVolume);

    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetAudioController()->SetExpectedMasterVolume(appVolume, laVolume) );
}


//-----------------------------------------------------------------------------
//  マスターボリューム期待値の取得
//
void GetExpectedVolumeBalance(float* pAppVolume, float* pLaVolume) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetAudioController()->GetMainAppletExpectedMasterVolume(pAppVolume));
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetAudioController()->GetLibraryAppletExpectedMasterVolume(pLaVolume));
}


//-----------------------------------------------------------------------------
//  ライブラリアプレット起動時に期待するテーマカラーを設定します。
//
void SetExpectedThemeColor( ThemeColorType themeColorType ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS( themeColorType < nn::oe::ThemeColorType_Max );
    g_ThemeColorType = themeColorType;
}


//-----------------------------------------------------------------------------
//  ライブラリアプレット起動時に期待するテーマカラーを設定します。
//
void SetExpectedThemeColorForSystem( ThemeColorType themeColorType ) NN_NOEXCEPT
{
    g_ThemeColorType = themeColorType;
}


//-----------------------------------------------------------------------------
//  ライブラリアプレット起動時に期待しているテーマカラーを取得します。
//
ThemeColorType GetExpectedThemeColor() NN_NOEXCEPT
{
    return g_ThemeColorType;
}


//-----------------------------------------------------------------------------
//  ThemeColorType_Default が意味しているテーマカラーを取得します。
//
ThemeColorType GetDefaultThemeColor() NN_NOEXCEPT
{
    // デフォルトは、本体設定のテーマで指定した色に合わせる
    return ThemeColorType_User;
}


//-----------------------------------------------------------------------------
//  通知メッセージの待機と取得
//
Message PopNotificationMessage() NN_NOEXCEPT
{
    for (;;)
    {
        os::WaitSystemEvent(&g_MessageEvent);
        auto message = GetNotificationMessage();
        if (message != 0)
        {
            return message;
        }
    }
}


//-----------------------------------------------------------------------------
//  通知メッセージの取得（ポーリング）
//
bool TryPopNotificationMessage(Message* pOutMessage) NN_NOEXCEPT
{
    if (!os::TryWaitSystemEvent(&g_MessageEvent))
    {
        return false;
    }

    am::AppletMessage message;
    NN_RESULT_TRY( am::GetCommonStateGetter()->ReceiveMessage(&message) )
        NN_RESULT_CATCH(am::ResultNoMessage)
        {
            return false;
        }
    NN_RESULT_END_TRY

    *pOutMessage = static_cast<Message>(message.message[0]);
    return true;
}


//-----------------------------------------------------------------------------
//  通知メッセージイベントの取得
//
os::SystemEventType* GetNotificationMessageEvent() NN_NOEXCEPT
{
    return &g_MessageEvent;
}


//-----------------------------------------------------------------------------
// プログラム状態変更通知と自動中断機能のハンドリングモード指定
//
void SetFocusHandlingMode(FocusHandlingMode mode) NN_NOEXCEPT
{
    std::lock_guard<MutexForInitialization> lk(g_OutOfFocusSuspendingMutex);

    auto p = am::GetSelfController();
    switch (mode)
    {
        case FocusHandlingMode_Suspend:
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetFocusHandlingMode(false, false, true));
            NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetOutOfFocusSuspendingEnabled(false));
            break;
        }
        case FocusHandlingMode_Notify:
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetFocusHandlingMode(true, true, false));
            NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetOutOfFocusSuspendingEnabled(false));
            break;
        }
        case FocusHandlingMode_SuspendAndNotify:
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetFocusHandlingMode(true, false, true));
            NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetOutOfFocusSuspendingEnabled(false));
            break;
        }
        case FocusHandlingMode_InFocusOnly:
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetFocusHandlingMode(false, false, true));
            NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetOutOfFocusSuspendingEnabled(true));
            break;
        }
        default: NN_UNEXPECTED_DEFAULT;
    }
}

void SetFocusHandlingModeForDebug(bool needFocusNotification, bool needBackgroundNotification, bool needSuspendBackground, bool needSuspendOutOfFocus) NN_NOEXCEPT
{
    std::lock_guard<MutexForInitialization> lk(g_OutOfFocusSuspendingMutex);

    auto p = am::GetSelfController();
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetFocusHandlingMode(needFocusNotification, needBackgroundNotification, needSuspendBackground) );
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->SetOutOfFocusSuspendingEnabled(needSuspendOutOfFocus));
}

//-----------------------------------------------------------------------------
// 現在のフォーカス状態の取得
//
FocusState GetCurrentFocusState() NN_NOEXCEPT
{
    Bit8 focusState;
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetCommonStateGetter()->GetCurrentFocusState( &focusState ) );

    return static_cast<FocusState>(focusState);
}


//-----------------------------------------------------------------------------
// 動作の再開を示すメッセージ通知の有無の指定
//
void SetResumeNotificationEnabled(bool enabled) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetSelfController()->SetRestartMessageEnabled( enabled ) );
}


//-----------------------------------------------------------------------------
// 動作モード変更通知有無の指定
//
void SetOperationModeChangedNotificationEnabled(bool enabled) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetSelfController()->SetOperationModeChangedNotification( enabled ) );
}


//-----------------------------------------------------------------------------
// 性能モード変更通知有無の指定
//
void SetPerformanceModeChangedNotificationEnabled(bool enabled) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetSelfController()->SetPerformanceModeChangedNotification( enabled ) );
}


//-----------------------------------------------------------------------------
// 現在の動作モードの取得
//
OperationMode GetOperationMode() NN_NOEXCEPT
{
    Bit8 mode;
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetCommonStateGetter()->GetOperationMode( &mode ) );

    return static_cast<OperationMode>(mode);
}


//-----------------------------------------------------------------------------
// 現在の性能モードの取得
//
PerformanceMode GetPerformanceMode() NN_NOEXCEPT
{
    Bit32 mode;
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetCommonStateGetter()->GetPerformanceMode( &mode ) );

    return static_cast<PerformanceMode>(mode);
}


//-----------------------------------------------------------------------------
// 起動パラメータの取得
//
bool TryPopLaunchParameter(size_t* pOutParameterSize, void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    applet::StorageHandle handle;
    if (!applet::TryPopFromApplicationParameterChannel(&handle, applet::LaunchParameterKind_User))
    {
        return false;
    }
    NN_UTIL_SCOPE_EXIT
    {
        applet::ReleaseStorage(handle);
    };

    auto realStorageSize = applet::GetStorageSize(handle);
    auto readSize = std::min(realStorageSize, bufferSize);
    if (readSize > 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(applet::ReadFromStorage(handle, 0, pOutBuffer, readSize));
    }
    *pOutParameterSize = realStorageSize;
    return true;
}


//-----------------------------------------------------------------------------
// 終了リクエストへハンドリングのための API
//
void EnterExitRequestHandlingSection() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_EnableHandlingForExitRequestCountMutex)> lk(g_EnableHandlingForExitRequestCountMutex);
    if (g_EnableHandlingForExitRequestCount == 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->LockExit());
    }
    ++g_EnableHandlingForExitRequestCount;
}

void LeaveExitRequestHandlingSection() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_EnableHandlingForExitRequestCountMutex)> lk(g_EnableHandlingForExitRequestCountMutex);
    NN_SDK_ASSERT(g_EnableHandlingForExitRequestCount > 0);
    --g_EnableHandlingForExitRequestCount;
    if (g_EnableHandlingForExitRequestCount == 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->UnlockExit());
    }
}

//-----------------------------------------------------------------------------
// アプリの自発終了（OceanKit 専用、アプリ異常終了シーンを出さない）
//
NN_NORETURN void ExitApplication() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_EnableHandlingForExitRequestCountMutex)> lk(g_EnableHandlingForExitRequestCountMutex);
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->Exit());
    for (;;)
    {
        os::SleepThread( TimeSpan::FromDays(1) );
    }
}

//-----------------------------------------------------------------------------
// アプリの終了と試遊台メニューへの復帰（試遊台専用）
// am 的には SA に Message_RequestToGoBackQuestMenu を通知して
// SA からアプリ強制終了して、QuestMenu を起動する流れとなる。
//
NN_NORETURN void ExitApplicationAndGoBackToInStoreDemoMenu() NN_NOEXCEPT
{
    NN_ABORT_UNLESS( IsInStoreDemoModeEnabled() );

    std::lock_guard<decltype(g_EnableHandlingForExitRequestCountMutex)> lk(g_EnableHandlingForExitRequestCountMutex);

    // oe::EnterExitRequestHandlingSection() の効果を強制的に解除
    if (g_EnableHandlingForExitRequestCount > 0)
    {
        g_EnableHandlingForExitRequestCount = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->UnlockExit());
    }

    // 試遊台環境でない場合はアボートする
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->ExitAndGoBackQuestMenu());
    for (;;)
    {
        os::SleepThread( TimeSpan::FromDays(1) );
    }
}


}}  // namespace nn::oe

