﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os/os_Mutex.h>
#include <nn/am/am_Shim.h>
#include <nn/applet/applet_IdlePolicyApi.h>
#include <nn/oe/oe_DisableAutoSleepApi.h>
#include <nn/oe/oe_IdlePolicyApi.h>
#include <nn/oe/oe_IdlePolicyApiForApplicationSpecificControl.h>
#include <nn/oe/oe_Result.h>
#include <nn/oe/oe_MediaPlaybackSectionApi.private.h>


namespace nn { namespace oe {

namespace {
    os::Mutex g_MediaPlaybackSectionMutex{false};
    uint64_t  g_MediaPlaybackSectionCount{0};
}

//-----------------------------------------------------------------------------
//  無操作状態検出時の自動スリープ時間、自動低輝度化時間の延長機能関連
//
void SetUserInactivityDetectionTimeExtended(bool isExtended) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetIdleTimeDetectionExtension(isExtended ? am::IdleTimeDetectionExtension_Enabled : am::IdleTimeDetectionExtension_Disabled));
}

void SetUserInactivityDetectionTimeExtendedUnsafe(bool isExtended) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetIdleTimeDetectionExtension(isExtended ? am::IdleTimeDetectionExtension_EnabledUnsafe : am::IdleTimeDetectionExtension_Disabled));
}

bool IsUserInactivityDetectionTimeExtended() NN_NOEXCEPT
{
    uint32_t extension;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->GetIdleTimeDetectionExtension(&extension));
    return extension != am::IdleTimeDetectionExtension_Disabled;
}

UserInactivityDetectionTimeExtensionMode GetUserInactivityDetectionTimeExtensionMode() NN_NOEXCEPT
{
    uint32_t extension;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->GetIdleTimeDetectionExtension(&extension));
    switch (extension)
    {
        case am::IdleTimeDetectionExtension_Disabled:
            return UserInactivityDetectionTimeExtensionMode_Disabled;
        case am::IdleTimeDetectionExtension_Enabled:
            return UserInactivityDetectionTimeExtensionMode_Enabled;
        case am::IdleTimeDetectionExtension_EnabledUnsafe:
            return UserInactivityDetectionTimeExtensionMode_EnabledUnsafe;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    // NEVER REACHED_HERE
}

//-----------------------------------------------------------------------------
//  ユーザの操作があったことを明示的に宣言
//
void ReportUserIsActive() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->ReportUserIsActive());
}

//-----------------------------------------------------------------------------
//  マルチメディアコンテンツ再生区間の制御
//
void BeginMediaPlaybackSection() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(g_MediaPlaybackSectionMutex);
    if (g_MediaPlaybackSectionCount == 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->SetMediaPlaybackStateForApplication(true));
    }
    ++g_MediaPlaybackSectionCount;
}

void EndMediaPlaybackSection() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(g_MediaPlaybackSectionMutex);
    if (g_MediaPlaybackSectionCount > 0)
    {
        --g_MediaPlaybackSectionCount;
        if (g_MediaPlaybackSectionCount == 0)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->SetMediaPlaybackStateForApplication(false));
        }
    }
}

//-----------------------------------------------------------------------------
//  自動スリープの有効化・無効化
//
void BeginAutoSleepDisabledSection() NN_NOEXCEPT
{
    nn::applet::BeginAutoSleepDisabledSection();
}

void EndAutoSleepDisabledSection() NN_NOEXCEPT
{
    nn::applet::EndAutoSleepDisabledSection();
}

bool IsAutoSleepDisabled() NN_NOEXCEPT
{
    return nn::applet::IsAutoSleepDisabled();
}

//-----------------------------------------------------------------------------

}}  // namespace nn::oe

