﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/apm/apm.h>
#include <nn/os/os_Mutex.h>

#include <nn/oe/oe_CaptureButtonControl.private.h>
#include <nn/applet/applet_Apis.h>
#include <nn/applet/applet_ApplicationSelf.h>
#include <nn/oe/oe_ApplicationSelf.private.h>
#include <nn/am/am_Shim.h>
#include <nn/oe/oe_SelfControlApis.h>
#include <nn/applet/applet_InternalFunctions.h>

namespace nn {
namespace oe {

NN_STATIC_ASSERT( oe::MessageCaptureButtonPressedShortly == applet::Message_CaptureButtonPressedShortly );

//-----------------------------------------------------------------------------

void InitializeApplet() NN_NOEXCEPT;

namespace {

bool    g_SkipNnOeInitialize = false;

struct MutexForInitialization
{
    os::MutexType    m_Mutex;

    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex( &m_Mutex );
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex( &m_Mutex );
    }
};

MutexForInitialization g_AppletMutex = { NN_OS_MUTEX_INITIALIZER(false) };

}

//-----------------------------------------------------------------------------

void Initialize() NN_NOEXCEPT
{
    // Initialize() 全体をロックで排他しておく
    std::lock_guard<MutexForInitialization> lock(g_AppletMutex);
    if (g_SkipNnOeInitialize)
    {
        return;
    }
    g_SkipNnOeInitialize = true;

    nn::apm::Initialize();
    InitializeApplet();
}

//-----------------------------------------------------------------------------

void BeginBlockingHomeButtonShortPressed(TimeSpan longPressTime) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->BeginBlockingHomeButton(longPressTime.GetNanoSeconds()));
}

void BeginBlockingHomeButtonShortPressed() NN_NOEXCEPT
{
    BeginBlockingHomeButtonShortPressed(0);
}

void EndBlockingHomeButtonShortPressed() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->EndBlockingHomeButton());
}

void BeginBlockingHomeButton() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->BeginBlockingHomeButtonShortAndLongPressed(0));
}

void EndBlockingHomeButton() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->EndBlockingHomeButtonShortAndLongPressed());
}

void FinishStartupLogo() NN_NOEXCEPT
{
    bool notified;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->NotifyRunning(&notified));
    if (!notified)
    {
        NN_SDK_LOG("warning: nn::oe::FinishStartupLogo() was called when LogoHandling is not 'Manual' in the .nmeta file");
    }
}

void NotifyRunningForTest() NN_NOEXCEPT
{
    FinishStartupLogo();
}

//-----------------------------------------------------------------------------

void BeginHandlingCaptureButtonShortPressed() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetRequiresCaptureButtonShortPressedMessage(true));
}

void EndHandlingCaptureButtonShortPressed() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetRequiresCaptureButtonShortPressedMessage(false));
}

//-----------------------------------------------------------------------------

void RequestToRelaunchApplication() NN_NOEXCEPT
{
    applet::RequestToRelaunchApplication(applet::InvalidStorageHandle);
}

void SetWirelessPriorityMode(WirelessPriorityMode wirelessPriorityMode) NN_NOEXCEPT
{
    if(wirelessPriorityMode == WirelessPriorityMode_Default)
    {
        applet::SetWirelessPriorityMode(applet::WirelessPriorityMode_OptimizedForBluetooth);
    }
    else if(wirelessPriorityMode == WirelessPriorityMode_OptimizedForWlan)
    {
        applet::SetWirelessPriorityMode(applet::WirelessPriorityMode_OptimizedForWlan);
    }
    else
    {
        NN_ABORT("Undefined WirelessPriorityMode in nn::oe::SetWirelessPriorityMode()\n");
    }
}

} // namespace oe
} // namespace nn
