﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ntc/shim/ntc_Shim.h>
#include <nn/ntc/detail/shim/ntc_IpcSession.h>
#include <nn/ntc/detail/shim/ntc_EnsureNetworkClockAvailabilityClientImpl.h>
#include <nn/time/detail/time_Log.h> // TIME の構造化ログを拝借

#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Mutex.h>
#include <mutex>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/ntc/service/ntc_Service.h>
#endif

namespace nn { namespace ntc { namespace shim {

namespace
{
    void InitializeIfWin() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        // Windows 環境は初回にサーバー起動する
        NN_FUNCTION_LOCAL_STATIC(nn::os::Mutex, s_Mutex, (false));
        NN_FUNCTION_LOCAL_STATIC(bool, s_IsFirst, = true);

        std::lock_guard<nn::os::Mutex> lock(s_Mutex);
        if(s_IsFirst)
        {
            nn::ntc::service::StartServer();
            s_IsFirst = false;
        }
#endif
    }

    std::aligned_storage<sizeof(nn::ntc::shim::CorrectionNetworkClockAsyncTask), 8u>::type g_CorrectionNetworkClockAsyncTaskStorage;
    nn::ntc::shim::CorrectionNetworkClockAsyncTask* GetCorrectionNetworkClockAsyncTaskForBlockApi() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::os::Mutex, s_Mutex, (false));
        NN_FUNCTION_LOCAL_STATIC(nn::ntc::shim::CorrectionNetworkClockAsyncTask*, s_pTask, = nullptr);

        std::lock_guard<nn::os::Mutex> lock(s_Mutex);
        if(!s_pTask)
        {
            s_pTask =
                new (&g_CorrectionNetworkClockAsyncTaskStorage) nn::ntc::shim::CorrectionNetworkClockAsyncTask(
                    nn::os::EventClearMode_AutoClear, EnsureNetworkClockAvailabilityMode_Default);
        }
        return s_pTask;
    }
}

nn::Result EnsureNetworkClockAvailability() NN_NOEXCEPT
{
    InitializeIfWin();

    NN_RESULT_DO(GetCorrectionNetworkClockAsyncTaskForBlockApi()->StartTask()); // 既に StartTask() されていたら内部では何も行われない

    while(GetCorrectionNetworkClockAsyncTaskForBlockApi()->IsProcessing())
    {
        // 複数クライアントが EnsureNetworkClockAvailability() を呼んでもデッドロックしないように時限待ち.
        // 待ち時間に深い意味はない.
        if(GetCorrectionNetworkClockAsyncTaskForBlockApi()->GetFinishNotificationEvent().TimedWait(nn::TimeSpan::FromSeconds(2)))
        {
            break;
        }
    }

    auto result = GetCorrectionNetworkClockAsyncTaskForBlockApi()->GetResult();
    if(result.IsFailure())
    {
        // TIME の構造化ログを拝借
        NN_DETAIL_TIME_WARN("EnsureNetworkClockAvailability() failed. (%08x, %03d-%04d)\n",
            result.GetInnerValueForDebug(),
            result.GetModule(), result.GetDescription());
    }

    return result;
}

void CancelEnsuringNetworkClockAvailability() NN_NOEXCEPT
{
    GetCorrectionNetworkClockAsyncTaskForBlockApi()->Cancel();
}


// 非同期 API
// -----------------------------------------------------------------------------------------

namespace
{
    detail::shim::EnsureNetworkClockAvailabilityClientImpl* GetAsyncRequestClientImpl(CorrectionNetworkClockAsyncTask::StorageType* pStorage) NN_NOEXCEPT
    {
        return (reinterpret_cast<detail::shim::EnsureNetworkClockAvailabilityClientImpl*>(pStorage));
    }
}

#define NN_NTC_SHIM_GET_ASYNC_SERVICE(pCorrectionNetworkClockAsyncTask) \
    (reinterpret_cast<ManagerBuffer*>(&pCorrectionNetworkClockAsyncTask->m_Storage)->service)

CorrectionNetworkClockAsyncTask::CorrectionNetworkClockAsyncTask(
    nn::os::EventClearMode eventClearMode,
    EnsureNetworkClockAvailabilityMode mode) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(StorageType) >= sizeof(detail::shim::EnsureNetworkClockAvailabilityClientImpl));
    NN_STATIC_ASSERT(NN_ALIGNOF(StorageType) % NN_ALIGNOF(detail::shim::EnsureNetworkClockAvailabilityClientImpl) == 0);

    InitializeIfWin();

    new(&m_Storage) detail::shim::EnsureNetworkClockAvailabilityClientImpl(eventClearMode, mode);
}

CorrectionNetworkClockAsyncTask::~CorrectionNetworkClockAsyncTask() NN_NOEXCEPT
{
    GetAsyncRequestClientImpl(&m_Storage)->~EnsureNetworkClockAvailabilityClientImpl();
}

nn::Result CorrectionNetworkClockAsyncTask::StartTask() NN_NOEXCEPT
{
    return GetAsyncRequestClientImpl(&m_Storage)->StartTask();
}

nn::Result CorrectionNetworkClockAsyncTask::GetResult() NN_NOEXCEPT
{
    return GetAsyncRequestClientImpl(&m_Storage)->GetResult();
}

nn::os::SystemEvent& CorrectionNetworkClockAsyncTask::GetFinishNotificationEvent() NN_NOEXCEPT
{
    return GetAsyncRequestClientImpl(&m_Storage)->GetFinishNotificationEvent();
}

void CorrectionNetworkClockAsyncTask::Cancel() NN_NOEXCEPT
{
    GetAsyncRequestClientImpl(&m_Storage)->Cancel();
}

bool CorrectionNetworkClockAsyncTask::IsProcessing() NN_NOEXCEPT
{
    return GetAsyncRequestClientImpl(&m_Storage)->IsProcessing();
}

nn::time::PosixTime CorrectionNetworkClockAsyncTask::GetServerPosixTime() NN_NOEXCEPT
{
    nn::time::PosixTime serverTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAsyncRequestClientImpl(&m_Storage)->GetServerTime(&serverTime));
    return serverTime;
}

void SuspendAutonomicTimeCorrection() NN_NOEXCEPT
{
    InitializeIfWin();

    nn::sf::SharedPointer<detail::service::IStaticService> pSession;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::shim::GetStaticServiceSharedPointer(&pSession));
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSession->SuspendAutonomicTimeCorrection());
}

void ResumeAutonomicTimeCorrection() NN_NOEXCEPT
{
    InitializeIfWin();

    nn::sf::SharedPointer<detail::service::IStaticService> pSession;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::shim::GetStaticServiceSharedPointer(&pSession));
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSession->ResumeAutonomicTimeCorrection());
}

}}} // nn::ntc::service
