﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ntc/service/ntc_Service.h>
#include <nn/ntc/detail/service/ntc_Common.h>

#include <nn/ntc/detail/service/ntc_AutonomicEnsureNetworkClockAvailabilityThread.h>
#include <nn/ntc/detail/service/ntc_HipcServerManager.h>
#include <nn/ntc/detail/service/core/ntc_SerializeTaskThread.h>
#include <nn/ntc/detail/service/ntc_AutonomicEnsureConfig.h>
#include <nn/nn_SystemThreadDefinition.h>

namespace nn { namespace ntc { namespace service {

namespace
{
    std::aligned_storage<sizeof(detail::service::HipcServerManager)>::type g_HipcServerMangerBuffer;
    detail::service::HipcServerManager* g_pHipcServerManager = nullptr;

    NN_OS_ALIGNAS_THREAD_STACK char g_ServerTimeDownloadThreadStack[1024 * 16];
    std::aligned_storage<sizeof(detail::service::core::SerializeTaskThread)>::type g_EnsureNetworkClockAvailabilityThreadBuffer;

    NN_OS_ALIGNAS_THREAD_STACK char g_AutonomicEnsureNetworkClockAvailabilityThreadStack[1024 * 8];
    std::aligned_storage<sizeof(detail::service::AutonomicEnsureNetworkClockAvailabilityThread)>::type g_AutonomicEnsureNetworkClockAvailabilityThreadBuffer;
    detail::service::AutonomicEnsureNetworkClockAvailabilityThread* g_pAutonomicEnsureNetworkClockAvailabilityThread = nullptr;
}
detail::service::core::SerializeTaskThread* g_EnsureNetworkClockAvailabilityThread = nullptr;

void StartServer() NN_NOEXCEPT
{
    NN_DETAIL_NTC_SERVER_LOG("Start server thread.\n");

    NN_SDK_ASSERT(g_EnsureNetworkClockAvailabilityThread == nullptr);
    g_EnsureNetworkClockAvailabilityThread = new (&g_EnsureNetworkClockAvailabilityThreadBuffer) detail::service::core::SerializeTaskThread();
    g_EnsureNetworkClockAvailabilityThread->StartThread(
        g_ServerTimeDownloadThreadStack,
        sizeof(g_ServerTimeDownloadThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(ntc, EnsureNetworkClockAvailabilityTask),
        NN_SYSTEM_THREAD_NAME(ntc, EnsureNetworkClockAvailabilityTask));

    if(detail::service::IsAutonomicCorrectionEnabled())
    {
        NN_SDK_ASSERT(g_pAutonomicEnsureNetworkClockAvailabilityThread == nullptr);
        g_pAutonomicEnsureNetworkClockAvailabilityThread =
            new (&g_AutonomicEnsureNetworkClockAvailabilityThreadBuffer) detail::service::AutonomicEnsureNetworkClockAvailabilityThread(
                detail::service::GetAutonomicCorrectionIntervalTimeSpan(),
                detail::service::GetAutonomicCorrectionRetryTimeSpan(),
                detail::service::GetAutonomicCorrectionImmediateTryCountMax(),
                detail::service::GetAutonomicCorrectionImmediateTryTimeSpan()
            );
        g_pAutonomicEnsureNetworkClockAvailabilityThread->StartThread(
            g_AutonomicEnsureNetworkClockAvailabilityThreadStack,
            sizeof(g_AutonomicEnsureNetworkClockAvailabilityThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(ntc, AutonomicEnsureNetworkClockAvailabilityThread),
            NN_SYSTEM_THREAD_NAME(ntc, AutonomicEnsureNetworkClockAvailabilityThread));

        // Windows 版の SuspendAutonomicTimeCorrection() 経由でサーバー起動された場合に、
        // Suspend 前に即時時刻補正が走るので Windows では EphemeralNetworkSystemClock のための即時補正要求は行わない.
        // Windows では EphemeralNetworkSystemClock はPC時計で必ず初期化されるので問題ない.
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        // nn::time::EphemeralNetworkSystemClock 有効化のため起動時に必ず補正をトリガーする.
        g_pAutonomicEnsureNetworkClockAvailabilityThread->ImmediatelyExecute();
#endif
    }
    else
    {
        NN_DETAIL_NTC_SERVER_LOG("Autonomic clock ensure thread is disabled.\n");
    }

    NN_SDK_ASSERT(g_pHipcServerManager == nullptr);
    g_pHipcServerManager = new (&g_HipcServerMangerBuffer) detail::service::HipcServerManager(g_pAutonomicEnsureNetworkClockAvailabilityThread);
    g_pHipcServerManager->StartThread();
}

void StopServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_pHipcServerManager);
    g_pHipcServerManager->StopThread();
    g_pHipcServerManager = nullptr;

    NN_SDK_ASSERT_NOT_NULL(g_EnsureNetworkClockAvailabilityThread);
    g_EnsureNetworkClockAvailabilityThread->StopThread(nn::ResultSuccess()); // TODO: result
    g_EnsureNetworkClockAvailabilityThread = nullptr;

    if(g_pAutonomicEnsureNetworkClockAvailabilityThread)
    {
        g_pAutonomicEnsureNetworkClockAvailabilityThread->StopThread();
        g_pAutonomicEnsureNetworkClockAvailabilityThread = nullptr;
    }
}

}}} // nn::ntc::service
