﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ntc/detail/shim/ntc_EnsureNetworkClockAvailabilityClientImpl.h>
#include <nn/ntc/detail/shim/ntc_IpcSession.h>

#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nifm/nifm_ApiClientManagement.h>

namespace nn { namespace ntc { namespace detail { namespace shim {

EnsureNetworkClockAvailabilityClientImpl::EnsureNetworkClockAvailabilityClientImpl(
    nn::os::EventClearMode eventClearMode,
    nn::ntc::EnsureNetworkClockAvailabilityMode mode) NN_NOEXCEPT
{
    nn::sf::SharedPointer<detail::service::IStaticService> pSession;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::shim::GetStaticServiceSharedPointer(&pSession));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pSession->GetEnsureNetworkClockAvailabilityService(
            &m_pService,
            mode,
            nn::nifm::GetClientId()
            ));
    NN_SDK_ASSERT_NOT_NULL(m_pService);

    nn::sf::NativeHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pService->GetSystemEventReadableHandle(&handle));
    m_FinishEvent.AttachReadableHandle(
        handle.GetOsHandle(), handle.IsManaged(), eventClearMode);
    handle.Detach();
}

EnsureNetworkClockAvailabilityClientImpl::~EnsureNetworkClockAvailabilityClientImpl() NN_NOEXCEPT
{
    m_pService = nullptr;
}

nn::Result EnsureNetworkClockAvailabilityClientImpl::StartTask() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pService->StartTask());
    NN_RESULT_SUCCESS;
}

nn::Result EnsureNetworkClockAvailabilityClientImpl::GetResult() NN_NOEXCEPT
{
    return m_pService->GetResult();
}

nn::os::SystemEvent& EnsureNetworkClockAvailabilityClientImpl::GetFinishNotificationEvent() NN_NOEXCEPT
{
    return m_FinishEvent;
}

void EnsureNetworkClockAvailabilityClientImpl::Cancel() NN_NOEXCEPT
{
    m_pService->Cancel();
}

bool EnsureNetworkClockAvailabilityClientImpl::IsProcessing() NN_NOEXCEPT
{
    bool isProcessing;
    auto result = m_pService->IsProcessing(&isProcessing);
    return result.IsSuccess() ? isProcessing : false;
}

nn::Result EnsureNetworkClockAvailabilityClientImpl::GetServerTime(nn::time::PosixTime* pOutServerTime) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutServerTime);

    nn::time::PosixTime outServerTime;
    NN_RESULT_DO(m_pService->GetServerTime(&outServerTime));

    *pOutServerTime = outServerTime;

    NN_RESULT_SUCCESS;
}

}}}} // nn::ntc::detail::shim
