﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/ntc/detail/service/ntc_EnsureNetworkClockAvailabilityTaskExecutor.h>

#include <nn/ntc/detail/service/ntc_SystemReport.h>
#include <nn/ntc/detail/service/core/ntc_SerializeTaskThread.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/time/time_StandardNetworkSystemClockPrivilegeApi.h>
#include <nn/time/time_EphemeralNetworkSystemClockPrivilegeApi.h>

#include <mutex>

#if defined(NN_BUILD_CONFIG_OS_WIN)
// windows では PC 時計を使ってネットワーク時計を補正する
#include <chrono>
#else
#include <nn/ntc/detail/service/ntc_ServerTimeDownloader.h>
#endif

namespace nn { namespace ntc { namespace detail { namespace service {

namespace
{
    Result GetServerPosixTime(nn::time::PosixTime* pOutServerTime) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutServerTime);

        nn::time::PosixTime serverTime;

#if defined(NN_BUILD_CONFIG_OS_WIN)
        // windows では PC 時計を使ってネットワーク時計を補正する.
        // windows でのみ socket, libcurl の初期化をアプリに要求すると手間だし、
        // そもそも PC 時計が基本的にはネットワーク時計になってるはずなので.
        auto tp = std::chrono::system_clock::now();
        time_t t = std::chrono::system_clock::to_time_t(tp);
        serverTime.value = static_cast<int64_t>(t);
#else
        NN_RESULT_DO(service::ServerTimeDownloader::DownloadPosixTimeFromServer(&serverTime));
#endif

        *pOutServerTime = serverTime;

        NN_RESULT_SUCCESS;
    }

    Result CorrectClock(const nn::time::PosixTime& currentTime) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::time::SetStandardNetworkSystemClockCurrentTime(currentTime));
        NN_RESULT_DO(nn::time::SetEphemeralNetworkSystemClockCurrentTime(currentTime));
        NN_RESULT_SUCCESS;
    }

    nn::Result ExecuteNetworkTimeCorrection() NN_NOEXCEPT
    {
        nn::time::PosixTime serverTime;

        auto start = nn::os::GetSystemTick();
        NN_RESULT_DO(GetServerPosixTime(&serverTime)); // これにかかった時間をレポートする
        auto end = nn::os::GetSystemTick();

        nn::time::SystemClockContext oldContext; // レポートに必要なコンテキストを保存( CorrectClock()で変わるため )
        NN_RESULT_DO(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&oldContext));

        NN_RESULT_DO(CorrectClock(serverTime));

        SendCorrectionSystemReport(oldContext, serverTime, (end - start).ToTimeSpan());

        NN_RESULT_SUCCESS;
    }
}

nn::Result DefaultEnsureNetworkClockAvailabilityTaskExecutor::Execute() NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    if(!service::ServerTimeDownloader::IsDownloadRequired())
    {
        NN_RESULT_SUCCESS;
    }
#endif

    return ExecuteNetworkTimeCorrection();
}

nn::Result ForcibleDownloadEnsureNetworkClockAvailabilityTaskExecutor::Execute() NN_NOEXCEPT
{
    return ExecuteNetworkTimeCorrection();
}

GetServerTimeTaskExecutor::GetServerTimeTaskExecutor() NN_NOEXCEPT
    : m_ServerTimeLock(false)
{
}

nn::Result GetServerTimeTaskExecutor::Execute() NN_NOEXCEPT
{
    nn::time::PosixTime serverTime;
    NN_RESULT_DO(GetServerPosixTime(&serverTime));

    // 補正しない

    std::lock_guard<nn::os::Mutex> lock(m_ServerTimeLock);
    m_ServerTime = serverTime;

    NN_RESULT_SUCCESS;
}

nn::Result GetServerTimeTaskExecutor::GetServerTime(nn::time::PosixTime* pOut) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ServerTimeLock);

    NN_RESULT_THROW_UNLESS(pOut, nn::time::ResultInvalidPointer());
    NN_RESULT_THROW_UNLESS(static_cast<bool>(m_ServerTime), nn::time::ResultNotFound());
    *pOut = *m_ServerTime;

    NN_RESULT_SUCCESS;
}

}}}} // nn::ntc::detail::service
