﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/ntc/detail/service/ntc_EnsureNetworkClockAvailabilityService.h>
#include <nn/ntc/detail/service/ntc_EnsureNetworkClockAvailabilityTaskExecutor.h>
#include <nn/ntc/detail/service/core/ntc_SerializeTaskThread.h>

namespace nn { namespace ntc { namespace service {
extern detail::service::core::SerializeTaskThread* g_EnsureNetworkClockAvailabilityThread;
}}}

namespace nn { namespace ntc { namespace detail { namespace service {

namespace
{
    // memo
    // m_TaskResource.m_pTaskExecutor に動的確保したインスタンスを入れるには、
    // smart pointer 的なもので本クラスとは別に寿命管理する必要がある.
    // 現時点では大域的にインスタンスを用意して凌ぐ.
    DefaultEnsureNetworkClockAvailabilityTaskExecutor g_DefaultEnsureNetworkClockAvailabilityTaskExecutor;
    ForcibleDownloadEnsureNetworkClockAvailabilityTaskExecutor g_ForcibleDownloadEnsureNetworkClockAvailabilityTaskExecutor;
    GetServerTimeTaskExecutor g_GetServerTimeTaskExecutor;
}

EnsureNetworkClockAvailabilityServiceBase::EnsureNetworkClockAvailabilityServiceBase(
    core::TaskExecutor* pTaskExecutor,
    const nn::nifm::ClientId& nifmClientId) NN_NOEXCEPT:
        m_TaskResource(nn::time::ResultNotStarted(), pTaskExecutor),
        m_TaskResourceNode(&m_TaskResource),
        m_NifmClientId(nifmClientId)
{
}

EnsureNetworkClockAvailabilityServiceBase::~EnsureNetworkClockAvailabilityServiceBase() NN_NOEXCEPT
{
    // クライアント側のサービスオブジェクト破棄と同期して呼ばれる

    // ダウンロードスレッド側管理のシステムイベントを削除
    nn::ntc::service::g_EnsureNetworkClockAvailabilityThread->UnregisterTaskResource(m_TaskResourceNode);
}

nn::Result EnsureNetworkClockAvailabilityServiceBase::GetSystemEventReadableHandle(nn::sf::Out<nn::sf::NativeHandle> outEvent) NN_NOEXCEPT
{
    outEvent.Set( nn::sf::NativeHandle(m_TaskResource.m_SystemEvent.GetReadableHandle(), false) );
    NN_RESULT_SUCCESS;
}

nn::Result EnsureNetworkClockAvailabilityServiceBase::StartTask() NN_NOEXCEPT
{
    NN_DETAIL_NTC_SERVER_LOG("Accept EnsureNetworkClockAvailabilityServiceBase::StartTask.\n");

    NN_RESULT_THROW_UNLESS(nn::nifm::IsAnyInternetRequestAccepted(m_NifmClientId), nn::time::ResultNetworkRequestNotAccepted());

    if(!nn::ntc::service::g_EnsureNetworkClockAvailabilityThread->IsRegisteredTask(m_TaskResourceNode))
    {
        auto isAlreadyRegisterd = nn::ntc::service::g_EnsureNetworkClockAvailabilityThread->RegisterTask(m_TaskResourceNode);
        NN_SDK_ASSERT(isAlreadyRegisterd, "[NTCSRV] same task was pushed already. %p", &m_TaskResourceNode);
        NN_UNUSED(isAlreadyRegisterd);
    }

    // 既に登録済であっても成功扱いで良しとする

    NN_RESULT_SUCCESS;
}

nn::Result EnsureNetworkClockAvailabilityServiceBase::GetResult() NN_NOEXCEPT
{
    return m_TaskResource.m_Result;
}

nn::Result EnsureNetworkClockAvailabilityServiceBase::Cancel() NN_NOEXCEPT
{
    // 既にタスクがなければ成功扱いで良いので、返り値のハンドリングはなし
    nn::ntc::service::g_EnsureNetworkClockAvailabilityThread->DoneTask(nn::time::ResultCanceled(), m_TaskResourceNode);

    NN_RESULT_SUCCESS;
}

nn::Result EnsureNetworkClockAvailabilityServiceBase::IsProcessing(nn::sf::Out<bool> outIsProcessing) NN_NOEXCEPT
{
    outIsProcessing.Set(
        nn::ntc::service::g_EnsureNetworkClockAvailabilityThread->IsRegisteredTask(m_TaskResourceNode));
    NN_RESULT_SUCCESS;
}

nn::Result EnsureNetworkClockAvailabilityServiceBase::GetServerTime(nn::sf::Out<nn::time::PosixTime> outServerTime) NN_NOEXCEPT
{
    NN_UNUSED(outServerTime);
    NN_RESULT_THROW(nn::time::ResultNotImplemented());
}

// ------------------------------------
DefaultEnsureNetworkClockAvailabilityService::DefaultEnsureNetworkClockAvailabilityService(const nn::nifm::ClientId& nifmClientId) NN_NOEXCEPT:
    EnsureNetworkClockAvailabilityServiceBase(&g_DefaultEnsureNetworkClockAvailabilityTaskExecutor, nifmClientId)
{
}

// ------------------------------------
ForcibleDownloadEnsureNetworkClockAvailabilityService::ForcibleDownloadEnsureNetworkClockAvailabilityService(const nn::nifm::ClientId& nifmClientId) NN_NOEXCEPT:
    EnsureNetworkClockAvailabilityServiceBase(&g_ForcibleDownloadEnsureNetworkClockAvailabilityTaskExecutor, nifmClientId)
{
}

// ------------------------------------
ServerTimeGetterService::ServerTimeGetterService(const nn::nifm::ClientId& nifmClientId) NN_NOEXCEPT:
    EnsureNetworkClockAvailabilityServiceBase(&g_GetServerTimeTaskExecutor, nifmClientId)
{
}
nn::Result ServerTimeGetterService::GetServerTime(nn::sf::Out<nn::time::PosixTime> outServerTime) NN_NOEXCEPT
{
    nn::time::PosixTime serverTime;
    NN_RESULT_DO(g_GetServerTimeTaskExecutor.GetServerTime(&serverTime));
    outServerTime.Set(serverTime);
    NN_RESULT_SUCCESS;
}

}}}} // nn::ntc::detail::service
