﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ntc/detail/service/ntc_AutonomicEnsureNetworkClockAvailabilityThread.h>
#include <nn/ntc/detail/service/ntc_EnsureNetworkClockAvailabilityService.h>
#include <nn/ntc/detail/service/ntc_ServerTimeDownloader.h>

#include <nn/nn_Abort.h>

#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>

namespace nn { namespace ntc { namespace detail { namespace service {

AutonomicEnsureNetworkClockAvailabilityThread::AutonomicEnsureNetworkClockAvailabilityThread(
        nn::TimeSpan intervalTimeSpan,
        nn::TimeSpan retryTimeSpan,
        int immediateTryCountMax,
        nn::TimeSpan immediateTryTimeSpan) NN_NOEXCEPT:
    m_StopEvent(nn::os::EventClearMode_ManualClear),
    m_ImmediateExecutionEvent(nn::os::EventClearMode_ManualClear),
    m_ResumeRequestEvent(nn::os::EventClearMode_ManualClear),
    m_SuspendRequestEvent(nn::os::EventClearMode_ManualClear),
    m_SuspendCounterLock(false),
    m_SuspendCounter(0),
    m_SuspendCompletionEvent(nn::os::EventClearMode_ManualClear),
    m_IntervalTimeSpan(intervalTimeSpan),
    m_RetryTimeSpan(retryTimeSpan),
    m_ImmediateTryCountMax(immediateTryCountMax),
    m_ImmediateTryTimeSpan(immediateTryTimeSpan)
{
}

void AutonomicEnsureNetworkClockAvailabilityThread::StartThread(
    char* stack, size_t stackSize, int priority, const char* threadName) NN_NOEXCEPT
{
    m_StopEvent.Clear();
    m_ResumeRequestEvent.Clear();
    m_SuspendRequestEvent.Clear();
    m_SuspendCompletionEvent.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&m_Thread, ThreadFunction, this, stack, stackSize, priority));
    nn::os::SetThreadNamePointer(&m_Thread, threadName);
    nn::os::StartThread(&m_Thread);
}

void AutonomicEnsureNetworkClockAvailabilityThread::StopThread() NN_NOEXCEPT
{
    RequestStopThread();
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);
}

Result AutonomicEnsureNetworkClockAvailabilityThread::Suspend() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_SuspendCounterLock);
    m_SuspendCounter++;

    if (m_SuspendCounter == 1)
    {
        m_SuspendRequestEvent.Signal();

        // IPC スレッドが止まるが用途的に問題なし
        m_SuspendCompletionEvent.Wait();
        m_SuspendCompletionEvent.Clear();
    }

    //NN_DETAIL_NTC_SERVER_LOG("[AutonomicEnsureNetworkClockAvailabilityThread] Suspend() called. m_SuspendCounter:%lld\n", m_SuspendCounter);

    NN_RESULT_SUCCESS;
}

Result AutonomicEnsureNetworkClockAvailabilityThread::Resume() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_SuspendCounterLock);

    NN_RESULT_THROW_UNLESS(m_SuspendCounter > 0, nn::time::ResultNotSuspended());

    m_SuspendCounter--;

    if (m_SuspendCounter == 0)
    {
        m_ResumeRequestEvent.Signal();
    }

    //NN_DETAIL_NTC_SERVER_LOG("[AutonomicEnsureNetworkClockAvailabilityThread] Resume() called. m_SuspendCounter:%lld\n", m_SuspendCounter);

    NN_RESULT_SUCCESS;
}

nn::Result AutonomicEnsureNetworkClockAvailabilityThread::GetElapsedFromLastCorrection(int64_t* pOutSeconds) const NN_NOEXCEPT
{
    nn::time::SystemClockContext context = {};
    NN_RESULT_DO(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&context));

    nn::time::SteadyClockTimePoint current;
    NN_RESULT_DO(nn::time::StandardSteadyClock::GetCurrentTimePoint(&current));

    int64_t result;
    NN_RESULT_DO(nn::time::GetSpanBetween(&result, context.timeStamp, current));

    *pOutSeconds = result;
    NN_RESULT_SUCCESS;
}

nn::TimeSpan AutonomicEnsureNetworkClockAvailabilityThread::GetTimeSpanUntilTaskExecutionRequired() NN_NOEXCEPT
{
    const nn::TimeSpan downloadElapsedThreshold = m_IntervalTimeSpan;

    if(ServerTimeDownloader::IsDownloadRequired())
    {
        return nn::TimeSpan(); // zero
    }

    int64_t elapsedSeconds;
    if(GetElapsedFromLastCorrection(&elapsedSeconds).IsFailure() || elapsedSeconds < 0)
    {
        return nn::TimeSpan(); // zero
    }

    int64_t waitSeconds = downloadElapsedThreshold.GetSeconds() - elapsedSeconds;
    if(waitSeconds < 0)
    {
        return nn::TimeSpan(); // zero
    }

    return nn::TimeSpan::FromSeconds(waitSeconds);
}

bool AutonomicEnsureNetworkClockAvailabilityThread::WaitForNextExecution(const nn::TimeSpan& waitSpanToNext) NN_NOEXCEPT
{
    NN_UTIL_SCOPE_EXIT
    {
        // サーバーへの無駄アクセス回避のため必ずクリアしとく
        m_ImmediateExecutionEvent.Clear();
    };

    while (NN_STATIC_CONDITION(true))
    {
        auto waitResult = nn::os::TimedWaitAny(
            waitSpanToNext,
            m_StopEvent.GetBase(),                  // 0
            m_ImmediateExecutionEvent.GetBase(),    // 1
            m_SuspendRequestEvent.GetBase()         // 2
        );

        switch (waitResult)
        {
        case -1:
            return true; // timeout. 次の補正実行時間が来た
        case 0:
            return false; // スレッド終了要求
        case 1:
            return true; // 即座に補正処理を実行する要求が来た
        case 2:
            WaitForResume(); // resume まで待機
            break; // 再度次回補正まで待機
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

bool AutonomicEnsureNetworkClockAvailabilityThread::WaitUntilNetworkConnectionConfirmed(nn::nifm::NetworkConnection* pNetworkConnection) NN_NOEXCEPT
{
    auto result = nn::nifm::SetRequestRequirementPreset(
        pNetworkConnection->GetRequestHandle(),
        nn::nifm::RequirementPreset_InternetForSystemProcessContinuous); // 永続的かつ共存可能なインターネット接続用プリセット
    NN_SDK_ASSERT(result.IsSuccess(), "[AutonomicEnsureNetworkClockAvailabilityThread] nn::nifm::SetRequestRequirementPreset() failed. (%08x, %03d-%04d)",
        result.GetInnerValueForDebug(),
        result.GetModule(), result.GetDescription());
    NN_UNUSED(result);

    pNetworkConnection->SubmitRequest();

    NN_DETAIL_NTC_SERVER_LOG("[AutonomicEnsureNetworkClockAvailabilityThread] Waiting network connection.\n");
    do
    {
        auto index = nn::os::WaitAny(
            m_StopEvent.GetBase(),                          // 0
            pNetworkConnection->GetSystemEvent().GetBase(), // 1
            m_SuspendRequestEvent.GetBase()                 // 2
            );

        switch (index)
        {
        case 0:
            return false; // スレッド終了要求がきた
        case 1:
            {
                // 多重待ちは EventClearMode_AutoClear でもシグナル解除されないので TryWait してクリア
                auto isSignaled = pNetworkConnection->GetSystemEvent().TryWait();
                NN_SDK_ASSERT(isSignaled);
                NN_UNUSED(isSignaled);
            }
            break;
        case 2:
            WaitForResume(); // resume まで待機
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    } while(!pNetworkConnection->IsAvailable());

    return true;
}

bool AutonomicEnsureNetworkClockAvailabilityThread::WaitForResume() NN_NOEXCEPT
{
    if (!m_SuspendRequestEvent.TryWait())
    {
        NN_SDK_ASSERT(false, "[AutonomicEnsureNetworkClockAvailabilityThread] m_SuspendRequestEvent not signaled");
        return true;
    }
    m_SuspendRequestEvent.Clear();

    // Suspend リクエストが来ていたら Resume リクエストまで待機

    NN_DETAIL_NTC_SERVER_LOG("[AutonomicEnsureNetworkClockAvailabilityThread] Suspend in.\n");
    m_SuspendCompletionEvent.Signal();

    auto index = nn::os::WaitAny(m_StopEvent.GetBase(), m_ResumeRequestEvent.GetBase());
    if(index == 0)
    {
        return false; // スレッド終了要求がきた
    }

    m_ResumeRequestEvent.Clear();
    NN_DETAIL_NTC_SERVER_LOG("[AutonomicEnsureNetworkClockAvailabilityThread] Suspend out.\n");
    return true;
}

void AutonomicEnsureNetworkClockAvailabilityThread::ThreadFunctionImpl() NN_NOEXCEPT
{
    // ここに来るまでに nifm::Initialize() は行われている前提.

#if 1
    nn::TimeSpan waitingTimeSpan = GetTimeSpanUntilTaskExecutionRequired();
#else
    // 起動直後に必ずダウンロードして補正する場合はここを有効に.
    // 補正パラメータが電源断で揮発するような時計が存在しない限り true にする必要はない.
    nn::TimeSpan waitingTimeSpan = nn::TimeSpan(); // zero
#endif

    while(!IsStopThreadRequired())
    {
        // 次の実行まで待つ
        if(!WaitForNextExecution(waitingTimeSpan)) // block API
        {
            break;
        }

        nn::nifm::NetworkConnection connection(nn::os::EventClearMode_AutoClear);
        if(!WaitUntilNetworkConnectionConfirmed(&connection)) // block API
        {
            break;
        }

        nn::Result result =
            ExecuteNetworkClockCorrectionWithImmediateRetry(&connection);// block API

        if(result.IsSuccess())
        {
            waitingTimeSpan = GetTimeSpanUntilTaskExecutionRequired();
        }
        else
        {
            // 失敗時は固定の時間待ってリトライ
            waitingTimeSpan = m_RetryTimeSpan;
        }
    }
}

nn::Result AutonomicEnsureNetworkClockAvailabilityThread::ExecuteNetworkClockCorrectionWithImmediateRetry(
    nn::nifm::NetworkConnection* pNetworkConnection) NN_NOEXCEPT
{
    // 最大 AutonomicCorrectionImmediateTryCountMax 回、成功するまで補正処理を試行する
    // ただし、ネットワーク利用不可であればリトライする意味がないのでそこで終了

    nn::Result result;
    for(int i = 0 ; i < m_ImmediateTryCountMax ; ++i)
    {
        result = ExecuteNetworkClockCorrectionImpl(); // block API
        NN_DETAIL_NTC_SERVER_LOG("[AutonomicEnsureNetworkClockAvailabilityThread] Autonomic network time correction result. (%08x, %03d-%04d)\n",
            result.GetInnerValueForDebug(),
            result.GetModule(), result.GetDescription());

        if(result.IsSuccess())
        {
            break;
        }

        if(!WaitForNextExecution(m_ImmediateTryTimeSpan)) // block API
        {
            break;
        }
        if(!pNetworkConnection->IsAvailable())
        {
            break;
        }
    }
    return result;
}

nn::Result AutonomicEnsureNetworkClockAvailabilityThread::ExecuteNetworkClockCorrectionImpl() NN_NOEXCEPT
{
    NN_DETAIL_NTC_SERVER_LOG("[AutonomicEnsureNetworkClockAvailabilityThread] Execute autonomic network time correction.\n");

    nn::ntc::detail::service::ForcibleDownloadEnsureNetworkClockAvailabilityService ensureService(nn::nifm::GetClientId());

    NN_RESULT_DO(ensureService.StartTask());

    nn::os::SystemEvent finishEvent; // TORIAEZU:sf 通さなくても直接 ensureService のイベントを参照できるはず
    nn::sf::NativeHandle handle;
    NN_RESULT_DO(ensureService.GetSystemEventReadableHandle(&handle));
    finishEvent.AttachReadableHandle(
        handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
    handle.Detach();

    auto index = nn::os::WaitAny(m_StopEvent.GetBase(), finishEvent.GetBase());

    if(index == 0) // スレッド自体の終了要求がきた
    {
        ensureService.Cancel();
    }

    return ensureService.GetResult();
}

}}}} // nn::ntc::detail::service
