﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <mutex>

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataTransaction.h>
#include <nn/fs/fs_SystemSaveData.h>

#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_VulnerabilityManager.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Result.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/nn_Version.h>

#include "ns_Config.h"
#include "ns_SystemUpdateUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace
    {
        const int64_t EntryFileSize = 4 * 1024;

        const char* GetEntryFilePath() NN_NOEXCEPT
        {
            static char entryFilePath[32] = {};
            if (entryFilePath[0] == '\0')
            {
                util::TSNPrintf(entryFilePath, sizeof(entryFilePath), "%s:/entry", SafeSystemVersionMountName);
            }
            return entryFilePath;
        }
    } // namespace

    bool VulnerabilityManager::NeedsUpdate(const ncm::ContentMetaKey& latest) NN_NOEXCEPT
    {
        return m_CachedSystemUpdateMetaKey != latest;
    }

    void VulnerabilityManager::CacheSystemUpdateMeta(const ncm::ContentMetaKey& meta) NN_NOEXCEPT
    {
        m_CachedSystemUpdateMetaKey = meta;
    }

    Result VulnerabilityManager::UpdateSafeSystemVersionInfo(nn::ncm::SystemUpdateId systemUpdateId, uint32_t version) NN_NOEXCEPT
    {
        auto latest = ncm::ContentMetaKey::Make(systemUpdateId, version);

        if (NeedsUpdate(latest))
        {
            NN_RESULT_DO(WriteSystemUpdateMetaKey(latest));
            NN_RESULT_DO(UpdateNeedsUpdateVulnerabilityFlag(latest));
            CacheSystemUpdateMeta(latest);
        }

        NN_RESULT_SUCCESS;
    }

    bool VulnerabilityManager::NeedsUpdateVulnerability() NN_NOEXCEPT
    {
        switch (m_NeedsUpdatePolicy)
        {
        case NeedsUpdatePolicy::Default:
            return m_NeedsUpdateVulnerability;

        case NeedsUpdatePolicy::ForceTrue:
            return true;

        case NeedsUpdatePolicy::ForceFalse:
            return false;

        default:
            NN_ABORT("Invalid NeedsUpdatePolicy set: %d\n", m_NeedsUpdatePolicy);
        }
    }

    Result VulnerabilityManager::UpdateNeedsUpdateVulnerabilityFlag(const ncm::ContentMetaKey& entry) NN_NOEXCEPT
    {
        bool needsUpdateVulnerability;
        NN_RESULT_DO(nn::ns::srv::NeedsUpdateVulnerability(&needsUpdateVulnerability, entry));
        m_NeedsUpdateVulnerability = needsUpdateVulnerability;

        NN_RESULT_SUCCESS;
    }

    Result VulnerabilityManager::InitializeNeedsUpdateVulnerabilityFlag() NN_NOEXCEPT
    {
        ncm::ContentMetaKey systemUpdateMetaKey = {};
        NN_RESULT_TRY(ReadSystemUpdateMetaKey(&systemUpdateMetaKey))
            NN_RESULT_CATCH_ALL
            {
                NN_DETAIL_NS_TRACE("[VulnerabilityManager] ReadSystemUpdateMetaKey failed: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
                auto key = GetSystemUpdateMetaKey();
                if (key)
                {
                    systemUpdateMetaKey = *key;
                }
            }
        NN_RESULT_END_TRY;

        NN_RESULT_DO(UpdateNeedsUpdateVulnerabilityFlag(systemUpdateMetaKey));
        CacheSystemUpdateMeta(systemUpdateMetaKey);

        NN_RESULT_SUCCESS;
    }

    Result VulnerabilityManager::MountSaveData() NN_NOEXCEPT
    {
        if (m_StorePolicy != StorePolicy::OnStorage)
        {
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_TRY(nn::fs::MountSystemSaveData(SafeSystemVersionMountName, SafeSystemVersionSaveDataId))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(fs::CreateSystemSaveData(SafeSystemVersionSaveDataId, SafeSystemVersionSaveDataSize, SafeSystemVersionSaveDataJournalSize, SafeSystemVersionSaveDataFlags));
                NN_RESULT_DO(fs::MountSystemSaveData(SafeSystemVersionMountName, SafeSystemVersionSaveDataId));
            }
        NN_RESULT_END_TRY;
        NN_RESULT_SUCCESS;
    }

    void VulnerabilityManager::InitializeImpl(StorePolicy storePolicy) NN_NOEXCEPT
    {
        m_StorePolicy = storePolicy;

        const char* Name = "vulnerability";

        {
            auto readSize = nn::settings::fwdbg::GetSettingsItemValue(&m_NeedsUpdatePolicy, sizeof(m_NeedsUpdatePolicy), Name, "needs_update_vulnerability_policy");
            if (readSize != sizeof(m_NeedsUpdatePolicy))
            {
                m_NeedsUpdatePolicy = NeedsUpdatePolicy::Default;
            }
            NN_ABORT_UNLESS(static_cast<int>(m_NeedsUpdatePolicy) >= 0 && m_NeedsUpdatePolicy < NeedsUpdatePolicy::PolicyCount, "Invalid needs update policy: %d", m_NeedsUpdatePolicy);
        }

        {
            auto readSize = nn::settings::fwdbg::GetSettingsItemValue(&m_EnableDebug, sizeof(m_EnableDebug), Name, "enable_debug");
            if (readSize != sizeof(m_EnableDebug))
            {
                m_EnableDebug = false;
            }
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(MountSaveData());
        NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeNeedsUpdateVulnerabilityFlag());
    }

    void VulnerabilityManager::Finalize() NN_NOEXCEPT
    {
        fs::Unmount(SafeSystemVersionMountName);
    }

    Result VulnerabilityManager::WriteSystemUpdateMetaKey(const ncm::ContentMetaKey& systemUpdateMetaKey) NN_NOEXCEPT
    {
        if (m_StorePolicy != StorePolicy::OnStorage)
        {
            NN_RESULT_SUCCESS;
        }

        std::lock_guard<os::Mutex> lock(m_SystemUpdateMetaKeyLock);

        auto entryFilePath = GetEntryFilePath();
        {
            fs::FileHandle file;
            NN_RESULT_TRY(fs::OpenFile(&file, entryFilePath, fs::OpenMode_Write))
                NN_RESULT_CATCH(fs::ResultPathNotFound)
                {
                    NN_RESULT_DO(fs::CreateFile(entryFilePath, EntryFileSize));
                    NN_RESULT_DO(fs::OpenFile(&file, entryFilePath, fs::OpenMode_Write));
                }
            NN_RESULT_END_TRY;
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            NN_RESULT_DO(fs::WriteFile(file, 0, &systemUpdateMetaKey, sizeof(systemUpdateMetaKey), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        }

        NN_RESULT_DO(fs::CommitSaveData(SafeSystemVersionMountName));

        NN_DETAIL_NS_TRACE("[VulnerabilityManager] WriteSystemUpdateMetaKey done.\n");

        NN_RESULT_SUCCESS;
    }

    Result VulnerabilityManager::ReadSystemUpdateMetaKey(ncm::ContentMetaKey* out) NN_NOEXCEPT
    {
        if (m_StorePolicy != StorePolicy::OnStorage)
        {
            *out = {};
            NN_RESULT_SUCCESS;
        }

        std::lock_guard<os::Mutex> lock(m_SystemUpdateMetaKeyLock);

        auto entryFilePath = GetEntryFilePath();

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, entryFilePath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        NN_RESULT_DO(fs::ReadFile(file, 0, out, sizeof(*out)));

        NN_RESULT_SUCCESS;
    }

}}}
