﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/srv/ns_UserResourceManagerImpl.h>

#include <nn/ns/detail/ns_IApplicationManagerInterface.sfdl.h>
#include <nn/ns/srv/ns_UserResourceManager.h>
#include <nn/ns/ns_Result.h>

#include <mutex>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ns { namespace srv {
namespace {

class ProgressMonitorForDeleteUserSaveDataAllImpl
{
private:
    static void ThreadFunction(void* data) NN_NOEXCEPT;

    ProgressMonitorForDeleteUserSaveDataAll m_ProgressMonitor;
    const account::Uid m_Uid;

    os::MutexType& m_Lock;
    os::ThreadType m_Thread;
    void* m_StackBuffer;
    size_t m_StackBufferSize;

    void Execute() NN_NOEXCEPT;

public:
    ProgressMonitorForDeleteUserSaveDataAllImpl(const account::Uid& uid, os::MutexType& lock) NN_NOEXCEPT;
    ~ProgressMonitorForDeleteUserSaveDataAllImpl() NN_NOEXCEPT;
    Result ExecuteAsync(int priority, void* stackBuffer, size_t stackBufferSize) NN_NOEXCEPT;

    // インターフェース
    Result GetSystemEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT;
    Result IsFinished(sf::Out<bool> pOut) const NN_NOEXCEPT;
    Result GetResult() const NN_NOEXCEPT;
    Result GetProgress(sf::Out<ns::detail::ProgressForDeleteUserSaveDataAll> pOut) const NN_NOEXCEPT;
};

os::MutexType g_ThreadStackLock = NN_OS_MUTEX_INITIALIZER(false);
NN_ALIGNAS(4096) char g_ThreadStack[4096 * 4];

const int Priority = NN_SYSTEM_THREAD_PRIORITY(ns, UserResourceManagerExecutor);

} // ~namespace nn::ns::srv::<anonymous>


// ---------------------------------------------------------------------------------------------
// UserResourceManagerImpl

Result UserResourceManagerImpl::CalculateUserSaveDataStatistics(sf::Out<ns::UserSaveDataStatistics> pOut, const account::Uid& uid) NN_NOEXCEPT
{
    return UserResourceManager::CalculateUserSaveDataStatistics(pOut.GetPointer(), uid);
}

Result UserResourceManagerImpl::DeleteUserSaveDataAll(sf::Out<sf::SharedPointer<detail::IProgressMonitorForDeleteUserSaveDataAll>> pOut, const account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(os::TryLockMutex(&g_ThreadStackLock), ResultAlreadyOccupied());

    auto p = FactoryType::CreateSharedEmplaced<detail::IProgressMonitorForDeleteUserSaveDataAll, ProgressMonitorForDeleteUserSaveDataAllImpl>(
        &m_ObjectAllocator, uid, g_ThreadStackLock);
    NN_RESULT_THROW_UNLESS(p, ResultSessionObjectAllocationFailure());

    NN_RESULT_DO(p.GetImpl().ExecuteAsync(Priority, g_ThreadStack, sizeof(g_ThreadStack)));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result UserResourceManagerImpl::DeleteUserSystemSaveData(const account::Uid& uid, fs::SystemSaveDataId sysSaveId) NN_NOEXCEPT
{
    return UserResourceManager::DeleteUserSystemSaveData(uid, sysSaveId);
}

Result UserResourceManagerImpl::UnregisterNetworkServiceAccount(const account::Uid& uid) NN_NOEXCEPT
{
    return UserResourceManager::UnregisterNetworkServiceAccount(uid);
}

Result UserResourceManagerImpl::UnregisterNetworkServiceAccountWithUserSaveDataDeletion(const account::Uid& uid) NN_NOEXCEPT
{
    return UserResourceManager::UnregisterNetworkServiceAccountWithUserSaveDataDeletion(uid);
}

// ---------------------------------------------------------------------------------------------
// ProgressMonitorForDeleteUserSaveDataAllImpl

namespace {

void ProgressMonitorForDeleteUserSaveDataAllImpl::ThreadFunction(void* data) NN_NOEXCEPT
{
    reinterpret_cast<ProgressMonitorForDeleteUserSaveDataAllImpl*>(data)->Execute();
}

ProgressMonitorForDeleteUserSaveDataAllImpl::ProgressMonitorForDeleteUserSaveDataAllImpl(const account::Uid& uid, os::MutexType& lock) NN_NOEXCEPT
    : m_Uid(uid)
    , m_Lock(lock)
    , m_StackBuffer(nullptr)
    , m_StackBufferSize(0ull)
{
    NN_SDK_ASSERT(os::IsMutexLockedByCurrentThread(&m_Lock));
}
ProgressMonitorForDeleteUserSaveDataAllImpl::~ProgressMonitorForDeleteUserSaveDataAllImpl() NN_NOEXCEPT
{
    if (m_StackBuffer != nullptr)
    {
        if (!m_ProgressMonitor.IsFinished())
        {
            m_ProgressMonitor.Cancel();
        }
        os::WaitThread(&m_Thread);
        os::DestroyThread(&m_Thread);
    }
    os::UnlockMutex(&m_Lock);
}
void ProgressMonitorForDeleteUserSaveDataAllImpl::Execute() NN_NOEXCEPT
{
    m_ProgressMonitor.MarkAsFinished(
        UserResourceManager::DeleteUserSaveDataAll(m_ProgressMonitor, m_Uid));
}
Result ProgressMonitorForDeleteUserSaveDataAllImpl::ExecuteAsync(int priority, void* stackBuffer, size_t stackBufferSize) NN_NOEXCEPT
{
    NN_RESULT_DO(os::CreateThread(
        &m_Thread, ThreadFunction, this, stackBuffer, stackBufferSize, priority));
    os::SetThreadNamePointer(&m_Thread, "ns::DeleteUserSaveDataAll");

    // 進捗を取得可能にする
    m_ProgressMonitor.Initialize();

    // リソースの保持
    m_StackBuffer = stackBuffer;
    m_StackBufferSize = stackBufferSize;

    // 処理の開始
    os::StartThread(&m_Thread);
    NN_RESULT_SUCCESS;
}
Result ProgressMonitorForDeleteUserSaveDataAllImpl::GetSystemEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    auto& e = m_ProgressMonitor.GetEventRef();
    *pOut = sf::NativeHandle(e.GetReadableHandle(), false);
    NN_RESULT_SUCCESS;
}
Result ProgressMonitorForDeleteUserSaveDataAllImpl::IsFinished(sf::Out<bool> pOut) const NN_NOEXCEPT
{
    *pOut = m_ProgressMonitor.IsFinished();
    NN_RESULT_SUCCESS;
}
Result ProgressMonitorForDeleteUserSaveDataAllImpl::GetResult() const NN_NOEXCEPT
{
    return m_ProgressMonitor.GetResult();
}
Result ProgressMonitorForDeleteUserSaveDataAllImpl::GetProgress(sf::Out<ns::detail::ProgressForDeleteUserSaveDataAll> pOut) const NN_NOEXCEPT
{
    *pOut = m_ProgressMonitor.GetProgress();
    NN_RESULT_SUCCESS;
}

} // ~namespace nn::ns::srv::<anonymous>

}}} // ~namespace nn::ns::srv
