﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_FatPrivate.h>
#include <nn/fs/fs_SystemDataUpdateEventPrivate.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_SystemUpdateApplyManager.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/updater/updater.h>
#include "ns_SystemUpdateUtil.h"
#include "ns_TaskUtil.h"

namespace nn { namespace ns { namespace srv {

namespace
{
    NN_ALIGNAS(4096) char BootImageUpdateBuffer[64 << 10];

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    updater::BootImageUpdateType GetBootImageUpdateType() NN_NOEXCEPT
    {
        int bootImageUpdateType;
        auto size = settings::fwdbg::GetSettingsItemValue(&bootImageUpdateType, sizeof(bootImageUpdateType), "systeminitializer", "boot_image_update_type");
        // 設定項目の取得に失敗した場合は旧来の挙動にする
        if (size != sizeof(bootImageUpdateType))
        {
            return updater::BootImageUpdateType::Original;
        }
        return updater::GetBootImageUpdateType(bootImageUpdateType);
    }
#endif

    Result MarkPreCommitForBootImages() NN_NOEXCEPT
    {
        // BootImages の更新は実機でのみとする
        // updater ライブラリ自体は、 BootImagePackage が無ければ win でも動作するが、
        // 各アプリがリンク対象を追加する必要もでてくるため嬉しくない
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        NN_RESULT_DO(updater::MarkVerifyingRequired(updater::TargetBootMode::Normal, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
        NN_RESULT_DO(updater::MarkVerifyingRequired(updater::TargetBootMode::Safe, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
#endif
        NN_RESULT_SUCCESS;
    }

    Result UpdateBootImages() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        auto f = [](updater::TargetBootMode mode, updater::BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT -> Result
        {
            ncm::SystemDataId bootImagePackageId;
            NN_RESULT_TRY(updater::GetBootImagePackageId(&bootImagePackageId, mode, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)))
                NN_RESULT_CATCH(updater::ResultBootImagePackageNotFound)
                {
                    // BootImagePackage が見つからないのは正常とする
                    // MarkVerified を行うため、見つからない場合もそのまま UpdateBootImages する
                    NN_DETAIL_NS_TRACE("BootImagePackage not installed\n");
                }
            NN_RESULT_END_TRY

            NN_RESULT_TRY(updater::UpdateBootImagesFromPackage(bootImagePackageId, mode, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer), bootImageUpdateType))
                NN_RESULT_CATCH(updater::ResultBootImagePackageNotFound) {}
            NN_RESULT_END_TRY

            NN_RESULT_DO(updater::MarkVerified(mode, BootImageUpdateBuffer, sizeof(BootImageUpdateBuffer)));
            NN_RESULT_SUCCESS;
        };
        auto bootImageUpdateType = GetBootImageUpdateType();
        // 先に safe の更新を行う
        NN_RESULT_DO(f(updater::TargetBootMode::Safe, bootImageUpdateType));
        NN_RESULT_DO(f(updater::TargetBootMode::Normal, bootImageUpdateType));
#endif

        NN_RESULT_SUCCESS;
    }
}

    void SystemUpdateApplyManager::Initialize(SystemReportManager* reportManager, ExFatDriverManager* exFatDriverManager) NN_NOEXCEPT
    {
        m_SystemReportManager = reportManager;
        m_ExFatDriverManager = exFatDriverManager;
    }

    Result SystemUpdateApplyManager::ApplyTask(nim::SystemUpdateTaskId id, bool updateBootImages, bool isRebootless) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_ApplyMutex);

        auto result = ReportThroughput(id);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[SystemUpdateApplyManager] Report throughput failed %08x\n", result.GetInnerValueForDebug());
        }

        result = ReportSystemUpdate(id, isRebootless);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[SystemUpdateApplyManager] Report system update failed %08x\n", result.GetInnerValueForDebug());
        }

        if (updateBootImages)
        {
            NN_RESULT_DO(MarkPreCommitForBootImages());
        }

        NN_RESULT_DO(nim::CommitSystemUpdateTask(id));
        NN_RESULT_DO(nim::DestroySystemUpdateTask(id));

        if (updateBootImages)
        {
            NN_RESULT_DO(UpdateBootImages());
        }

        if (isRebootless)
        {
            // システムデータが更新された(可能性がある)ことを通知する
            NN_RESULT_DO(fs::NotifySystemDataUpdateEvent());
        }

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateApplyManager::ApplyReceivedTask(nim::LocalCommunicationReceiveSystemUpdateTaskId id) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_ApplyMutex);

        auto result = ReportReceivedSystemUpdate(id);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[SystemUpdateApplyManager] Report system update failed %08x\n", result.GetInnerValueForDebug());
        }

        NN_RESULT_DO(MarkPreCommitForBootImages());
        NN_RESULT_DO(nim::CommitLocalCommunicationReceiveSystemUpdateTask(id));
        NN_RESULT_DO(nim::DestroyLocalCommunicationReceiveSystemUpdateTask(id));

        NN_RESULT_DO(UpdateBootImages());

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateApplyManager::ApplyPackageTask(ncm::PackageSystemUpdateTask* task) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_ApplyMutex);

        auto result = ReportPackageSystemUpdate(task);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[SystemUpdateApplyManager] Report system update failed %08x\n", result.GetInnerValueForDebug());
        }

        NN_RESULT_DO(MarkPreCommitForBootImages());
        NN_RESULT_DO(task->Commit());

        NN_RESULT_DO(UpdateBootImages());

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateApplyManager::NotifyDownloadTaskCompleted(nim::SystemUpdateTaskId id) NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[SystemUpdateApplyManager] System update download task is completed.\n");

        nim::SystemUpdateTaskInfo info;
        NN_RESULT_DO(nim::GetSystemUpdateTaskInfo(&info, id));
        NN_SDK_ASSERT(info.applyInfo != ncm::SystemUpdateTaskApplyInfo::CannotJudgeYet);

        if (info.applyInfo == ncm::SystemUpdateTaskApplyInfo::RequireNoReboot)
        {
            NN_DETAIL_NS_TRACE("[SystemUpdateApplyManager] System update is applicable without reboot.\n");
            NN_RESULT_TRY(ApplyTask(id, false, true))
                NN_RESULT_CATCH_ALL
                {
                    NN_DETAIL_NS_TRACE("[SystemUpdateApplyManager] Applying rebootless system update failed 0x%08llx, Destroy task.\n");
                    NN_RESULT_DO(nim::DestroySystemUpdateTask(id));
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
            NN_DETAIL_NS_TRACE("[SystemUpdateApplyManager] System update is applied automatically.\n");
        }

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateApplyManager::ReportThroughput(nim::SystemUpdateTaskId id) NN_NOEXCEPT
    {
        nim::SystemUpdateTaskInfo info;
        NN_RESULT_DO(nim::GetSystemUpdateTaskInfo(&info, id));
        auto throughput = CalculateThroughput(info.downloaded, info.elapsedTime);
        m_SystemReportManager->ReportSystemUpdatePerformance(info.key, info.progress.totalSize, throughput);

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateApplyManager::ReportSystemUpdate(nim::SystemUpdateTaskId id, bool isRebootless) NN_NOEXCEPT
    {
        // 更新後の SystemUpdateMetaId と SystemUpdateMetaVersion を取得する
        uint64_t destinationSystemUpdateMetaId = 0;
        uint32_t destinationSystemUpdateMetaVersion = 0;
        {
            auto count = nim::ListSystemUpdateTask(&id, 1);
            if (count > 0)
            {
                nim::SystemUpdateTaskInfo info;
                NN_RESULT_DO(nn::nim::GetSystemUpdateTaskInfo(&info, id));
                destinationSystemUpdateMetaId = info.key.id;
                destinationSystemUpdateMetaVersion = info.key.version;
            }
        }

        NN_RESULT_DO(ReportSystemUpdateCommon(SystemUpdateType::Nup, destinationSystemUpdateMetaId, destinationSystemUpdateMetaVersion, isRebootless));

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateApplyManager::ReportReceivedSystemUpdate(nim::LocalCommunicationReceiveSystemUpdateTaskId id) NN_NOEXCEPT
    {
        // 更新後の SystemUpdateMetaId と SystemUpdateMetaVersion を取得する
        uint64_t destinationSystemUpdateMetaId = 0;
        uint32_t destinationSystemUpdateMetaVersion = 0;
        {
            auto count = nim::ListLocalCommunicationReceiveSystemUpdateTask(&id, 1);
            if (count > 0)
            {
                nim::LocalCommunicationReceiveSystemUpdateTaskInfo info;
                NN_RESULT_DO(nn::nim::GetLocalCommunicationReceiveSystemUpdateTaskInfo(&info, id));
                destinationSystemUpdateMetaId = info.key.id;
                destinationSystemUpdateMetaVersion = info.key.version;
            }
        }

        NN_RESULT_DO(ReportSystemUpdateCommon(SystemUpdateType::Lup, destinationSystemUpdateMetaId, destinationSystemUpdateMetaVersion, false));

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateApplyManager::ReportPackageSystemUpdate(ncm::PackageSystemUpdateTask* task)  NN_NOEXCEPT
    {
        // 更新後の SystemUpdateMetaId と SystemUpdateMetaVersion を取得する
        uint64_t destinationSystemUpdateMetaId = 0;
        uint32_t destinationSystemUpdateMetaVersion = 0;
        {
            auto systemUpdateMetaKey = task->GetSystemUpdateMetaKey();
            if (systemUpdateMetaKey)
            {
                destinationSystemUpdateMetaId = systemUpdateMetaKey->id;
                destinationSystemUpdateMetaVersion = systemUpdateMetaKey->version;
            }
        }

        NN_RESULT_DO(ReportSystemUpdateCommon(SystemUpdateType::Cup, destinationSystemUpdateMetaId, destinationSystemUpdateMetaVersion, false));

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateApplyManager::ReportSystemUpdateCommon(SystemUpdateType type, uint64_t destinationSystemUpdateMetaId, uint32_t destinationSystemUpdateMetaVersion, bool isRebootless) NN_NOEXCEPT
    {
        // 現在の SystemUpdateMetaId と SystemUpdateMetaVersion を取得する
        uint64_t sourceSystemUpdateMetaId = 0;
        uint32_t sourceSystemUpdateMetaVersion = 0;
        {
            auto systemUpdateMetaKey = GetSystemUpdateMetaKey();
            if (systemUpdateMetaKey)
            {
                sourceSystemUpdateMetaId = systemUpdateMetaKey->id;
                sourceSystemUpdateMetaVersion = systemUpdateMetaKey->version;
            }
        }

        // 現在 ExFat をサポートしているかを取得する
        bool isExFatSupported = fs::IsExFatSupported();

        // 更新後 ExFat をサポートするかを取得する
        bool isExFatSupportedAfterUpdate = m_ExFatDriverManager->IsDriverDownloadedAtLeastOnce();

        m_SystemReportManager->ReportSystemUpdate(type, sourceSystemUpdateMetaId, sourceSystemUpdateMetaVersion, destinationSystemUpdateMetaId, destinationSystemUpdateMetaVersion, isExFatSupported, isExFatSupportedAfterUpdate, isRebootless);

        NN_RESULT_SUCCESS;
    }
}}}
