﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ns/srv/ns_SystemReportManager.h>
#include <nn/ns/detail/ns_Log.h>

// 実際のレポート処理は HORIZON でのみ行う。
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_StorageUtil.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/result/result_HandlingUtility.h>

// Result を返す expression を受けて、失敗の場合には引数なしで return する。
#define NN_NS_SREPO_DO(expression) \
    do \
    { \
        auto _nn_ns_srepo_do_result = (expression); \
        if ( _nn_ns_srepo_do_result.IsFailure() ) \
        { \
            NN_DETAIL_NS_TRACE("[SystemReportManager] Failed: %s\n  Module: %d\n  Description: %d\n  InnerValue: 0x%08x\n", \
                NN_MACRO_STRINGIZE(expression), _nn_ns_srepo_do_result.GetModule(), _nn_ns_srepo_do_result.GetDescription(), _nn_ns_srepo_do_result.GetInnerValueForDebug()); \
            return; \
        } \
    } while( NN_STATIC_CONDITION(false) )

#endif

namespace nn { namespace ns { namespace srv {

    namespace
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        const nn::ApplicationId PrepoApplicationIdForReportingStorageUsage = {0x0100000000001016};
        const nn::ApplicationId PrepoApplicationIdForReportingDownloadThroughput = {0x0100000000001018};
        const nn::ApplicationId PrepoApplicationIdForReportingSystemUpdate = { 0x0100000000001021 };

        Result GetSdCardFreeSpaceSize(int64_t* pOutValue) NN_NOEXCEPT
        {
            ncm::ContentStorage storage;
            NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::SdCard));
            NN_RESULT_DO(storage.GetFreeSpaceSize(pOutValue));
            NN_RESULT_SUCCESS;
        }

        Result GetSdCardTotalSpaceSize(int64_t* pOutValue) NN_NOEXCEPT
        {
            ncm::ContentStorage storage;
            NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::SdCard));
            NN_RESULT_DO(storage.GetTotalSpaceSize(pOutValue));
            NN_RESULT_SUCCESS;
        }
#endif
    }

    void SystemReportManager::Initialize() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        size_t size = settings::fwdbg::GetSettingsItemValue(&m_IsEnabled, sizeof(m_IsEnabled), "systemreport", "enabled");
        NN_ABORT_UNLESS_EQUAL(size, sizeof(m_IsEnabled));
#endif

        NN_DETAIL_NS_TRACE("[SystemReportManager] SystemReport is %s.\n", m_IsEnabled ? "enabled" : "disabled");
    }

    void SystemReportManager::InitializeForMaintenanceMode() NN_NOEXCEPT
    {
        m_IsEnabled = false;

        NN_DETAIL_NS_TRACE("[SystemReportManager] SystemReport is disabled, since boot mode is Maintenance.\n");
    }

    void SystemReportManager::ReportApplicationInstall(ApplicationInstallReportType type, ncm::ApplicationId id) const NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[SystemReportManager] ReportApplicationInstall(%u, 0x%016llx) : %s\n", static_cast<Bit8>(type), id.value, m_IsEnabled ? "enabled" : "disabled");
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        if( m_IsEnabled )
        {
            prepo::SystemPlayReport report("application_install");
            NN_NS_SREPO_DO(report.SetApplicationId(PrepoApplicationIdForReportingStorageUsage));
            char buffer[64];
            report.SetBuffer(buffer, sizeof(buffer));

            NN_NS_SREPO_DO(report.Add("Type", static_cast<int64_t>(type)));
            NN_NS_SREPO_DO(report.Add("ApplicationId", prepo::Any64BitId{ id.value }));
            NN_NS_SREPO_DO(report.Save());
        }
#else
        NN_UNUSED(type);
        NN_UNUSED(id);
        NN_DETAIL_NS_TRACE("[SystemReportManager] Unsupported OS.\n");
#endif

    }

    void SystemReportManager::ReportSdCardInfo() const NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[SystemReportManager] ReportSdCardInfo() : %s.\n", (m_IsEnabled && m_IsSdCardReportEnabled) ? "enabled" : "disabled");
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        if( m_IsEnabled && m_IsSdCardReportEnabled )
        {
            uint8_t sdCid[fs::SdCardCidSize];
            NN_NS_SREPO_DO(fs::GetSdCardCid(sdCid, sizeof(sdCid)));

            char sdCidStr[fs::SdCardCidSize * 2 + 1] = {};
            for( size_t i = 0; i < fs::SdCardCidSize; i++ )
            {
                util::SNPrintf(sdCidStr + i * 2, 3, "%02x", sdCid[i]);
            }

            prepo::SystemPlayReport report("sdcard_info");
            NN_NS_SREPO_DO(report.SetApplicationId(PrepoApplicationIdForReportingStorageUsage));
            char buffer[128];
            report.SetBuffer(buffer, sizeof(buffer));

            NN_NS_SREPO_DO(report.Add("Cid", sdCidStr));
            NN_NS_SREPO_DO(report.Add("Mid", static_cast<int64_t>(fs::GetMidFromSdCardCid(sdCid))));
            char oid[fs::SdCardCidOidSize + 1] = {}; // + 1 for NULL.
            fs::GetOidFromSdCardCid(oid, sizeof(oid), sdCid);
            if( std::strlen(oid) > 0 )
            {
                NN_NS_SREPO_DO(report.Add("OemId", oid));
            }
            char pnm[fs::SdCardCidPnmSize + 1] = {}; // + 1 for NULL.
            fs::GetPnmFromSdCardCid(pnm, sizeof(pnm), sdCid);
            if( std::strlen(pnm) > 0 )
            {
                NN_NS_SREPO_DO(report.Add("Pnm", pnm));
            }
            NN_NS_SREPO_DO(report.Add("Prv", static_cast<int64_t>(fs::GetPrvFromSdCardCid(sdCid))));
            NN_NS_SREPO_DO(report.Add("MdtY", static_cast<int64_t>(fs::GetMdtYearCodeFromSdCardCid(sdCid))));
            NN_NS_SREPO_DO(report.Add("MdtM", static_cast<int64_t>(fs::GetMdtMonthCodeFromSdCardCid(sdCid))));
            int64_t totalSpaceSize = 0;
            NN_NS_SREPO_DO(GetSdCardTotalSpaceSize(&totalSpaceSize));
            NN_NS_SREPO_DO(report.Add("TotalSpace", totalSpaceSize));
            NN_NS_SREPO_DO(report.Save());
        }
#else
        NN_DETAIL_NS_TRACE("[SystemReportManager] Unsupported OS.\n");
#endif
    }

    void SystemReportManager::ReportSdCardFreeSpace() const NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[SystemReportManager] ReportSdCardFreeSpace() : %s.\n", (m_IsEnabled && m_IsSdCardReportEnabled) ? "enabled" : "disabled");
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        if( m_IsEnabled && m_IsSdCardReportEnabled )
        {
            prepo::SystemPlayReport report("sdcard_freespace");
            NN_NS_SREPO_DO(report.SetApplicationId(PrepoApplicationIdForReportingStorageUsage));
            char buffer[64];
            report.SetBuffer(buffer, sizeof(buffer));

            int64_t freeSpaceSize;
            NN_NS_SREPO_DO(GetSdCardFreeSpaceSize(&freeSpaceSize));
            NN_NS_SREPO_DO(report.Add("FreeSpace", freeSpaceSize));
            NN_NS_SREPO_DO(report.Save());
        }
#else
        NN_DETAIL_NS_TRACE("[SystemReportManager] Unsupported OS.\n");
#endif
    }

    void SystemReportManager::ReportSystemUpdatePerformance(const ncm::ContentMetaKey& key, int64_t downloadSize, int64_t throughput) const NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        if (m_IsEnabled)
        {
            prepo::SystemPlayReport report("systemupdate_dl_throughput");
            NN_NS_SREPO_DO(report.SetApplicationId(PrepoApplicationIdForReportingDownloadThroughput));
            char buffer[256];
            report.SetBuffer(buffer, sizeof(buffer));

            NN_NS_SREPO_DO(report.Add("ContentMetaId", prepo::Any64BitId{ key.id }));
            NN_NS_SREPO_DO(report.Add("Version", prepo::Any64BitId{ key.version }));
            NN_NS_SREPO_DO(report.Add("DownloadSize", downloadSize));
            NN_NS_SREPO_DO(report.Add("ThroughputKBps", throughput));

            NN_NS_SREPO_DO(report.Save());
        }
#else
        NN_UNUSED(key);
        NN_UNUSED(downloadSize);
        NN_UNUSED(throughput);
        NN_DETAIL_NS_TRACE("[SystemReportManager] Unsupported OS.\n");
#endif
    }

    void SystemReportManager::ReportApplicationInstallPerformance(ncm::ApplicationId id, const ncm::ContentMetaKey& key, ncm::StorageId storageId, int64_t downloadSize, int64_t throughput) const NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        if (m_IsEnabled)
        {
            prepo::SystemPlayReport report("application_dl_throughput");
            NN_NS_SREPO_DO(report.SetApplicationId(PrepoApplicationIdForReportingDownloadThroughput));
            char buffer[256];
            report.SetBuffer(buffer, sizeof(buffer));

            NN_NS_SREPO_DO(report.Add("ApplicationId", prepo::Any64BitId{ id.value }));
            NN_NS_SREPO_DO(report.Add("ContentMetaId", prepo::Any64BitId{ key.id }));
            NN_NS_SREPO_DO(report.Add("ContentMetaType", ncm::GetContentMetaTypeString(key.type)));
            NN_NS_SREPO_DO(report.Add("Version", prepo::Any64BitId{ key.version }));
            NN_NS_SREPO_DO(report.Add("StorageId", ncm::GetStorageIdStringForPlayReport(storageId)));
            NN_NS_SREPO_DO(report.Add("DownloadSize", downloadSize));
            NN_NS_SREPO_DO(report.Add("ThroughputKBps", throughput));

            NN_NS_SREPO_DO(report.Save());
        }
#else
        NN_UNUSED(id);
        NN_UNUSED(key);
        NN_UNUSED(storageId);
        NN_UNUSED(downloadSize);
        NN_UNUSED(throughput);
        NN_DETAIL_NS_TRACE("[SystemReportManager] Unsupported OS.\n");
#endif
    }

    void SystemReportManager::ReportSystemUpdate(SystemUpdateType type, uint64_t sourceSystemUpdateMetaId, uint32_t sourceSystemUpdateMetaVersion, uint64_t destinationSystemUpdateMetaId, uint32_t destinationSystemUpdateMetaVersion, bool isExFatSupported, bool isExFatSupportedAfterUpdate, bool isRebootless) const NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        if (m_IsEnabled)
        {
            prepo::SystemPlayReport report("systemupdate_pass");
            NN_NS_SREPO_DO(report.SetApplicationId(PrepoApplicationIdForReportingSystemUpdate));
            char buffer[256];
            report.SetBuffer(buffer, sizeof(buffer));

            int64_t sourceExFatStatus = isExFatSupported ? 1 : 0;
            int64_t destinationExFatStatus = isExFatSupportedAfterUpdate ? 1 : 0;
            int64_t rebootless = isRebootless ? 1 : 0;

            NN_NS_SREPO_DO(report.Add("Type", static_cast<int64_t>(type)));
            NN_NS_SREPO_DO(report.Add("SourceSystemUpdateMetaId", prepo::Any64BitId{ sourceSystemUpdateMetaId }));
            NN_NS_SREPO_DO(report.Add("SourceSystemUpdateMetaVersion", static_cast<int64_t>(sourceSystemUpdateMetaVersion)));
            NN_NS_SREPO_DO(report.Add("SourceExFatStatus", sourceExFatStatus));
            NN_NS_SREPO_DO(report.Add("DestinationSystemUpdateMetaId", prepo::Any64BitId{ destinationSystemUpdateMetaId }));
            NN_NS_SREPO_DO(report.Add("DestinationSystemUpdateMetaVersion", static_cast<int64_t>(destinationSystemUpdateMetaVersion)));
            NN_NS_SREPO_DO(report.Add("DestinationExFatStatus", destinationExFatStatus));
            NN_NS_SREPO_DO(report.Add("Rebootless", rebootless));

            NN_NS_SREPO_DO(report.Save());
        }
#else
        NN_UNUSED(type);
        NN_UNUSED(sourceSystemUpdateMetaId);
        NN_UNUSED(sourceSystemUpdateMetaVersion);
        NN_UNUSED(destinationSystemUpdateMetaId);
        NN_UNUSED(destinationSystemUpdateMetaVersion);
        NN_UNUSED(isExFatSupported);
        NN_UNUSED(isExFatSupportedAfterUpdate);
        NN_UNUSED(isRebootless);
        NN_DETAIL_NS_TRACE("[SystemReportManager] Unsupported OS.\n");
#endif
    }

}}}
