﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ShutdownManager.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include "ns_CleanupUtil.h"
#include "ns_Config.h"
#include "ns_DebugUtil.h"
#include "ns_LogUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {

        void CleanupBase(ApplicationEntityManager* entityManager, IntegratedContentManager* integrated, ApplicationRecordDatabase* recordDb, LockedBufferManager* lockedBufferManager) NN_NOEXCEPT
        {
            NN_NS_TRACE_RESULT_IF_FAILED(integrated->RepairInvalidFileAttribute(ncm::StorageId::SdCard),
                                         "[ShutdownManager] Failed to fix file attribute\n");
            NN_NS_TRACE_RESULT_IF_FAILED(CleanupCacheStorage(recordDb, integrated, lockedBufferManager, IsCleanupOrphanCacheStorageEnabledForDebug()),
                                         "[ShutdownManager] Failed to cleanup cache storage.\n");
            NN_NS_TRACE_RESULT_IF_FAILED(entityManager->DeleteRedundantApplicationEntity(),
                                         "[ShutdownManager] Failed to delete redandant application entities\n");
        }

        void CleanupStorage(IntegratedContentManager* integrated, ncm::StorageId storageId) NN_NOEXCEPT
        {
            NN_NS_TRACE_RESULT_IF_FAILED(integrated->CleanupAllPlaceHolder(storageId),
                                         "[ShutdownManager] Failed to cleanup all placeholder\n");
            NN_NS_TRACE_RESULT_IF_FAILED(integrated->CleanupFragments(storageId),
                                         "[ShutdownManager] Failed to delete fragments\n");
        }

        void CleanupForSdCardInsertion(ApplicationEntityManager* entityManager, IntegratedContentManager* integrated, ApplicationRecordDatabase* recordDb, LockedBufferManager* lockedBufferManager) NN_NOEXCEPT
        {
            CleanupBase(entityManager, integrated, recordDb, lockedBufferManager);
            CleanupStorage(integrated, ncm::StorageId::SdCard);
        }

        void CleanupForForceShutdown(ApplicationEntityManager* entityManager, IntegratedContentManager* integrated, ApplicationRecordDatabase* recordDb, LockedBufferManager* lockedBufferManager) NN_NOEXCEPT
        {
            CleanupBase(entityManager, integrated, recordDb, lockedBufferManager);
            CleanupStorage(integrated, ncm::StorageId::Any);

            NN_NS_TRACE_RESULT_IF_FAILED(CleanupAllNetworkInstallTask(),
                                         "[ShutdownManager] Failed to cleanup network install task.\n");
            NN_NS_TRACE_RESULT_IF_FAILED(CleanupAllApplyDeltaTask(),
                                         "[ShutdownManager] Failed to cleanup apply delta task.\n");
            NN_NS_TRACE_RESULT_IF_FAILED(recordDb->CleanupRedundant(),
                                         "[ShutdownManager] Failed to cleanup redundant application record.\n");
        }

    }
    Result ShutdownManager::Initialize(ApplicationEntityManager* entityManager, DataStore* dataStore, IntegratedContentManager* integrated, RequestServer* requestServer, ApplicationRecordDatabase* recordDb, PushNotificationDispatcher* notificationDispatcher, LockedBufferManager* lockedBufferManager, bool needsCleanup) NN_NOEXCEPT
    {
        if (dataStore->DetectsForceShutdown())
        {
            NN_DETAIL_NS_TRACE("[ShutdownManager] Cleanup due to force shutdown\n");

            auto begin = os::GetSystemTick().ToTimeSpan();
            NN_UNUSED(begin);

            CleanupForForceShutdown(entityManager, integrated, recordDb, lockedBufferManager);

            NN_DETAIL_NS_TRACE("[ShutdownManager] Took %lld ms to cleanup force shutdown\n", (os::GetSystemTick().ToTimeSpan() - begin).GetMilliSeconds());
        }
        else if (needsCleanup)
        {
            NN_DETAIL_NS_TRACE("[ShutdownManager] Cleanup due to sdcard insertion\n");


            auto begin = os::GetSystemTick().ToTimeSpan();
            NN_UNUSED(begin);

            CleanupForSdCardInsertion(entityManager, integrated, recordDb, lockedBufferManager);

            NN_DETAIL_NS_TRACE("[ShutdownManager] Took %lld ms to cleanup sdcard insertion\n", (os::GetSystemTick().ToTimeSpan() - begin).GetMilliSeconds());
        }

        {
            NN_NS_TRACE_RESULT_IF_FAILED(CleanupAllPlaceHolderAndFragmentsIfNoTask(integrated),
                                         "[ShutdownManager] Failed to cleanup placeholders and fragments when no task\n");
        }

        m_DataStore = dataStore;
        m_RequestServer = requestServer;
        m_PushNotificationDispatcher = notificationDispatcher;

        NN_RESULT_SUCCESS;
    }

    void ShutdownManager::InitializeForMaintenanceMode() NN_NOEXCEPT {}

    Result ShutdownManager::PrepareShutdown() NN_NOEXCEPT
    {
        // メンテナンスモード用に初期化された場合は何もしない
        if (m_RequestServer)
        {
            m_PushNotificationDispatcher->Stop();
            m_Stopper = m_RequestServer->Stop();
            NN_RESULT_DO(nim::PrepareShutdown());
            NN_RESULT_DO(m_DataStore->PrepareShutdown());
        }

        NN_RESULT_SUCCESS;
    }
}}}
