﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_MountPrivate.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Uuid.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/ns/srv/ns_SdCardManager.h>
#include <nn/ns/srv/ns_MarkFile.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Random.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/hid/system/hid_InputDetection.h>
#endif

#include <mutex>

#include "ns_Config.h"
#include "ns_CleanupUtil.h"
#include "ns_DebugUtil.h"

namespace nn { namespace ns { namespace srv {

    NN_DEFINE_STATIC_CONSTANT(const int SdCardManager::MarkFileDirPathLength);

    namespace {
        Result ConvertFsResult(Result result)
        {
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH_CONVERT(
                    fs::ResultSdCardAccessFailed,
                    ns::ResultSdCardAccessFailed())
                NN_RESULT_CATCH_CONVERT(
                    fs::ResultUsableSpaceNotEnough,
                    ResultSdCardNotEnoughSpace())
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

        bool IsUnexpectedResult(Result result)
        {
            return !(
                result.IsSuccess() ||
                result <= ResultSdCardNoOwnership() ||
                result <= ResultSdCardNeedsSystemUpdate() ||
                result <= ResultSdCardFileSystemCorrupted() ||
                result <= ResultSdCardDatabaseCorrupted() ||
                result <= ResultSdCardNotMounted() ||
                result <= ResultSdCardNotInserted() ||
                result <= ResultSdCardAccessFailed() ||
                result <= ResultSdCardNotEnoughSpace()
                );
        }

        Result NotifyExFatDriverRequiredIfPossible() NN_NOEXCEPT
        {
            NN_RESULT_TRY(NotifyExFatDriverRequired())
                NN_RESULT_CATCH(ResultAlreadyOccupied) { NN_RESULT_THROW(ResultSdCardNotMounted()); }
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        Result MountSdCard(const char* mountName) NN_NOEXCEPT
        {
            NN_RESULT_TRY(fs::MountSdCard(mountName))
                NN_RESULT_CATCH(fs::ResultExFatUnavailable)
                {
                    NN_RESULT_DO(NotifyExFatDriverRequiredIfPossible());
                    NN_RESULT_THROW(ResultSdCardNeedsSystemUpdate());
                }
                NN_RESULT_CATCH(fs::ResultDataCorrupted)
                {
                    NN_RESULT_THROW(ResultSdCardFileSystemCorrupted());
                }
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        Result CheckMarkFile(bool* pOutFound, const char* markFileDirPath, const MarkData* dstData) NN_NOEXCEPT
        {
            MarkFile markFile;
            markFile.Initialize(markFileDirPath);

            NN_RESULT_DO(markFile.HasMarkFile(pOutFound));

            if (!*pOutFound)
            {
                NN_RESULT_SUCCESS;
            }

            MarkData markData;
            NN_RESULT_DO(markFile.ReadMarkData(&markData));

            if (markData != *dstData)
            {
                NN_RESULT_THROW(ResultSdCardNoOwnership());
            }

            NN_RESULT_SUCCESS;
        }

        Result InitializeDatabase(ncm::StorageId storageId) NN_NOEXCEPT
        {
            NN_RESULT_DO(ncm::CreateContentStorage(storageId));
            NN_RESULT_DO(ncm::CreateContentMetaDatabase(storageId));

            NN_RESULT_SUCCESS;
        }

        Result OpenDatabase(ncm::ContentMetaDatabase* pOutDb, ncm::ContentStorage* pOutStorage, ncm::StorageId storageId) NN_NOEXCEPT
        {
            NN_RESULT_TRY(ncm::OpenContentMetaDatabase(pOutDb, storageId))
                // TORIAEZU: 正式な Result を返すようになったら修正する
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_THROW(ResultSdCardDatabaseCorrupted());
                }
            NN_RESULT_END_TRY
            NN_RESULT_TRY(ncm::OpenContentStorage(pOutStorage, storageId))
                // TORIAEZU: 正式な Result を返すようになったら修正する
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_THROW(ResultSdCardDatabaseCorrupted());
                }
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        Result VerifyDatabase(ncm::StorageId storageId) NN_NOEXCEPT
        {
            NN_RESULT_TRY(ncm::VerifyContentStorage(storageId))
                NN_RESULT_CATCH(ncm::ResultInvalidContentStorageBase)
                {
                    NN_RESULT_THROW(ResultSdCardDatabaseCorrupted());
                }
                // VerifyDatabase を呼ばれている時点で、コンテンツストレージが作られている想定なので、
                // ncm::ResultContentStorageBaseNotFound が返ってきた場合でも ResultSdCardDatabaseCorrupted とする
                NN_RESULT_CATCH(ncm::ResultContentStorageBaseNotFound)
                {
                    NN_RESULT_THROW(ResultSdCardDatabaseCorrupted());
                }
            NN_RESULT_END_TRY

            NN_RESULT_TRY(ncm::VerifyContentMetaDatabase(storageId))
                NN_RESULT_CATCH(ncm::ResultInvalidContentMetaDatabase)
                {
                    NN_RESULT_THROW(ResultSdCardDatabaseCorrupted());
                }
                // セーブファイルが見つからなかった
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                    NN_RESULT_THROW(ResultSdCardDatabaseCorrupted());
                }
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        Result GenerateMarkFile(const char* markFileDirPath, const MarkData* pMarkData) NN_NOEXCEPT
        {
            MarkFile markFile;
            markFile.Initialize(markFileDirPath);

            NN_RESULT_DO(markFile.GenerateMarkFile(pMarkData, nullptr, 0));

            NN_RESULT_SUCCESS;
        }

        Result ForceDeleteIfExist(const char* path) NN_NOEXCEPT
        {
            fs::DirectoryEntryType entry;
            NN_RESULT_TRY(fs::GetEntryType(&entry, path))
                NN_RESULT_CATCH(fs::ResultPathNotFound)
                {
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY

            if (entry == fs::DirectoryEntryType_Directory)
            {
                NN_RESULT_DO(fs::DeleteDirectoryRecursively(path));
            }
            else
            {
                NN_RESULT_DO(fs::DeleteFile(path));
            }

            NN_RESULT_SUCCESS;
        }

        bool IsIncompleteSdCard(const char* saveDirPath) NN_NOEXCEPT
        {
            // sdcard:/Nintendo/save
            // sdcard:/Nintendo/Contents/registered
            // sdcard:/Nintendo/Contents/placehld
            // の全てが無かった場合に0.12 系ファームウェアで初期化された SD カードだとする

            auto result = ncm::VerifyContentStorage(ncm::StorageId::SdCard);
            if (!ncm::ResultContentStorageBaseNotFound::Includes(result))
            {
                return false;
            }

            fs::DirectoryEntryType entry;
            result = fs::GetEntryType(&entry, saveDirPath);
            return fs::ResultPathNotFound::Includes(result);
        }

        class SdCardStateSaveData
        {
        public:
            ~SdCardStateSaveData() NN_NOEXCEPT;
            Result Initialize(const SaveDataInfo& info) NN_NOEXCEPT;
            Result ReadMarkData(MarkData* outValue) const NN_NOEXCEPT;
            Result SetEncryptionSeed() const NN_NOEXCEPT;
            Result IsAttached(bool* outValue) const NN_NOEXCEPT;
            Result SetState(bool isAttached) NN_NOEXCEPT;
            Result NeedsCleanup(bool* outValue, bool isAttached) const NN_NOEXCEPT;

        private:
            static const int FilePathLength = 32;
            typedef kvdb::BoundedString<FilePathLength> Path;

            struct PersistentData
            {
                fs::EncryptionSeed seed;
                bool isAttached;
                Bit8 reserved0x17[3];
                Bit8 cid[fs::SdCardCidSize];
                Bit8 reserved[460];
            };
            NN_STATIC_ASSERT(sizeof(MarkData) + sizeof(PersistentData) == 512);

            SaveDataInfo m_SaveDataInfo;
            MarkFile m_MarkFile;
            bool m_NeedsUnmount{};
            PersistentData m_Data;

        private:
            void InitializePersistentData(PersistentData* outValue) NN_NOEXCEPT;
            Result CompareSdCard(bool* outValue) const NN_NOEXCEPT;
        };

        SdCardStateSaveData::~SdCardStateSaveData() NN_NOEXCEPT
        {
            if (m_NeedsUnmount)
            {
                fs::Unmount(m_SaveDataInfo.mountName);
            }
        }

        Result SdCardStateSaveData::Initialize(const SaveDataInfo& info) NN_NOEXCEPT
        {
            m_SaveDataInfo = info;

            Path dirPath;
            dirPath.AssignFormat("%s:/", m_SaveDataInfo.mountName);

            m_MarkFile.Initialize(dirPath);

            bool hasMarkFile;
            NN_RESULT_DO(m_MarkFile.HasMarkFile(&hasMarkFile));
            if (!hasMarkFile)
            {
                InitializePersistentData(&m_Data);
                m_MarkFile.GenerateMarkFile(&m_Data, sizeof(m_Data));
                NN_RESULT_DO(fs::CommitSaveData(m_SaveDataInfo.mountName));
            }
            else
            {
                size_t dataSize;
                NN_RESULT_DO(m_MarkFile.GetExternalDataSize(&dataSize));
                if (dataSize != sizeof(PersistentData))
                {
                    MarkData markData;
                    m_MarkFile.ReadMarkData(&markData);

                    NN_RESULT_DO(m_MarkFile.DeleteMarkFile());

                    InitializePersistentData(&m_Data);
                    NN_RESULT_DO(m_MarkFile.GenerateMarkFile(&markData, &m_Data, sizeof(m_Data)));

                    NN_RESULT_DO(fs::CommitSaveData(m_SaveDataInfo.mountName));
                }
                else
                {
                    m_MarkFile.ReadExternalData(&m_Data, sizeof(m_Data));
                }
            }

            NN_RESULT_SUCCESS;
        }


        Result SdCardStateSaveData::ReadMarkData(MarkData* outValue) const NN_NOEXCEPT
        {
            NN_RESULT_DO(m_MarkFile.ReadMarkData(outValue));
            NN_RESULT_SUCCESS;
        }

        Result SdCardStateSaveData::SetEncryptionSeed() const NN_NOEXCEPT
        {
            NN_RESULT_DO(fs::SetSdCardEncryptionSeed(m_Data.seed));

            NN_RESULT_SUCCESS;
        }

        Result SdCardStateSaveData::IsAttached(bool* outValue) const NN_NOEXCEPT
        {
            *outValue = m_Data.isAttached;
            NN_RESULT_SUCCESS;
        }

        Result SdCardStateSaveData::SetState(bool isAttached) NN_NOEXCEPT
        {
            m_Data.isAttached = isAttached;
            if (isAttached)
            {
                NN_RESULT_DO(fs::GetSdCardCid(m_Data.cid, sizeof(m_Data.cid)));
            }
            NN_RESULT_DO(m_MarkFile.WriteExternalData(&m_Data, sizeof(m_Data)));

            NN_RESULT_DO(fs::CommitSaveData(m_SaveDataInfo.mountName));

            NN_RESULT_SUCCESS;
        }

        Result SdCardStateSaveData::NeedsCleanup(bool* outValue, bool isAttached) const NN_NOEXCEPT
        {
            if (isAttached)
            {
                bool isSameSdCard;
                auto cidState = GetSdCardCidState();
                switch(cidState)
                {
                case SdCardCidState::Real:
                    NN_RESULT_DO(CompareSdCard(&isSameSdCard));
                    break;
                case SdCardCidState::Same:
                    isSameSdCard = true;
                    break;
                case SdCardCidState::Different:
                    isSameSdCard = false;
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
                *outValue = !isSameSdCard || !m_Data.isAttached;
            }
            else
            {
                *outValue = false;
            }
            NN_RESULT_SUCCESS;
        }

        void SdCardStateSaveData::InitializePersistentData(PersistentData* outValue) NN_NOEXCEPT
        {
            *outValue = {};
            os::GenerateRandomBytes(outValue->seed.value, sizeof(outValue->seed.value));
        }

        Result SdCardStateSaveData::CompareSdCard(bool* outValue) const NN_NOEXCEPT
        {
            // CID のデータが入っていなかったら、本体更新によって最初に起動したと判断し、以前起動時と同じ SD カードが挿されていると判断する
            Bit8 cid[fs::SdCardCidSize] = {};
            if (std::memcmp(cid, m_Data.cid, sizeof(cid)) == 0)
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }

            NN_RESULT_DO(fs::GetSdCardCid(cid, sizeof(cid)));

            *outValue = std::memcmp(cid, m_Data.cid, sizeof(cid)) == 0;
            NN_RESULT_SUCCESS;
        }

    }

    Result SdCardManager::Initialize(const SaveDataInfo& info) NN_NOEXCEPT
    {
        {
            std::lock_guard<os::Mutex> stateLock(m_StateLock);

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
            capsrv::InitializeAlbumControl();
#endif // NN_BUILD_CONFIG_OS_WIN

            SdCardStateSaveData sdState;
            NN_RESULT_DO(sdState.Initialize(info));
            NN_RESULT_DO(sdState.SetEncryptionSeed());

            NN_RESULT_DO(sdState.ReadMarkData(&m_SystemMarkData));

            NN_RESULT_DO(fs::OpenSdCardDetectionEventNotifier(&m_Notifier));
            NN_RESULT_DO(m_Notifier->BindEvent(&m_SdCardDetectEvent, os::EventClearMode_ManualClear));
            os::ClearSystemEvent(&m_SdCardDetectEvent);

            os::InitializeEvent(&m_EndEvent, false, os::EventClearMode_ManualClear);
            os::InitializeEvent(&m_EmulationEvent, false, os::EventClearMode_ManualClear);

            UpdateMountStateResult(InitializeSdCard());
            m_IsSdCardAttached = m_CurrentResult.IsSuccess();

            NN_RESULT_DO(sdState.NeedsCleanup(&m_NeedsSystemCleanup, m_IsSdCardAttached));
            NN_RESULT_DO(sdState.SetState(m_IsSdCardAttached));
            fs::SetSdCardAccessibility(m_IsSdCardAttached);

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
            if (IsAttached())
            {
                capsrv::NotifyAlbumStorageIsAvailable(capsrv::AlbumStorage_Sd);
            }
#endif // NN_BUILD_CONFIG_OS_WIN
        }

        {
            std::lock_guard<os::Mutex> threadLock(m_ThreadLock);
            m_Thread.emplace();
            NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_Thread.value(), [](void* p)
                {
                    reinterpret_cast<SdCardManager*>(p)->Run();
                }, this, m_Stack, m_StackSize, NN_SYSTEM_THREAD_PRIORITY(nssrv, DetectSdCardTask)));
            os::SetThreadNamePointer(&m_Thread.value(), NN_SYSTEM_THREAD_NAME(nssrv, DetectSdCardTask));
            os::StartThread(&m_Thread.value());
        }

        NN_RESULT_SUCCESS;
    }

    void SdCardManager::CleanupThread() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_ThreadLock);
        if (m_Thread)
        {
            os::SignalEvent(&m_EndEvent);

            os::WaitThread(&m_Thread.value());
            os::DestroyThread(&m_Thread.value());

            // スレッドが復帰することはないので、イベントもそれに併せて破棄する
            os::DestroySystemEvent(&m_SdCardDetectEvent);

            m_Thread = util::nullopt;
        }
    }

    void SdCardManager::Finalize() NN_NOEXCEPT
    {
        Unmount();

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
        capsrv::FinalizeAlbumControl();
#endif // NN_BUILD_CONFIG_OS_WIN
    }

    void SdCardManager::Run() NN_NOEXCEPT
    {
        for (;;)
        {
            auto index = os::WaitAny(&m_SdCardDetectEvent, &m_EndEvent, &m_EmulationEvent);
            if (index == 1)
            {
                os::ClearEvent(&m_EndEvent);
                return;
            }
            else if (index == 2)
            {
                os::ClearEvent(&m_EmulationEvent);
            }
            else
            {
                os::ClearSystemEvent(&m_SdCardDetectEvent);
            }

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
            // 無操作状態を解除
            nn::hid::system::NotifyInputDetector(nn::hid::system::InputSourceId::SdCardSlot::Mask);
            NN_DETAIL_NS_TRACE("[SdCardManager] Notified hid input detector\n");
#endif // NN_BUILD_CONFIG_OS_WIN

            // 利用可能状態だったSD カードが抜かれた
            if (!m_IsInsertingEmulation && IsAttached())
            {
                NN_DETAIL_NS_TRACE("[SdCardManager] SdCard remove detected\n");

                DetachSdCard();
                m_RemovedEvent.Signal();

                // Attach の状態の場合、必ず再起動するので、ループを抜けても問題ない
                return;
            }

            // 起動後に SD カードが挿入された
            else if (!m_IsRemovingEmulation && (CanAccessSdCard() || m_IsInsertingEmulation))
            {
                NN_DETAIL_NS_TRACE("[SdCardManager] SdCard insert detected\n");

                UpdateMountStateResult(Verify());
                m_ChangedEvent.Signal();
                m_IsInsertingEmulation = false;
            }
            // 起動後に SD カードが抜去された
            else
            {
                NN_DETAIL_NS_TRACE("[SdCardManager] SdCard remove detected\n");

                UpdateMountStateResult(ResultSdCardNotInserted());
                m_ChangedEvent.Signal();
                m_IsRemovingEmulation = false;
            }
        }
    }

    Result SdCardManager::GetMountStatusChangedEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
    {
        outValue.Set(sf::NativeHandle(m_ChangedEvent.GetReadableHandle(), false));
        NN_RESULT_SUCCESS;
    }

    Result SdCardManager::GetRemovedEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
    {
        outValue.Set(sf::NativeHandle(m_RemovedEvent.GetReadableHandle(), false));
        NN_RESULT_SUCCESS;
    }

    Result SdCardManager::Cleanup() NN_NOEXCEPT
    {
        NN_RESULT_DO(GetCleanupSdCardResultForDebug());

        std::lock_guard<os::Mutex> lock(m_StateLock);
        NN_RESULT_THROW_UNLESS(!IsAttached(), ResultSdCardIsBusy());
        NN_RESULT_THROW_UNLESS(CanAccessSdCard(), ResultSdCardNotInserted());

        NN_RESULT_DO(ncm::InactivateContentMetaDatabase(ncm::StorageId::SdCard));
        NN_RESULT_DO(ncm::InactivateContentStorage(ncm::StorageId::SdCard));
        // TORIAEZU: (SIGLO-46492) fs::DeleteSaveData でセーブデータを消せないので、 save ディレクトリを直接消す
        // NN_RESULT_DO(ncm::CleanupContentMetaDatabase(ncm::StorageId::SdCard));

        NN_RESULT_DO(MountSdCard(MountName));

        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount(MountName);
        };

        // TORIAEZU: (SIGLO-46492) fs::DeleteSaveData でセーブデータを消せないので、 save ディレクトリを直接消す
        NN_RESULT_DO(ForceDeleteIfExist(m_SaveDirPath));

        // マークファイルの削除
        MarkFile markFile;
        markFile.Initialize(m_MarkFileDirPath);
        bool hasFile;
        NN_RESULT_DO(markFile.HasMarkFile(&hasFile));
        if (hasFile)
        {
            NN_RESULT_DO(markFile.DeleteMarkFile());
        }

        // コンテンツディレクトリの削除
        NN_RESULT_DO(ForceDeleteIfExist(m_ContentsDirPath));

        NN_RESULT_SUCCESS;
    }

    Result SdCardManager::CheckMountStatus() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);
        return m_CurrentResult;
    }

    Result SdCardManager::GetLastMountUnexpectedResult() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);
        return m_LastMountUnexpectedResult;
    }

    bool SdCardManager::IsAttached() const NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);
        return m_IsSdCardAttached;
    }

    Result SdCardManager::InitializeSdCard() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(IsMountSdCardEnabled() && CanAccessSdCard(), ResultSdCardNotInserted());
        NN_RESULT_DO(MountSdCard(MountName));

        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount(MountName);
        };

        bool hasMarkFile;
        NN_RESULT_DO(CheckMarkFile(&hasMarkFile, m_MarkFileDirPath, &m_SystemMarkData));

        if (!hasMarkFile || IsIncompleteSdCard(m_SaveDirPath))
        {
            NN_RESULT_DO(InitializeDatabase(ncm::StorageId::SdCard));
            if (!hasMarkFile)
            {
                NN_RESULT_DO(GenerateMarkFile(m_MarkFileDirPath, &m_SystemMarkData));
            }
        }
        else
        {
            NN_RESULT_DO(VerifyDatabase(ncm::StorageId::SdCard));
        }

        bool isSuccess = false;
        NN_RESULT_DO(ncm::ActivateContentStorage(ncm::StorageId::SdCard));
        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                // Result は無視
                ncm::InactivateContentStorage(ncm::StorageId::SdCard);
            }
        };
        NN_RESULT_DO(ncm::ActivateContentMetaDatabase(ncm::StorageId::SdCard));
        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                // Result は無視
                ncm::InactivateContentMetaDatabase(ncm::StorageId::SdCard);
            }
        };

        ncm::ContentMetaDatabase db;
        ncm::ContentStorage storage;

        NN_RESULT_DO(OpenDatabase(&db, &storage, ncm::StorageId::SdCard));
        m_ContentManager->Register(ncm::StorageId::SdCard, std::move(db), std::move(storage));

        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    Result SdCardManager::Verify() NN_NOEXCEPT
    {
        NN_RESULT_DO(MountSdCard(MountName));

        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount(MountName);
        };

        bool hasMarkFile;
        NN_RESULT_DO(CheckMarkFile(&hasMarkFile, m_MarkFileDirPath, &m_SystemMarkData));

        if (!IsIncompleteSdCard(m_SaveDirPath))
        {
            NN_RESULT_TRY(VerifyDatabase(ncm::StorageId::SdCard))
                NN_RESULT_CATCH(ResultSdCardDatabaseCorrupted)
                {
                    if (hasMarkFile)
                    {
                        NN_RESULT_RETHROW;
                    }
                    else
                    {
                        // マークファイルがないときは再起動すれば正常に使えるので、正常状態と同じ
                        NN_RESULT_THROW(ResultSdCardNotMounted());
                    }
                }
            NN_RESULT_END_TRY;
        }

        NN_RESULT_THROW(ResultSdCardNotMounted());
    }

#if defined (NN_BUILD_CONFIG_OS_WIN)
    // TORIAEZU: Win32 環境では fs::IsSdCardInserted() が常に false になるので、代わりにメソッドを用意する
    bool SdCardManager::CanAccessSdCard() NN_NOEXCEPT
    {
        bool isEnable = true;
        auto result = fs::MountSdCardForDebug(MountName);
        if (result.IsFailure())
        {
            isEnable = false;
        }

        if (isEnable)
        {
            fs::Unmount(MountName);
        }
        return isEnable;
    }
#else
    bool SdCardManager::CanAccessSdCard() NN_NOEXCEPT
    {
        return fs::IsSdCardInserted();
    }
#endif

    void SdCardManager::DetachSdCard() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);
        if (IsAttached())
        {
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
            capsrv::NotifyAlbumStorageIsUnavailable(capsrv::AlbumStorage_Sd);
#endif // NN_BUILD_CONFIG_OS_WIN

            m_ContentManager->Unregister(ncm::StorageId::SdCard);
            NN_ABORT_UNLESS_RESULT_SUCCESS(ncm::InactivateContentStorage(ncm::StorageId::SdCard));
            NN_ABORT_UNLESS_RESULT_SUCCESS(ncm::InactivateContentMetaDatabase(ncm::StorageId::SdCard));
            m_ViewManager->InvalidateCache();

            m_IsSdCardAttached = false;
            fs::SetSdCardAccessibility(false);
            m_CurrentResult = ResultSdCardNotInserted();
        }
    }

    void SdCardManager::Unmount() NN_NOEXCEPT
    {
        CleanupThread();
        DetachSdCard();
    }

    Result SdCardManager::NeedsSystemUpdateToFormat(bool* outValue) NN_NOEXCEPT
    {
        *outValue = false;
        NN_RESULT_TRY(fs::FormatSdCardDryRun())
            NN_RESULT_CATCH(fs::ResultExFatUnavailable)
            {
                NN_RESULT_DO(NotifyExFatDriverRequired());
                *outValue = true;
            }
            NN_RESULT_CATCH_ALL
            {
                std::lock_guard<os::Mutex> lock(m_StateLock);
                m_LastFormatUnexpectedResult = NN_RESULT_CURRENT_RESULT;
                NN_RESULT_THROW(ResultSdCardFormatUnexpected());
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result SdCardManager::Format() NN_NOEXCEPT
    {
        m_RequestStopper = m_RequestServer->Stop();

        NN_RESULT_DO(DoFormatOperation(CleanupAllNetworkInstallTask()));
        NN_RESULT_DO(DoFormatOperation(CleanupApplyDeltaTask(ncm::StorageId::SdCard)));
        NN_RESULT_DO(DoFormatOperation(m_RecordDb->CleanupRedundant()));

        Unmount();

        NN_RESULT_DO(DoFormatOperation(fs::FormatSdCard()));

        NN_RESULT_SUCCESS;
    }

    Result SdCardManager::GetLastFormatUnexpectedResult() const NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);
        return m_LastFormatUnexpectedResult;
    }

    void SdCardManager::UpdateMountStateResult(Result result) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);

        m_CurrentResult = ConvertFsResult(result);

        if (IsUnexpectedResult(m_CurrentResult))
        {
            m_CurrentResult = ResultSdCardMountUnexpected();
            m_LastMountUnexpectedResult = result;
        }
    }

    Result SdCardManager::DoFormatOperation(Result result) NN_NOEXCEPT
    {
        if (result.IsFailure())
        {
            std::lock_guard<os::Mutex> lock(m_StateLock);
            m_LastFormatUnexpectedResult = result;
            NN_RESULT_THROW(ResultSdCardFormatUnexpected());
        }
        else
        {
            NN_RESULT_SUCCESS;
        }
    }

    bool SdCardManager::NeedsSystemCleanup() const NN_NOEXCEPT
    {
        return m_NeedsSystemCleanup;
    }

    void SdCardManager::Insert() NN_NOEXCEPT
    {
        m_IsInsertingEmulation = true;
        os::SignalEvent(&m_EmulationEvent);
    }

    void SdCardManager::Remove() NN_NOEXCEPT
    {
        m_IsRemovingEmulation = true;
        os::SignalEvent(&m_EmulationEvent);
    }

}}}
