﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/srv/ns_RightsEnvironment.h>

#ifdef NN_BUILD_CONFIG_OS_HORIZON

#include <nn/nn_Common.h>
#include <nn/nn_SdkAbort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <array>
#include <type_traits>
#include <cstring>

#include <nn/es/es_Types.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/es/es_ActiveRightsContextApi.h>
#include <nn/es/es_ActiveRightsContextTypes.h>
#include "ns_RightsUtil.h"
#include "ns_AccountUtil.h"

namespace nn { namespace ns { namespace srv {

namespace {

inline RightsEnvironmentHandle CreateRightsEnvironmentHandle(es::ActiveRightsContextHandle esHandle) NN_NOEXCEPT
{
    RightsEnvironmentHandle handle = {};
    static_assert(sizeof(esHandle) <= sizeof(handle), "");
    std::memcpy(&handle, &esHandle, sizeof(esHandle));
    return handle;
}

inline es::ActiveRightsContextHandle GetEsActiveRightsContextHandle(RightsEnvironmentHandle handle) NN_NOEXCEPT
{
    es::ActiveRightsContextHandle esHandle = {};
    static_assert(sizeof(esHandle) <= sizeof(handle), "");
    std::memcpy(&esHandle, &handle, sizeof(esHandle));
    return esHandle;
}

}

Result RightsEnvironmentManager::CreateRightsEnvironment(RightsEnvironmentHandle* pOut) NN_NOEXCEPT
{
    auto esHandle = es::CreateActiveRightsContext();
    *pOut = CreateRightsEnvironmentHandle(esHandle);
    NN_RESULT_SUCCESS;
}

void RightsEnvironmentManager::DestroyRightsEnvironment(RightsEnvironmentHandle handle) NN_NOEXCEPT
{
    auto esHandle = GetEsActiveRightsContextHandle(handle);
    es::CloseActiveRightsContextHandle(esHandle);
}

Result RightsEnvironmentManager::AddTargetApplicationToRightsEnvironment(RightsEnvironmentHandle handle, ncm::ApplicationId applicationId, const IntegratedContentManager* pIntegratedContentManager, ApplicationRecordDatabase* pApplicationRecordDatabase) NN_NOEXCEPT
{
    auto esHandle = GetEsActiveRightsContextHandle(handle);
    ApplicationRecordAccessor acc;
    NN_RESULT_DO(pApplicationRecordDatabase->Open(&acc, applicationId));
    for (auto&& storageContentMetaKey : acc.GetSpan())
    {
        auto contentMetaKey = storageContentMetaKey.key;
        bool hasKey;
        NN_RESULT_DO(pIntegratedContentManager->HasContentMetaKey(&hasKey, contentMetaKey));
        if (!hasKey)
        {
            continue;
        }
        auto rightsIds = std::array<es::RightsId, 10>(); // ここのバッファは現想定では 1 しかないため、10 もあれば十分
        int n;
        NN_RESULT_DO(GetRightsIdList(&n, rightsIds.data(), static_cast<int>(rightsIds.size()), contentMetaKey, pIntegratedContentManager));
        // TODO: GetRightsIdListLightly を使う？
        NN_RESULT_DO(es::RegisterRightsIdList(esHandle, rightsIds.data(), n));
    }
    NN_RESULT_SUCCESS;
}

Result RightsEnvironmentManager::SetUsersToRightsEnvironment(RightsEnvironmentHandle handle, bool anyUser, util::Span<const account::Uid> uids) NN_NOEXCEPT
{
    auto esHandle = GetEsActiveRightsContextHandle(handle);
    std::array<account::NintendoAccountId, account::UserCountMax> naids;
    NN_ABORT_UNLESS(static_cast<size_t>(uids.size()) <= naids.size(), "TODO:(Result) PreConditionViolation");
    auto n = 0;
    for (auto&& uid : uids)
    {
        account::NintendoAccountId naid;
        NN_RESULT_DO(GetNintendoAccountId(&naid, uid));
        naids[n++] = naid;
    }
    return es::SetUsersOfAccountRestrictedRights(esHandle, naids.data(), n, anyUser);
}

Result RightsEnvironmentManager::ActivateRightsEnvironment(RightsEnvironmentHandle handle) NN_NOEXCEPT
{
    auto esHandle = GetEsActiveRightsContextHandle(handle);
    es::CheckRightsIdListValidity(esHandle);
    es::RemoveUnavailableRightsIds(esHandle);
    return es::BeginUsingActiveRightsContext(esHandle);
}

void RightsEnvironmentManager::DeactivateRightsEnvironment(RightsEnvironmentHandle handle) NN_NOEXCEPT
{
    auto esHandle = GetEsActiveRightsContextHandle(handle);
    es::EndUsingActiveRightsContext(esHandle);
}

Result RightsEnvironmentManager::GetRightsEnvironmentStatus(RightsEnvironmentStatus* pOut, RightsEnvironmentHandle handle) NN_NOEXCEPT
{
    auto esHandle = GetEsActiveRightsContextHandle(handle);
    RightsEnvironmentStatus ret = {};
    auto status = es::GetActiveRightsContextStatus(esHandle);
    ret.isAvailable = status == es::ActiveRightsContextStatus::ActiveRightsContextStatus_Active;
    auto expirationTime = es::GetActiveRightsContextExpiredTime(esHandle);
    ret.expirationTimeInTick = expirationTime.GetInt64Value();
    *pOut = ret;
    NN_RESULT_SUCCESS;
}

int RightsEnvironmentManager::ListActivatedRights(es::RightsId* outList, int listCount, RightsEnvironmentHandle handle) NN_NOEXCEPT
{
    auto esHandle = GetEsActiveRightsContextHandle(handle);
    return es::ListUsingRightsIds(outList, listCount, esHandle);
}

ApplicationRightsStatus RightsEnvironmentManager::GetApplicationRightsStatus(RightsEnvironmentHandle handle) NN_NOEXCEPT
{
    auto esHandle = GetEsActiveRightsContextHandle(handle);

    // NotChecked を解消する
    es::CheckRightsIdListValidity(esHandle);

    auto status = es::GetActiveRightsContextStatus(esHandle);
    switch (status)
    {
    case es::ActiveRightsContextStatus_Active: return ApplicationRightsStatus::Available;
    case es::ActiveRightsContextStatus_Expired: return ApplicationRightsStatus::Expired;
    case es::ActiveRightsContextStatus_Inactive: return ApplicationRightsStatus::Unavailable;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

// Warning ///////////////////////////////////////////////
// 関数追加時には、非 Horiozn 環境用のものを忘れず追加すること //
// Warning ///////////////////////////////////////////////

}}}

#else

#include <nn/result/result_HandlingUtility.h> // NOLINT(analysis/include)

namespace nn { namespace ns { namespace srv {

Result RightsEnvironmentManager::CreateRightsEnvironment(RightsEnvironmentHandle* pOut) NN_NOEXCEPT
{
    *pOut = {};
    NN_RESULT_SUCCESS;
}

void RightsEnvironmentManager::DestroyRightsEnvironment(RightsEnvironmentHandle) NN_NOEXCEPT
{
}

Result RightsEnvironmentManager::AddTargetApplicationToRightsEnvironment(RightsEnvironmentHandle, ncm::ApplicationId, const IntegratedContentManager*, ApplicationRecordDatabase*) NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

Result RightsEnvironmentManager::SetUsersToRightsEnvironment(RightsEnvironmentHandle, bool, util::Span<const account::Uid>) NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

Result RightsEnvironmentManager::ActivateRightsEnvironment(RightsEnvironmentHandle) NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

void RightsEnvironmentManager::DeactivateRightsEnvironment(RightsEnvironmentHandle) NN_NOEXCEPT
{
}

Result RightsEnvironmentManager::GetRightsEnvironmentStatus(RightsEnvironmentStatus* pOut, RightsEnvironmentHandle) NN_NOEXCEPT
{
    *pOut = {};
    NN_RESULT_SUCCESS;
}

int RightsEnvironmentManager::ListActivatedRights(es::RightsId*, int, RightsEnvironmentHandle) NN_NOEXCEPT
{
    return 0;
}

ApplicationRightsStatus RightsEnvironmentManager::GetApplicationRightsStatus(RightsEnvironmentHandle) NN_NOEXCEPT
{
    return ApplicationRightsStatus::Available;
}

}}}

#endif
