﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bgtc/bgtc_Api.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_RequestServerManager.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <mutex>

#define NN_RSM_TRACE(...) NN_DETAIL_NS_TRACE("[RequestServerManager]: " __VA_ARGS__)

namespace nn { namespace ns { namespace srv {
    namespace {
        void NotifyTaskStarting() NN_NOEXCEPT
        {
            NN_RSM_TRACE("  bgtc::NotifyTaskStarting()\n");
            // bgtc::NotifyTaskStarting の ABORT でのハンドリングは禁止されているのでログを出すだけ
            auto result = bgtc::NotifyTaskStarting();
            if (result.IsFailure())
            {
                NN_RSM_TRACE("  bgtc::NotifyTaskStarting() failed. %08x\n", result.GetInnerValueForDebug());
            }
        }

        void NotifyTaskFinished() NN_NOEXCEPT
        {
            NN_RSM_TRACE("  bgtc::NotifyTaskFinished()\n");
            bgtc::NotifyTaskFinished();
        }

        bool IsTaskRunnableNetworkConnection(nifm::NetworkConnection* connection) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(connection);

            auto isFullAwake = !bgtc::IsInHalfAwake();
            auto isAvailable = connection->IsAvailable();
            auto isRequestOnHold = connection->IsRequestOnHold();
            auto isTaskRunnableNetworkConnection = isFullAwake || isAvailable || isRequestOnHold;
            NN_RSM_TRACE("IsTaskRunnableNetworkConnection\n");
            NN_RSM_TRACE("  IsFullAwake    : %s\n", isFullAwake ? "true" : "false");
            NN_RSM_TRACE("  IsAvailable    : %s\n", isAvailable ? "true" : "false");
            NN_RSM_TRACE("  IsRequestOnHold: %s\n", isRequestOnHold ? "true" : "false");
            NN_RSM_TRACE("  * -> IsTaskRunnableNetworkConnection: %s\n", isTaskRunnableNetworkConnection ? "true" : "false");

            return isTaskRunnableNetworkConnection;
        }

        bool IsTaskRunnable(nifm::NetworkConnection* connection, bool hasRequest) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(connection);

            return hasRequest && IsTaskRunnableNetworkConnection(connection);
        }

        class State
        {
        public:
            virtual State* Update(bool hasRequest) NN_NOEXCEPT = 0;
            virtual void Enter() NN_NOEXCEPT = 0;
        };

        class TaskStartingState : public State
        {
        public:
            virtual State* Update(bool hasRequest) NN_NOEXCEPT NN_OVERRIDE
            {
                if (!IsTaskRunnable(m_Connection, hasRequest))
                {
                    return m_TaskFinishedState;
                }
                return this;
            }

            virtual void Enter() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RSM_TRACE("Enter [TaskStartingState]\n");
                NotifyTaskStarting();
            }

            void Initialize(nifm::NetworkConnection* connection, State* taskFinishedState) NN_NOEXCEPT
            {
                m_Connection = connection;
                m_TaskFinishedState = taskFinishedState;
            }
        private:
            nifm::NetworkConnection* m_Connection;
            State* m_TaskFinishedState;
        };

        class TaskFinishedState : public State
        {
        public:
            virtual State* Update(bool hasRequest) NN_NOEXCEPT NN_OVERRIDE
            {
                if (IsTaskRunnable(m_Connection, hasRequest))
                {
                    return m_TaskStartingState;
                }
                return this;
            }

            virtual void Enter() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RSM_TRACE("Enter [TaskFinishedState]\n");
                NotifyTaskFinished();
            }

            void Initialize(nifm::NetworkConnection* connection, State* taskStartingState) NN_NOEXCEPT
            {
                m_Connection = connection;
                m_TaskStartingState = taskStartingState;
            }
        private:
            nifm::NetworkConnection* m_Connection;
            State* m_TaskStartingState;
        };

        class StateMachine
        {
        public:
            void Run(RequestServer* requestServerList[],
                int requestServerCount,
                os::SystemEvent* stopEvent,
                os::SystemEvent* connectionEvent,
                os::SystemEvent* bgtcTriggerEvent,
                nifm::NetworkConnection& connection) NN_NOEXCEPT
            {
                MultiWaitSystemEvent multiWait;
                for (int i = 0; i < requestServerCount; ++i)
                {
                    multiWait.Link(&requestServerList[i]->GetHasRequestChangedEvent());
                }
                auto bgtcTriggerEventIndex = multiWait.Link(bgtcTriggerEvent);
                auto connectionEventIndex = multiWait.Link(connectionEvent);
                auto stopEventIndex = multiWait.Link(stopEvent);

                TaskStartingState startingState;
                TaskFinishedState finishedState;
                startingState.Initialize(&connection, &finishedState);
                finishedState.Initialize(&connection, &startingState);

                State* state = &finishedState;

                while (NN_STATIC_CONDITION(true))
                {
                    state = state->Update(HasRequest(requestServerList, requestServerCount));
                    state->Enter();

                    auto signaled = multiWait.WaitAny();

                    if (signaled == stopEventIndex)
                    {
                        NN_RSM_TRACE("StopEvent signaled.\n");
                        break;
                    }
                    else if (signaled == bgtcTriggerEventIndex)
                    {
                        NN_RSM_TRACE("ScheduleBackgroundTaskEvent signaled.\n");
                        NN_RSM_TRACE("  Current state = %s\n", bgtc::IsInHalfAwake() ? "HalfAwake" : "FullAwake");
                        bgtcTriggerEvent->Clear();

                        if (bgtc::IsInHalfAwake())
                        {
                            NN_RSM_TRACE("    Submit network request\n");
                            connection.SubmitRequest();
                        }
                    }
                    else if (signaled == connectionEventIndex)
                    {
                        NN_RSM_TRACE("ConnectionEvent signaled.\n");
                        NN_RSM_TRACE("  IsAvailable     = %s\n", connection.IsAvailable() ? "true" : "false");
                        NN_RSM_TRACE("  IsRequestOnHold = %s\n", connection.IsRequestOnHold() ? "true" : "false");
                        connectionEvent->Clear();
                    }
                    else if (signaled >= 0 && signaled < requestServerCount)
                    {
                        NN_RSM_TRACE("HasRequestChangedEvent signaled.\n");
                        requestServerList[signaled]->GetHasRequestChangedEvent().Clear();
                    }

                }
            }
        private:
            bool HasRequest(RequestServer* requestServerList[], int requestServerCount) NN_NOEXCEPT
            {
                for (int i = 0; i < requestServerCount; ++i)
                {
                    if (requestServerList[i]->HasRequest())
                    {
                        NN_RSM_TRACE("  RequestServer[%d] has request\n", i);
                        NN_RSM_TRACE("  * -> HasRequest = true\n");
                        return true;
                    }
                }
                NN_RSM_TRACE("  HasRequest = false\n");
                return false;
            }

        };
    }


    RequestServerManager::RequestServerManager(uint8_t* stack, size_t stackSize) NN_NOEXCEPT
        : m_Stack(stack), m_StackSize(stackSize), m_Thread()
    {}

    void RequestServerManager::Initialize(RequestServer* requestServerForAppManager, RequestServer* requestServerForSystemUpdate) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(requestServerForAppManager);
        NN_SDK_ASSERT_NOT_NULL(requestServerForSystemUpdate);

        m_RequestServerList[0] = requestServerForAppManager;
        m_RequestServerList[1] = requestServerForSystemUpdate;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, [](void* arg) {
            reinterpret_cast<RequestServerManager*>(arg)->Run();
        }, this, m_Stack, m_StackSize, NN_SYSTEM_THREAD_PRIORITY(ns, RequestServerManager)));

        os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(ns, RequestServerManager));
        os::StartThread(&m_Thread);
    }

    void RequestServerManager::Finalize() NN_NOEXCEPT
    {
        m_StopEvent.Signal();
        os::WaitThread(&m_Thread);
        os::DestroyThread(&m_Thread);
    }

    // -------------------------------

    void RequestServerManager::Run() NN_NOEXCEPT
    {
        NN_RSM_TRACE("Start.\n");

        nifm::NetworkConnection connection(os::EventClearMode_ManualClear);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::SetRequestRequirementPreset(connection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));

        StateMachine sm;
        sm.Run(m_RequestServerList, MaxRequestServerCount, &m_StopEvent, &connection.GetSystemEvent(), &bgtc::GetTriggerEvent(), connection);
    }
}}}
