﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ResultMenu.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/ns/detail/ns_Log.h>

namespace nn { namespace ns { namespace srv {
namespace
{
class ConnectionResolver
{
    NN_DISALLOW_COPY(ConnectionResolver);
    NN_DISALLOW_MOVE(ConnectionResolver);
public:
    ConnectionResolver() NN_NOEXCEPT
    : m_Connection(os::EventClearMode_ManualClear),
      m_RequirementPreset(nifm::RequirementPreset_InternetForSystemProcessPersistent)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::SetRequestRequirementPreset(m_Connection.GetRequestHandle(), m_RequirementPreset));
        m_Connection.SubmitRequest();
    }
    ~ConnectionResolver() NN_NOEXCEPT {}

    os::SystemEvent* GetEvent() NN_NOEXCEPT { return &m_Connection.GetSystemEvent(); }

    RequestList::ProcessType GetProcessType() NN_NOEXCEPT
    {
        if (m_Connection.IsAvailable())
        {
            if (m_RequirementPreset == nifm::RequirementPreset_InternetForSystemProcessContinuous)
            {
                return RequestList::ProcessType::Lightweight;
            }
            else
            {
                return RequestList::ProcessType::All;
            }
        }
        else
        {
            return RequestList::ProcessType::Offline;
        }
    }
    bool NeedsCleanup(RequestList::ProcessType preType) NN_NOEXCEPT
    {
        // 実行しているリクエストと、今のネットワーク状態にずれがあったら一度実行中のタスクをクリア
        return (preType == RequestList::ProcessType::Offline && m_Connection.IsAvailable())
            || (preType != RequestList::ProcessType::Offline && !m_Connection.IsAvailable());
    }

    bool IsContinousRequirementRecommended(bool needsLightweight) NN_NOEXCEPT
    {
        return needsLightweight && !m_Connection.IsAvailable() && m_Connection.GetResult() <= nifm::ResultLowPriority() && m_RequirementPreset != nifm::RequirementPreset_InternetForSystemProcessContinuous;
    }

    void ChangeToContinousRequirement() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[RequestServer] Connection rejected due to low priority\n");

        m_RequirementPreset = nifm::RequirementPreset_InternetForSystemProcessContinuous;
        m_Connection.CancelRequest();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::SetRequestRequirementPreset(m_Connection.GetRequestHandle(), m_RequirementPreset));
        m_Connection.SubmitRequest();

        NN_DETAIL_NS_TRACE("[RequestServer] Continuous connection request submitted\n");
    }

    void RevertToPersistentRequirement() NN_NOEXCEPT
    {
        if (m_RequirementPreset == nifm::RequirementPreset_InternetForSystemProcessContinuous)
        {
            m_RequirementPreset = nifm::RequirementPreset_InternetForSystemProcessPersistent;
            m_Connection.CancelRequest();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::SetRequestRequirementPreset(m_Connection.GetRequestHandle(), m_RequirementPreset));
            m_Connection.SubmitRequest();
        }
    }

private:
    nifm::NetworkConnection m_Connection;
    nifm::RequirementPreset m_RequirementPreset;
};
}

    RequestServer::~RequestServer() NN_NOEXCEPT
    {
        if (m_IsStarted)
        {
            Finalize();
        }
    }

    void RequestServer::Initialize(TimeSpan retryAfter) NN_NOEXCEPT
    {
        m_RetryAfter = retryAfter;
        StartImpl();
        m_IsStarted = true;
    }

    void RequestServer::Finalize() NN_NOEXCEPT
    {
        StopImpl();
        m_IsStarted = false;
    }

    RequestServer::ManagedStop RequestServer::Stop() NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        if (m_StopCount == 0)
        {
            StopImpl();
        }

        m_StopCount++;

        return ManagedStop(this);
    }

    void RequestServer::Unstop() NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        m_StopCount--;

        if (m_StopCount == 0)
        {
            StartImpl();
        }
    }

    void RequestServer::StartImpl() NN_NOEXCEPT
    {
        m_StopEvent.Clear();
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_Thread, [](void* p)
        {
            reinterpret_cast<RequestServer*>(p)->Run();
        },
        this, m_Stack, m_StackSize, NN_SYSTEM_THREAD_PRIORITY(nssrv, RequestServerTask)));
        os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(nssrv, RequestServerTask));
        os::StartThread(&m_Thread);
    }

    void RequestServer::StopImpl() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[RequestServer] Stop\n");
        m_StopEvent.Signal();
        os::WaitThread(&m_Thread);
        os::DestroyThread(&m_Thread);
    }

    void RequestServer::Run() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[RequestServer] Run\n");

        NN_UTIL_SCOPE_EXIT
        {
            NN_DETAIL_NS_TRACE("[RequestServer] Cleanup and signal idle event\n");
            m_List->Cleanup();

            // SIGLO-48321
            // 誰かが止めている状態で半起床に入った場合に半起床が終わらなくなってしまう件をカバーするために
            // 実際にタスクがあっても、リクエストサーバが止まっている時は必ず無い扱いにする。
            UpdateHasRequest(false);
        };

        if (!m_List->NeedsProcess(RequestList::ProcessType::All))
        {
            NN_DETAIL_NS_TRACE("[RequestServer] No needs to process\n");
            return;
        }

        UpdateHasRequest(true);

        ConnectionResolver connection;
        while (m_List->NeedsProcess(RequestList::ProcessType::All))
        {
            NN_DETAIL_NS_TRACE("[RequestServer] Loop\n");

            bool needsProcessLightweight = m_List->NeedsProcess(RequestList::ProcessType::Lightweight);
            if (connection.IsContinousRequirementRecommended(needsProcessLightweight))
            {
                connection.ChangeToContinousRequirement();
            }
            else if (!needsProcessLightweight)
            {
                connection.RevertToPersistentRequirement();
            }

            auto processType = connection.GetProcessType();
            m_List->Prepare(processType);

            // Prepare によって m_List->HasPreparedRequest() が変わる可能性があり、それをケアせず wait してしまうと、
            // 半起床時にずっとスリープしないことになってしまうため、再チェックして 0 なら抜ける
            // ※通常時は問題は起こらない
            if (!m_List->NeedsProcess(RequestList::ProcessType::All))
            {
                return;
            }

            {
                // 前提
                // - 「HasRequest == true && ネットワーク接続確立済み」または「HasRequest == true && ネットワーク接続試行中」の状態でのみ半起床が継続される。
                //       （これのハンドルは RequestServerManager がやっている。）
                // - processType == ProcessType::All の場合はネットワーク接続が確立されている。
                //
                // ネットワーク接続が確立されているにも関わらず準備されたリクエストがない (!m_List->HasPreparedRequest()) ケースは、何らかの理由で Prepare が成功しない状態になっている。
                // （例えば、バージョンリストのリクエストを処理するには Nup チェックの完了が必要だが、Nup チェックがそもそも終わらないようなケース。）
                // Prepare を完了できないことから HasRequest が false になるタイミングが存在しない。
                // つまり、この状態でループが回り続けると、半起床が終わらない。
                // これを回避するために、HasRequest を false にしている。
                //
                // ただし、ネットワーク接続が確立されていない状態(ProcessType != All)では、ネットワーク接続試行中の可能性があるので HasRequest を false にしてはいけない。
                // この状態からはネットワーク接続が確立されるか失敗されるかのどちらかの状態に遷移する。（もしくは初めからネットワーク接続失敗状態。）
                // ネットワーク接続が失敗の場合は、前提条件の通り半起床が終了する。
                // ネットワーク接続が確立された場合は、ループが周り Prepare が再度実行され、その結果いかんで HasRequest の値が変わり、半起床の継続と終了が決まる。
                bool hasNoPreparedRequest = (processType == RequestList::ProcessType::All && !m_List->HasPreparedRequest());
                if (hasNoPreparedRequest)
                {
                    NN_DETAIL_NS_TRACE("[RequestServer] Request waits other events to be prepared\n");
                    UpdateHasRequest(false);
                }

                os::SystemEvent* events[] = { connection.GetEvent(), &m_StopEvent };
                m_List->WaitAlongWith(events, 2);

                if (hasNoPreparedRequest)
                {
                    UpdateHasRequest(true);
                }
            }

            if (m_StopEvent.TryWait())
            {
                return;
            }
            auto isNetworkEvent = connection.GetEvent()->TryWait();
            connection.GetEvent()->Clear();

            if (connection.NeedsCleanup(processType))
            {
                m_List->Cleanup();
            }
            else if (isNetworkEvent)
            {
                // ネットワーク接続が変化しないけれどもイベントが起こることもあるので、その場合はただループを回す
            }
            else
            {
                if (!m_List->HandleDone())
                {
                    NN_DETAIL_NS_TRACE("[RequestServer] Failed to request. Wait %lld ms since the request can work\n", m_RetryAfter.GetMilliSeconds());

                    // HasRequest を False にするタイミングがないと、半起床中に起き続けてしまうことになるため
                    // 再接続までの待ち時間を HasRequest False のタイミングとし、半起床中に寝れるようにする。
                    //
                    // 半起床中にサーバエラーが出た場合はリトライ待ち中にスリープに入るという動作になる。
                    UpdateHasRequest(false);
                    if (m_StopEvent.TimedWait(m_RetryAfter))
                    {
                        return;
                    }
                    UpdateHasRequest(true);
                }
                connection.RevertToPersistentRequirement();
            }
        }
    }

    void RequestServer::UpdateHasRequest(bool value) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_HasRequestLock);

        m_HasRequest = value;
        m_HasRequestChangedEvent.Signal();
    }
}}}
