﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/ns/srv/ns_PushNotificationDisptcher.h>

#include <nn/npns.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "ns_PushNotificationReader.h"
#include "ns_SystemUpdateUtil.h"
#include "ns_NpnsUtil.h"
#include "ns_NotificationToken.h"
#include "ns_SystemUpdateTopic.h"

#include <mutex>

namespace nn { namespace ns { namespace srv {

    namespace {
        npns::NotificationData g_NotificationData = {};

        void UpdateSafeSystemVersion(VulnerabilityManager* vulnerabilityManager, const NotificationInfo& info) NN_NOEXCEPT
        {
            if (info.type == NotificationInfo::NotificationType::SystemUpdate)
            {
                auto result = ProcessRequiredVersionNotification(vulnerabilityManager, info);
                if (result.IsFailure())
                {
                    NN_PN_TRACE("Failed ProcessRequiredVersionNotification. Result = %08x\n", result.GetInnerValueForDebug());
                }
            }
        }
    } // namespace

    Result PushNotificationDispatcher::Initialize(nn::Bit64 listenTarget, ApplicationInstallRequestList* requestList, RequestServer* requestServer, VulnerabilityManager* vulnerabilityManager, ApplicationVersionManager* applicationVersionManager) NN_NOEXCEPT
    {
        auto config = MakeDefaultConfig();
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AsyncTaskManager.Initialize(requestList, requestServer, applicationVersionManager));
        return InitializeCommon(listenTarget, config, vulnerabilityManager);
    }

    Result PushNotificationDispatcher::Initialize(nn::Bit64 listenTarget, Config& config, ApplicationInstallRequestList* requestList, RequestServer* requestServer, VulnerabilityManager* vulnerabilityManager, ApplicationVersionManager* applicationVersionManager, AsyncTaskManager::NotificationHandlerHolder handlers[], int handlersCount, AsyncTaskManager::Config& ntmConfig) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AsyncTaskManager.Initialize(handlers, handlersCount, ntmConfig, requestList, requestServer, applicationVersionManager));
        return InitializeCommon(listenTarget, config, vulnerabilityManager);
    }

    Result PushNotificationDispatcher::InitializeCommon(
        nn::Bit64 listenTarget,
        const Config& config,
        VulnerabilityManager* vulnerabilityManager) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(npns::InitializeForSystem());

        m_VulnerabilityManager = vulnerabilityManager;

        // コンフィグ
        m_ListenTarget = listenTarget;
        m_Config = config;

        // イベント複数待ちの初期化
        nn::npns::GetReceiveEvent(m_NotificationEvent);
        m_MultiWait.Link(&m_StopEvent);
        m_MultiWait.Link(&m_NotificationEvent);

        // 受信スレッドスタート
        Start();

        NN_RESULT_SUCCESS;
    }

    PushNotificationDispatcher::Config PushNotificationDispatcher::MakeDefaultConfig() NN_NOEXCEPT
    {
        Config config = {};
        const char* Name = "ns.notification";
        nn::settings::fwdbg::GetSettingsItemValue(
            &config.retryIntervalMin, sizeof(config.retryIntervalMin), Name, "retry_interval_min");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.retryIntervalMax, sizeof(config.retryIntervalMax), Name, "retry_interval_max");

        nn::settings::fwdbg::GetSettingsItemValue(
            &config.isRequestOnColdBootEnabled, sizeof(config.isRequestOnColdBootEnabled), Name, "enable_request_on_cold_boot");

        config.isNotificationRegistrationEnabled = true;

        return config;
    }

    void PushNotificationDispatcher::Finalize() NN_NOEXCEPT
    {
        Stop();
        npns::FinalizeForSystem();
    }

    void PushNotificationDispatcher::Start() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_DispatchThreadLock);

        if (!m_DispatchThreadRunning)
        {
            m_DispatchThreadRunning = true;
            // スレッド作成
            NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_Thread, [](void* p)
            {
                reinterpret_cast<PushNotificationDispatcher*>(p)->DispatchThreadFunc();
            },
            this, m_Stack, m_StackSize, NN_SYSTEM_THREAD_PRIORITY(nssrv, RequestServerTask)));
            os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(nssrv, RequestServerTask));
            os::StartThread(&m_Thread);
        }
    }

    void PushNotificationDispatcher::Stop() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_DispatchThreadLock);

        if (m_DispatchThreadRunning)
        {
            m_StopEvent.Signal();
            os::WaitThread(&m_Thread);
            os::DestroyThread(&m_Thread);
            m_DispatchThreadRunning = false;
        }
    }

    PushNotificationDispatcher::EventIndex PushNotificationDispatcher::WaitAnyEvents() NN_NOEXCEPT
    {
        auto signaled = m_MultiWait.WaitAny();
        return static_cast<EventIndex>(signaled);
    }

    void PushNotificationDispatcher::SetupNotifications() NN_NOEXCEPT
    {
        NN_PN_TRACE("RetryIntervalMin  : %d\n", m_Config.retryIntervalMin);
        NN_PN_TRACE("RetryIntervalMax  : %d\n", m_Config.retryIntervalMax);
        NN_PN_TRACE("RegisterNT        : %d\n", m_Config.isNotificationRegistrationEnabled);

        NotificationTokenCreationTask notificationTokenTask(m_Config.isNotificationRegistrationEnabled);
        DynamicRightsNotificationTokenCreationTask dynamicRightsNotificationTokenTask(m_Config.isNotificationRegistrationEnabled);

        INpnsOnlineTask* npnsOnlineProcesses[] = {
            &notificationTokenTask,
            &dynamicRightsNotificationTokenTask,
        };

        auto npnsProcessResult =
            ProcessWithNpnsOnlineApi(npnsOnlineProcesses, sizeof(npnsOnlineProcesses) / sizeof(npnsOnlineProcesses[0]), &m_StopEvent, m_Config.retryIntervalMin, m_Config.retryIntervalMax);
        if (npnsProcessResult <= ns::ResultCanceled())
        {
            NN_PN_TRACE("Npns online process is canceled.\n");
            return;
        }
    }

    void PushNotificationDispatcher::DispatchThreadFunc() NN_NOEXCEPT
    {
        // プッシュ通知受信の設定
        SetupNotifications();
        m_SetupCompleted = true;

        // 受信開始
        NN_PN_TRACE("Listen: %016llx.\n", m_ListenTarget);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::ListenTo({ m_ListenTarget }));

        // 待ち受けの前に1度だけ実行する処理
        ProcessBeforeWaiting();

        // 待ち受け処理
        while (NN_STATIC_CONDITION(true))
        {
            NN_PN_TRACE("Waiting event\n");
            auto signaled = m_MultiWait.WaitAny();

            NN_PN_TRACE("Event signaled.\n");
            if (signaled == EventIndex::StopEvent)
            {
                NN_PN_TRACE("StopEvent is signaled.\n");
                m_StopEvent.Clear();
                break;
            }
            else if (signaled == EventIndex::NotificationEvent)
            {
                NN_PN_TRACE("NotificationEvent is signaled.\n");
                Dispatch();
                m_NotificationEvent.Clear();
            }
            else
            {
                NN_ABORT("Not come here.");
            }
        }
    }

    void PushNotificationDispatcher::Dispatch() NN_NOEXCEPT
    {
        Result receiveResult;
        while ((receiveResult = npns::Receive(&g_NotificationData)).IsSuccess())
        {
            NotificationDataReader reader;
            auto parseResult = reader.Initialize(g_NotificationData);

            if (parseResult.IsFailure())
            {
                NN_PN_TRACE(
                    "Invalid notification data. result = %08x\n", parseResult.GetInnerValueForDebug());
                continue;
            }

            auto notificationInfo = reader.GetNotificationInfo();
            m_AsyncTaskManager.Register(notificationInfo);

            UpdateSafeSystemVersion(m_VulnerabilityManager, notificationInfo);
            PushLastInfo(notificationInfo);
        }

        NN_SDK_ASSERT(receiveResult.IsSuccess() ||
            receiveResult <= npns::ResultNotReceived());
    }



    void PushNotificationDispatcher::PushLastInfo(const NotificationInfo& info) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_LastInfoMutex);

        m_LastInfoList[m_LastInfoCount % MaxLastInfoCount] = info;
        m_LastInfoCount++;
    }

    Result PushNotificationDispatcher::RequestDownloadTaskList(const nn::nim::ETag& eTag, bool notifiesRequiredSystemUpdate) NN_NOEXCEPT
    {
        return m_AsyncTaskManager.RequestDownloadTaskList(eTag, notifiesRequiredSystemUpdate);
    }

    Result PushNotificationDispatcher::RequestVersionList(const nn::nim::ETag& eTag) NN_NOEXCEPT
    {
        return m_AsyncTaskManager.RequestVersionList(eTag);
    }

    Result PushNotificationDispatcher::IsNotificationSetupCompleted(sf::Out<bool> outValue) NN_NOEXCEPT
    {
        *outValue = m_SetupCompleted;
        NN_RESULT_SUCCESS;
    }

    Result PushNotificationDispatcher::GetLastNotificationInfoCount(sf::Out<int64_t> outValue) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_LastInfoMutex);

        *outValue = m_LastInfoCount;
        NN_RESULT_SUCCESS;
    }

    Result PushNotificationDispatcher::ListLastNotificationInfo(sf::Out<int> outCount, const sf::OutArray<ns::NotificationInfo>& outList) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_LastInfoMutex);

        auto lastCount = m_LastInfoCount;
        auto maxLastInfoCount = MaxLastInfoCount; // MaxLastInfoCount を std::min に直接入れると undefined reference?
        auto maxReadCount = std::min(static_cast<int>(outList.GetLength()), maxLastInfoCount);
        int readCount = 0;

        while (lastCount != 0 && readCount < maxReadCount)
        {
            outList[readCount] = m_LastInfoList[(lastCount - 1) % MaxLastInfoCount];

            lastCount--;
            readCount++;
        }

        *outCount = readCount;
        NN_RESULT_SUCCESS;
    }

    void PushNotificationDispatcher::ProcessBeforeWaiting() NN_NOEXCEPT
    {
        if (m_Config.isRequestOnColdBootEnabled)
        {
            {
                auto result = ns::RequestBackgroundNetworkUpdate();
                if (result.IsFailure())
                {
                    NN_PN_TRACE("Failed RequestBackgroundNetworkUpdate: %08x\n", result.GetInnerValueForDebug());
                }
            }

            auto emptyEtag = nim::ETag::MakeEmpty();

            {
                auto result = RequestDownloadTaskList(emptyEtag, false);
                if (result.IsFailure())
                {
                    NN_PN_TRACE("Failed RequestDownloadTaskList: %08x\n", result.GetInnerValueForDebug());
                }
            }

            {
                auto result = RequestVersionList(emptyEtag);
                if (result.IsFailure())
                {
                    NN_PN_TRACE("Failed RequestVersionList: %08x\n", result.GetInnerValueForDebug());
                }
            }
        }
    }
}}}
