﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/npns.h>
#include <nn/npns/npns_ApiSystem.h>

#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/ns/srv/ns_PushNotificationDisptcher.h>

#include "ns_NpnsUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace
    {
        bool IsNpnsConnected() NN_NOEXCEPT
        {
            auto state = npns::GetState();
            return  state == npns::State_Connected ||
                    state == npns::State_ConnectedOnHalfAwake;
        }

        Result WaitNpnsConnected(ManualClearSystemEvent* cancelEvent) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(cancelEvent);

            NN_PN_TRACE("Waiting npns conection\n");
            nn::os::SystemEvent npnsStateChangedEvent;
            NN_RESULT_DO(nn::npns::GetStateChangeEvent(npnsStateChangedEvent));

            MultiWaitSystemEvent multiWait;
            auto cancelEventIndex = multiWait.Link(cancelEvent);
            auto npnsStateChangedEventIndex = multiWait.Link(&npnsStateChangedEvent);

            while (!IsNpnsConnected())
            {
                auto signaled = multiWait.WaitAny();
                if (signaled == cancelEventIndex)
                {
                    NN_RESULT_THROW(ns::ResultCanceled());
                }
                else if (signaled == npnsStateChangedEventIndex)
                {
                    npnsStateChangedEvent.Clear();
                }
            }
            NN_RESULT_SUCCESS;
        }

        Result PerformTasks(INpnsOnlineTask* tasks[], int taskCount, ManualClearSystemEvent* cancelEvent) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(tasks);
            NN_SDK_ASSERT_NOT_NULL(cancelEvent);

            for (int i = 0; i < taskCount; ++i)
            {
                auto p = tasks[i];
                if (p->GetStatus() == INpnsOnlineTask::Status::NeedProcess)
                {
                    auto processResult = p->Process(cancelEvent);
                    NN_RESULT_TRY(processResult)
                        NN_RESULT_CATCH(ns::ResultCanceled)
                        {
                            NN_RESULT_RETHROW;
                        }
                        NN_RESULT_CATCH_ALL
                        {
                            NN_PN_TRACE("Npns online task[%d] failed, Result = %08x\n", i, processResult.GetInnerValueForDebug());
                        }
                    NN_RESULT_END_TRY;
                }
            }

            NN_RESULT_SUCCESS;
        }

        bool NeedsRetry(const INpnsOnlineTask* tasks[], int taskCount) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(tasks);

            // INFO: 1つでも完了になっていないタスクがある場合（エラーが発生している場合）はやり直し
            for (int i = 0; i < taskCount; ++i)
            {
                auto& p = tasks[i];
                if (p->GetStatus() == INpnsOnlineTask::Status::NeedProcess)
                {
                    return true;
                }
            }
            return false;
        }
    } // namespace

    // npns のステートが State_Connected になるまで待ってから npns の api を呼ぶ処理を行う。
    Result ProcessWithNpnsOnlineApi(INpnsOnlineTask* tasks[], int taskCount, ManualClearSystemEvent* cancelEvent, uint32_t retryIntervalMin, uint32_t retryIntervalMax) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(cancelEvent);

        auto retryInterval = retryIntervalMin;

        while (NN_STATIC_CONDITION(true))
        {
            NN_RESULT_DO(WaitNpnsConnected(cancelEvent));
            NN_RESULT_DO(PerformTasks(tasks, taskCount, cancelEvent));

            auto needRetry = NeedsRetry(const_cast<const INpnsOnlineTask**>(tasks), taskCount);
            if (!needRetry)
            {
                NN_RESULT_SUCCESS;
            }

            // ウェイトは倍倍にする。上限は retryIntervalMax。
            NN_PN_TRACE("Interval %d sec.\n", retryInterval);
            cancelEvent->TimedWait(nn::TimeSpan::FromSeconds(retryInterval));
            retryInterval = std::min(retryInterval * 2, retryIntervalMax);
        }
    }
}}}
