﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/fs.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ns/ns_Result.h>
#include "ns_JsonUtil.h"
#include "ns_StringUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        Result MakeContentMetaTypeFromString(ncm::ContentMetaType* outValue, const char* str) NN_NOEXCEPT
        {
            if (strcmp(str, "Application") == 0) { *outValue = ncm::ContentMetaType::Application; NN_RESULT_SUCCESS; }
            if (strcmp(str, "Patch") == 0) { *outValue = ncm::ContentMetaType::Patch; NN_RESULT_SUCCESS; }
            if (strcmp(str, "AddOnContent") == 0) { *outValue = ncm::ContentMetaType::AddOnContent; NN_RESULT_SUCCESS; }
            if (strcmp(str, "Delta") == 0) { *outValue = ncm::ContentMetaType::Delta; NN_RESULT_SUCCESS; }

            NN_RESULT_THROW(ResultCommunicationFailed());
        }

    }


    Result JsonParser::MakeBit64FromString(Bit64* outValue, const char* str) NN_NOEXCEPT
    {
        char* endPtr;
        auto value = NN_NS_STR_TO_ULL(str, &endPtr, 16);
        NN_RESULT_THROW_UNLESS(*endPtr == '\0', ns::ResultCommunicationFailed());

        *outValue = value;

        NN_RESULT_SUCCESS;
    }

    Result JsonParser::FindObject(nne::rapidjson::Document::ValueType** outValue, nne::rapidjson::Document::ValueType* document, const char* name, nne::rapidjson::Type type) NN_NOEXCEPT
    {
        auto genericValue = document->FindMember(name);
        NN_RESULT_THROW_UNLESS(genericValue != document->MemberEnd() && genericValue->value.GetType() == type, ns::ResultCommunicationFailed());

        *outValue = &genericValue->value;

        NN_RESULT_SUCCESS;
    }

    Result JsonParser::FindArray(nne::rapidjson::Document::ValueType** outValue, nne::rapidjson::Document::ValueType* document, const char* name) NN_NOEXCEPT
    {
        return FindObject(outValue, document, name, nne::rapidjson::kArrayType);
    }

    Result JsonParser::FindString(const char** str, size_t* strSize, nne::rapidjson::Document::ValueType* document, const char* name) NN_NOEXCEPT
    {
        nne::rapidjson::Document::ValueType* strObj;
        NN_RESULT_DO(FindObject(&strObj, document, name, nne::rapidjson::kStringType));

        *str = strObj->GetString();
        if (strSize != nullptr)
        {
            *strSize = strObj->GetStringLength();
        }

        NN_RESULT_SUCCESS;
    }

    Result JsonParser::FindUint32(uint32_t* outValue, nne::rapidjson::Document::ValueType* document, const char* name) NN_NOEXCEPT
    {
        nne::rapidjson::Document::ValueType* numObj;
        NN_RESULT_DO(FindObject(&numObj, document, name, nne::rapidjson::kNumberType));

        *outValue = numObj->GetUint();

        NN_RESULT_SUCCESS;
    }

    Result JsonParser::FindUint64(uint64_t* outValue, nne::rapidjson::Document::ValueType* document, const char* name) NN_NOEXCEPT
    {
        nne::rapidjson::Document::ValueType* numObj;
        NN_RESULT_DO(FindObject(&numObj, document, name, nne::rapidjson::kNumberType));

        *outValue = numObj->GetUint64();

        NN_RESULT_SUCCESS;
    }

    Result JsonParser::FindContentMetaType(ncm::ContentMetaType* outValue, nne::rapidjson::Document::ValueType* document, const char* name) NN_NOEXCEPT
    {
        const char* str;
        NN_RESULT_DO(FindString(&str, nullptr, document, name));
        NN_RESULT_DO(MakeContentMetaTypeFromString(outValue, str));

        NN_RESULT_SUCCESS;

    }

    Result JsonParser::FindBool(bool* outValue, nne::rapidjson::Document::ValueType* document, const char* name) NN_NOEXCEPT
    {
        auto boolObj = document->FindMember(name);
        NN_RESULT_THROW_UNLESS(boolObj != document->MemberEnd(), ns::ResultCommunicationFailed());
        NN_RESULT_THROW_UNLESS(boolObj->value.IsBool(), ns::ResultCommunicationFailed());
        *outValue = boolObj->value.GetBool();
        NN_RESULT_SUCCESS;
    }

    Result JsonParser::InitializeDocument(nne::rapidjson::Document* document, char* textBuffer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!document->ParseInsitu(textBuffer).HasParseError(), ns::ResultCommunicationFailed());

        NN_RESULT_SUCCESS;
    }
}}}
