﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>

#include <nn/ns/srv/ns_ExFatDriverManager.h>
#include <nn/ns/detail/ns_Log.h>
#include "ns_Config.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        typedef kvdb::BoundedString<32> Path;
        Path GetSaveDataPath() NN_NOEXCEPT
        {
            Path path;
            path.AssignFormat("%s:/%s", ExFatDriverMountName, "save.dat");
            return path;
        }
    }

    ExFatDriverManager::~ExFatDriverManager() NN_NOEXCEPT
    {
        if (m_IsInitialized)
        {
            fs::Unmount(ExFatDriverMountName);
        }
    }

    Result ExFatDriverManager::Initialize() NN_NOEXCEPT
    {
        bool needsCreateFile = false;

        fs::DisableAutoSaveDataCreation();
        NN_RESULT_TRY(fs::MountSystemSaveData(ExFatDriverMountName, ExFatDriverSaveDataId))
            NN_RESULT_CATCH(fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(fs::CreateSystemSaveData(ExFatDriverSaveDataId, ExFatDriverManagerSaveDataSize, ExFatDriverManagerSaveDataJournalSize, ExFatDriverManagerSaveDataFlags));
                NN_RESULT_DO(fs::MountSystemSaveData(ExFatDriverMountName, ExFatDriverSaveDataId));
                needsCreateFile = true;
            }
        NN_RESULT_END_TRY
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_IsInitialized)
            {
                fs::Unmount(ExFatDriverMountName);
            }
        };

        if (needsCreateFile)
        {
            NN_RESULT_DO(fs::CreateFile(GetSaveDataPath(), sizeof(SaveData)));
            SaveData saveData = {};
            m_SaveData = saveData;
            NN_RESULT_DO(Flush());
        }
        else
        {
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, GetSaveDataPath(), fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            NN_RESULT_DO(fs::ReadFile(file, 0, &m_SaveData, sizeof(m_SaveData)));
        }

        NN_RESULT_TRY(IsExFatInstalled(&m_IsExFatInstalled))
            NN_RESULT_CATCH_ALL
            {
                NN_DETAIL_NS_TRACE("[ExFatDriverManager] IsExFatInstalled failed %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
                m_IsExFatInstalled = false;
            }
        NN_RESULT_END_TRY

        m_IsInitialized = true;

        NN_RESULT_SUCCESS;
    }

    Result ExFatDriverManager::NotifyDriverRequired() NN_NOEXCEPT
    {
        if (m_SaveData.isDriverRequired)
        {
            NN_RESULT_SUCCESS;
        }

        m_SaveData.isDriverRequired = true;
        NN_RESULT_DO(Flush());

        NN_RESULT_SUCCESS;
    }

    Result ExFatDriverManager::NotifyDriverDownloaded() NN_NOEXCEPT
    {
        if (m_SaveData.isDriverDownloadedAtLeastOnce)
        {
            NN_RESULT_SUCCESS;
        }

        m_SaveData.isDriverDownloadedAtLeastOnce = true;
        NN_RESULT_DO(Flush());

        NN_RESULT_SUCCESS;
    }

    Result ExFatDriverManager::Clear() NN_NOEXCEPT
    {
        SaveData saveData = {};
        m_SaveData = saveData;
        NN_RESULT_DO(Flush());
        NN_RESULT_SUCCESS;
    }

    Result ExFatDriverManager::Flush() NN_NOEXCEPT
    {
        {
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, GetSaveDataPath(), fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            NN_RESULT_DO(fs::WriteFile(file, 0, &m_SaveData, sizeof(m_SaveData), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        }
        NN_RESULT_DO(fs::CommitSaveData(ExFatDriverMountName));

        NN_RESULT_SUCCESS;
    }

    Result ExFatDriverManager::IsExFatInstalled(bool* outValue) NN_NOEXCEPT
    {
        *outValue = false;

        // 4.0NUP 時点では最適化のために、BootImagePackage の Attribute のみ確認する
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuiltInSystem));

        const int KeyCount = 16;
        ncm::ContentMetaKey keys[KeyCount];
        auto count = db.ListContentMeta(keys, KeyCount, ncm::ContentMetaType::BootImagePackage);

        for (int i = 0; i < count.listed; ++i)
        {
            Bit8 attributes;
            NN_RESULT_DO(db.GetAttributes(&attributes, keys[i]));
            if (attributes & ncm::ContentMetaAttribute_IncludesExFatDriver)
            {
                *outValue = true;
                break;
            }
        }

        NN_RESULT_SUCCESS;
    }
}}}
