﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_Result.private.h>
#include <nn/ns/srv/detail/json/ns_RapidJsonApi.h>
#include <nn/ns/srv/detail/json/ns_RapidJsonInputStream.h>
#include <nn/ns/srv/detail/json/ns_JsonAdaptor.h>
#include <nn/ns/srv/detail/json/ns_ResultForJson.h>
#include <nn/ns/srv/ns_DownloadTask.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include <rapidjson/encodedstream.h>

#include "ns_DownloadTaskListParser.h"
#include "ns_JsonUtil.h"
#include "ns_StringUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace
    {
        Result MakeContentMetaTypeFromString(ncm::ContentMetaType* outValue, const char* str) NN_NOEXCEPT
        {
            if (strcmp(str, "Application") == 0) { *outValue = ncm::ContentMetaType::Application; NN_RESULT_SUCCESS; }
            if (strcmp(str, "Patch") == 0) { *outValue = ncm::ContentMetaType::Patch; NN_RESULT_SUCCESS; }
            if (strcmp(str, "AddOnContent") == 0) { *outValue = ncm::ContentMetaType::AddOnContent; NN_RESULT_SUCCESS; }
            if (strcmp(str, "Delta") == 0) { *outValue = ncm::ContentMetaType::Delta; NN_RESULT_SUCCESS; }

            NN_RESULT_THROW(ResultDownloadTaskListUnknownContentMetaType());
        }

        const int MaxPathBufferLength = 48;
        const int MaxNodeDepth = 8;

        template <int NodeDepthMax, int PathLengthMax>
        class DownloadTaskListAdaptorBase
        {
            NN_STATIC_ASSERT(NodeDepthMax >= MaxNodeDepth && PathLengthMax >= MaxPathBufferLength);
            NN_DISALLOW_COPY(DownloadTaskListAdaptorBase);

        public:
            typedef detail::json::JsonPath<NodeDepthMax, PathLengthMax> JsonPathType;

        public:
            explicit DownloadTaskListAdaptorBase() NN_NOEXCEPT {}

            Result Adapt(int32_t) const NN_NOEXCEPT { NN_RESULT_SUCCESS; }

            void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {}
            void Update(const JsonPathType&, const char*, int) NN_NOEXCEPT {}

            void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
            void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
            void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}
            void Update(const JsonPathType&, double) NN_NOEXCEPT {}
            void Update(const JsonPathType&, detail::json::Node, bool) NN_NOEXCEPT {}
        };

        class LastError
        {
        public:
            LastError() NN_NOEXCEPT : m_LastError(nn::ResultSuccess())
            {}
            Result Update(Result result) NN_NOEXCEPT
            {
                if (result.IsFailure())
                {
                    m_LastError = result;
                }
                return m_LastError;
            }

            Result GetResult() NN_NOEXCEPT
            {
                return m_LastError;
            }
        private:
            Result m_LastError;
        };


        // DTL の tasks[n].titles[m] の配列に含まれる含まれるエントリのアダプタ
        // 下記のようなオブジェクトをパースして ContentMetaKey にする
        // {
        //     "id": "0100cb3000063001",
        //     "version" : 0,
        //     "type" : "AddOnContent"
        // }
        class DownloadTaskTitleAdaptor :
            public detail::json::ExtensibleJsonAdaptorBase<DownloadTaskListAdaptorBase<MaxNodeDepth, MaxPathBufferLength>>
        {
        public:
            DownloadTaskTitleAdaptor() NN_NOEXCEPT :
                m_TaskIndex(0), m_TitleIndex(0), m_IdFound(false), m_VersionFound(false), m_TypeFound(false)
            {}

            virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_SUCCESS;
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE
            {
                UpdateLookupPathForVersion();
                if (jsonPath.Match(m_LookupPathBuffer))
                {
                    m_Entry.version = static_cast<uint32_t>(value);
                    m_VersionFound = true;
                    return true;
                }

                return true;
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_UNUSED(valueLength);

                UpdateLookupPathForId();
                if (jsonPath.Match(m_LookupPathBuffer))
                {
                    if (m_LastError.Update(JsonParser::MakeBit64FromString(&m_Entry.id, value)).IsFailure())
                    {
                        return false;
                    }
                    m_IdFound = true;
                    return true;
                }

                UpdateLookupPathForType();
                if (jsonPath.Match(m_LookupPathBuffer))
                {
                    if (m_LastError.Update(MakeContentMetaTypeFromString(&m_Entry.type, value)).IsFailure())
                    {
                        return false;
                    }
                    m_TypeFound = true;
                    return true;
                }

                return true;
            }

            virtual bool UpdateImpl(const JsonPathType&, detail::json::Node, bool) NN_NOEXCEPT NN_OVERRIDE
            {
                return true;
            }

            Result Validate() NN_NOEXCEPT
            {
                NN_RESULT_DO(GetLastError());

                if (!m_IdFound || !m_VersionFound || !m_TypeFound) {
                    NN_RESULT_THROW(nn::ns::ResultDownloadTaskListInvalidTitleObject());
                }
                NN_RESULT_SUCCESS;
            }

            void UpdateIndex(int32_t taskIndex, int32_t titleIndex) NN_NOEXCEPT
            {
                m_TaskIndex = taskIndex;
                m_TitleIndex = titleIndex;
                m_IdFound = false;
                m_VersionFound = false;
                m_TypeFound = false;
            }

            const ncm::ContentMetaKey& GetEntry() NN_NOEXCEPT
            {
                return m_Entry;
            }

            Result GetLastError() NN_NOEXCEPT
            {
                return m_LastError.GetResult();
            }

        private:
            char m_LookupPathBuffer[MaxPathBufferLength];
            int m_TaskIndex;
            int m_TitleIndex;
            ncm::ContentMetaKey m_Entry;

            bool m_IdFound;
            bool m_VersionFound;
            bool m_TypeFound;
            LastError m_LastError;

        private:
            void UpdateLookupPathForId() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.tasks[%d].titles[%d].id", m_TaskIndex, m_TitleIndex);
            }
            void UpdateLookupPathForType() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.tasks[%d].titles[%d].type", m_TaskIndex, m_TitleIndex);
            }
            void UpdateLookupPathForVersion() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.tasks[%d].titles[%d].version", m_TaskIndex, m_TitleIndex);
            }

        };


        // DTL の tasks[n] に含まれるオブジェクトのアダプタ
        //
        // 下記のようなオブジェクトをパースして DownloadTask にマップする。
        // ただし、titles の配列に含まれるオブジェクトは DownloadTaskTitleAdaptor でパースする。
        //    {
        //      "id": "7c3714a1-4e0a-449c-8482-09c353b5b534",
        //      "owner_application": "0100cb3000062000",
        //      "titles": [
        //           ...
        //      ],
        //      "config": 1
        //    }
        class DownloadTaskAdaptor :
            public detail::json::ExtensibleJsonAdaptorBase<DownloadTaskListAdaptorBase<MaxNodeDepth, MaxPathBufferLength>>
        {
        public:
            NN_IMPLICIT DownloadTaskAdaptor(DownloadTask* downloadTask) NN_NOEXCEPT :
                m_TaskIndex(0), m_TitleIndex(0), m_Entry(downloadTask), m_IdFound(false), m_OwnerIdFound(false), m_ConfigFound(false), m_TitlesFound(false)
            {
            }

            virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_SUCCESS;
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE
            {
                UpdateLookupPathForConfig();
                if (jsonPath.Match(m_LookupPathBuffer))
                {
                    m_Entry->installConfig = static_cast<uint32_t>(value);
                    m_ConfigFound = true;
                    return true;
                }

                return m_TitleAdaptor.UpdateImpl(jsonPath, value);
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_UNUSED(valueLength);
                UpdateLookupPathForId();
                if (jsonPath.Match(m_LookupPathBuffer))
                {
                    // INFO: valueLength には文字列としての長さが, Uuid::StringSize にはバッファサイズが入る。
                    //       valueLength の末尾に NULL 終端分の 1 バイトを足して比較している。
                    if (valueLength + 1 != util::Uuid::StringSize)
                    {
                        m_LastError.Update(ns::ResultDownloadTaskListInvalidUuid());
                        return false;
                    }
                    m_Entry->uuid.FromString(value);
                    m_IdFound = true;
                    return true;
                }

                UpdateLookupPathForOwnerId();
                if (jsonPath.Match(m_LookupPathBuffer))
                {
                    if (m_LastError.Update(JsonParser::MakeBit64FromString(&m_Entry->applicationId.value, value)).IsFailure())
                    {
                        return false;
                    }
                    m_OwnerIdFound = true;
                    return true;
                }

                return m_TitleAdaptor.UpdateImpl(jsonPath, value, valueLength);
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, detail::json::Node node, bool isStart) NN_NOEXCEPT NN_OVERRIDE
            {
                if (node.kind == detail::json::Node::Kind::Object && isStart)
                {
                    UpdateLookupPathForTitle();
                    if (jsonPath.Match(m_LookupPathBuffer))
                    {
                        if (m_TitleIndex >= m_Entry->MaxKeyCount)
                        {
                            m_LastError.Update(ns::ResultDownloadTaskListTooManyTitles());
                            return false;
                        }

                        m_TitleAdaptor.UpdateIndex(m_TaskIndex, m_TitleIndex);
                        return true;
                    }
                }

                if (node.kind == detail::json::Node::Kind::Object && !isStart)
                {
                    UpdateLookupPathForTitle();
                    if (jsonPath.Match(m_LookupPathBuffer))
                    {
                        if (m_TitleAdaptor.Validate().IsFailure())
                        {
                            return false;
                        }

                        m_Entry->keyList[m_TitleIndex] = m_TitleAdaptor.GetEntry();
                        m_Entry->keyCount++;
                        m_TitleIndex++;
                        return true;
                    }
                }

                if (node.kind == detail::json::Node::Kind::Array && isStart)
                {
                    UpdateLookupPathForTitles();
                    if (jsonPath.Match(m_LookupPathBuffer))
                    {
                        m_TitlesFound = true;
                        return true;
                    }
                }

                return m_TitleAdaptor.UpdateImpl(jsonPath, node, isStart);
            }

            Result Validate() NN_NOEXCEPT
            {
                NN_RESULT_DO(GetLastError());

                if (!m_IdFound || !m_TitlesFound || !m_OwnerIdFound)
                {
                    NN_RESULT_THROW(nn::ns::ResultDownloadTaskListInvalidTaskObject());
                }
                NN_RESULT_SUCCESS;
            }

            void UpdateIndex(int32_t index) NN_NOEXCEPT
            {
                m_TaskIndex = index;
                m_IdFound = false;
                m_OwnerIdFound = false;
                m_ConfigFound = false;
                m_TitlesFound = false;

                m_Entry->keyCount = 0;
                m_Entry->installConfig = ncm::InstallConfig_LatestAddOnContentsAuto;
                m_TitleIndex = 0;
            }

            DownloadTask* GetEntry() NN_NOEXCEPT
            {
                return m_Entry;
            }

            Result GetLastError() NN_NOEXCEPT
            {
                NN_RESULT_DO(m_TitleAdaptor.GetLastError());

                return m_LastError.GetResult();
            }

        private:
            char m_LookupPathBuffer[MaxPathBufferLength];
            int m_TaskIndex;
            int m_TitleIndex;
            DownloadTask* m_Entry;
            bool m_IdFound;
            bool m_OwnerIdFound;
            bool m_ConfigFound;
            bool m_TitlesFound;
            DownloadTaskTitleAdaptor m_TitleAdaptor;

            LastError m_LastError;

        private:
            void UpdateLookupPathForId() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.tasks[%d].id", m_TaskIndex);
            }

            void UpdateLookupPathForOwnerId() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.tasks[%d].owner_application", m_TaskIndex);
            }

            void UpdateLookupPathForConfig() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.tasks[%d].config", m_TaskIndex);
            }

            void UpdateLookupPathForTitle() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.tasks[%d].titles[%d]", m_TaskIndex, m_TitleIndex);
            }

            void UpdateLookupPathForTitles() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.tasks[%d].titles[0]", m_TaskIndex);
            }
        };

        // DTL 全体のアダプタ
        // tasks エントリを一つパースするたびに、コンストラクタで設定したコールバックを呼び出す。
        //
        // tasks オブジェクトを DownloadTaskAdaptor を使ってパースする
        template<typename CallbackFunctionType>
        class DownloadTaskListAdaptor :
            public detail::json::ExtensibleJsonAdaptorBase<DownloadTaskListAdaptorBase<MaxNodeDepth, MaxPathBufferLength>>,
            public detail::json::Cancellable
        {
        public:
            explicit DownloadTaskListAdaptor(DownloadTask* downloadTaskBuffer, CallbackFunctionType& callbackFunction) NN_NOEXCEPT :
                m_TaskIndex(0), m_DownloadTaskAdaptor(downloadTaskBuffer), m_CallbackFunction(callbackFunction)
            {
            }

            virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_SUCCESS;
            }

            virtual bool UpdateImpl(const JsonPathType&, bool) NN_NOEXCEPT NN_OVERRIDE
            {
                return false;
            }
            virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE
            {
                return m_DownloadTaskAdaptor.UpdateImpl(jsonPath, value);
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE
            {
                return m_DownloadTaskAdaptor.UpdateImpl(jsonPath, value, valueLength);
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, detail::json::Node node, bool isStart) NN_NOEXCEPT NN_OVERRIDE
            {
                if (node.kind == detail::json::Node::Kind::Object && isStart)
                {
                    UpdateLookupPathForTask();
                    if (jsonPath.Match(m_LookupPathBuffer))
                    {
                        m_DownloadTaskAdaptor.UpdateIndex(m_TaskIndex);
                        return true;
                    }
                }

                if (node.kind == detail::json::Node::Kind::Object && !isStart)
                {
                    UpdateLookupPathForTask();
                    if (jsonPath.Match(m_LookupPathBuffer))
                    {
                        if (m_DownloadTaskAdaptor.Validate().IsFailure())
                        {
                            this->Cancel();
                            return false;
                        }

#if !defined(NN_SDK_BUILD_RELEASE)
                        auto task = m_DownloadTaskAdaptor.GetEntry();
                        NN_DETAIL_NS_TRACE("DownloadTask parsed. OwnerId: %016llx, KeyCount: %d\n", task->applicationId.value, task->keyCount);
#endif
                        if(m_LastError.Update(m_CallbackFunction(m_DownloadTaskAdaptor.GetEntry())).IsFailure())
                        {
                            NN_DETAIL_NS_TRACE("Download task callback failed: %08x\n", m_LastError.GetResult().GetInnerValueForDebug());
                            return false;
                        }

                        m_TaskIndex++;
                        return true;
                    }
                }

                return m_DownloadTaskAdaptor.UpdateImpl(jsonPath, node, isStart);
            }

            Result GetLastError() NN_NOEXCEPT
            {
                NN_RESULT_DO(m_DownloadTaskAdaptor.GetLastError());

                return m_LastError.GetResult();
            }
        private:
            int m_TaskIndex;
            char m_LookupPathBuffer[MaxPathBufferLength];
            LastError m_LastError;
            DownloadTaskAdaptor m_DownloadTaskAdaptor;
            CallbackFunctionType& m_CallbackFunction;

        private:
            void UpdateLookupPathForTask() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.tasks[%d]", m_TaskIndex);
            }

        };


        // SAX に空のハンドラ。JSON 全体のバリデーション用。
        class DownloadTaskListValidationAdaptor :
            public detail::json::ExtensibleJsonAdaptorBase<DownloadTaskListAdaptorBase<MaxNodeDepth, MaxPathBufferLength>>,
            public detail::json::Cancellable
        {
        public:
            explicit DownloadTaskListValidationAdaptor() NN_NOEXCEPT {}
            virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE { NN_RESULT_SUCCESS; }
            virtual bool UpdateImpl(const JsonPathType&, bool) NN_NOEXCEPT NN_OVERRIDE { return true; }
            virtual bool UpdateImpl(const JsonPathType&, int64_t) NN_NOEXCEPT NN_OVERRIDE { return true; }
            virtual bool UpdateImpl(const JsonPathType&, const char*, int) NN_NOEXCEPT NN_OVERRIDE { return true; }
            virtual bool UpdateImpl(const JsonPathType&, detail::json::Node, bool) NN_NOEXCEPT NN_OVERRIDE { return true; }
        };

    } // namespace

    template<typename InputStreamType>
    Result DownloadTaskListParser::ParseTask(DownloadTask* downloadTaskBuffer, InputStreamType& inputStream, DownloadTaskCallback& callbackFunction) NN_NOEXCEPT
    {
        DownloadTaskListAdaptor<DownloadTaskCallback> adaptor(downloadTaskBuffer, callbackFunction);

        NN_RESULT_TRY(detail::json::ImportJsonByRapidJson(adaptor, inputStream, &adaptor))
            NN_RESULT_CATCH_CONVERT(nn::ns::ResultCanceled, adaptor.GetLastError())
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    template<typename InputStreamType>
    Result DownloadTaskListParser::Validate(InputStreamType& inputStream) NN_NOEXCEPT
    {
        DownloadTaskListValidationAdaptor adaptor;
        NN_RESULT_DO(detail::json::ImportJsonByRapidJson(adaptor, inputStream, &adaptor));
        NN_RESULT_SUCCESS;
    }


    template Result DownloadTaskListParser::ParseTask<detail::json::AsyncDataInputStreamForRapidJson<>>(DownloadTask* downloadTaskBuffer, detail::json::AsyncDataInputStreamForRapidJson<>& inputStream, DownloadTaskCallback& callbackFunction);
    template Result DownloadTaskListParser::ParseTask<detail::json::EncodedMemoryInputStreamForRapidJson>(DownloadTask* downloadTaskBuffer, detail::json::EncodedMemoryInputStreamForRapidJson& inputStream, DownloadTaskCallback& callbackFunction);

    template Result DownloadTaskListParser::Validate<detail::json::AsyncDataInputStreamForRapidJson<>>(detail::json::AsyncDataInputStreamForRapidJson<>& inputStream);
    template Result DownloadTaskListParser::Validate<detail::json::EncodedMemoryInputStreamForRapidJson>(detail::json::EncodedMemoryInputStreamForRapidJson& inputStream);

}}}
