﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/ns/detail/ns_IAsync.sfdl.h>
#include <nn/ns/srv/ns_DownloadTaskListManager.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_ScopeExit.h>

#include "ns_AsyncEnsureDownloadTaskImpl.h"
#include "ns_SystemSaveDataUtil.h"
#include "ns_StringUtil.h"
#include "ns_TaskUtil.h"
#include "ns_InstallUtil.h"
#include "ns_DownloadTaskListParser.h"

#include <nn/ns/srv/detail/json/ns_RapidJsonInputStream.h>


namespace nn { namespace ns { namespace srv {
    namespace {
        const Bit64     DownloadTaskListSaveDataId              = 0x8000000000000046;
        const int64_t   DownloadTaskListSaveDataSize            = 0x18000; // 64KB + 32KB
        const int64_t   DownloadTaskListSaveDataJournalSize     = 0x18000; // 64KB + 32KB
        const int       DownloadTaskListSaveDataFlags           = 0; // TORIAEZU: fs チームから指示があったら修正する

        typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

        const char* MountName = "dtlman";

        typedef kvdb::BoundedString<32> Path;

        Path MakeTaskStatusListPath() NN_NOEXCEPT
        {
            Path path(MountName);
            path.Append(":/tsl");

            return path;
        }

        Result FilterKeys(DownloadTask* task, ApplicationRecordDatabase* db) NN_NOEXCEPT
        {
            // ContentMetaType::Patch だけが特殊化されているという前提条件での最適化
            bool canCreatePatch;
            NN_RESULT_DO(CanCreateNetworkInstallTask(&canCreatePatch, task->applicationId, ncm::ContentMetaType::Patch, true));

            for (int i = task->keyCount - 1; i >= 0; --i)
            {
                bool skip = false;
                if (task->keyList[i].type == ncm::ContentMetaType::Patch && !canCreatePatch)
                {
                    skip = true;
                    NN_DETAIL_NS_TRACE("[DownloadTaskListManager] Patch task exists, skip key %016llx, %u\n", task->keyList[i].id, task->keyList[i].version);
                }
                else
                {
                    bool hasRecord;
                    bool installed;
                    NN_RESULT_DO(db->HasContentMetaKey(&hasRecord, &installed, task->keyList[i], task->applicationId));
                    if (hasRecord && installed)
                    {
                        NN_DETAIL_NS_TRACE("[DownloadTaskListManager] Already installed, skip key %016llx, %u\n", task->keyList[i].id, task->keyList[i].version);
                        skip = true;
                    }
                }

                if (skip)
                {
                    // そのタスクは作れないので、memmove する
                    int moveSize = task->keyCount - i - 1;
                    std::memmove(&(task->keyList[i]), &(task->keyList[i + 1]), sizeof(ncm::ContentMetaKey) * moveSize);
                    task->keyCount--;
                }
            }
            NN_RESULT_SUCCESS;
        }
        Result ResetTaskAttribute(const nim::NetworkInstallTaskId& taskId) NN_NOEXCEPT
        {
            nim::NetworkInstallTaskInfo info;
            NN_RESULT_DO(nim::GetNetworkInstallTaskInfo(&info, taskId));

            // Add するときに、再開可能なタスクは Runnable に戻す
            if (IsResumable(info.attribute) || ToState(info.attribute) == ApplicationDownloadState::Finished)
            {
                NN_RESULT_DO(nim::SetNetworkInstallTaskAttribute(taskId, static_cast<Bit64>(ApplicationDownloadState::Runnable)));
            }

            NN_RESULT_SUCCESS;
        }
    }

    DownloadTaskListManager::~DownloadTaskListManager() NN_NOEXCEPT
    {
        if (m_Record)
        {
            Finalize();
        }
    }

    Result DownloadTaskListManager::Initialize(ApplicationRecordDatabase* record) NN_NOEXCEPT
    {
        fs::DisableAutoSaveDataCreation();
        NN_RESULT_TRY(fs::MountSystemSaveData(MountName, DownloadTaskListSaveDataId))
            NN_RESULT_CATCH(fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(fs::CreateSystemSaveData(DownloadTaskListSaveDataId, DownloadTaskListSaveDataSize, DownloadTaskListSaveDataJournalSize, DownloadTaskListSaveDataFlags));
                NN_RESULT_DO(fs::MountSystemSaveData(MountName, DownloadTaskListSaveDataId));
            }
        NN_RESULT_END_TRY

        SystemSaveDataSignature signature("DTLMAN", 0);

        bool hasValidSignature;
        NN_RESULT_DO(signature.IsValidOn(&hasValidSignature, MountName));
        if (!hasValidSignature)
        {
            NN_DETAIL_NS_TRACE("[DownloadTaskListManager] No valid signature. Create.\n");

            NN_RESULT_DO(signature.WriteTo(MountName));
            NN_RESULT_DO(fs::CreateFile(MakeTaskStatusListPath(), sizeof(m_TaskStatusList)));
            NN_RESULT_DO(Clear());
        }
        else
        {
            NN_RESULT_DO(Load());
        }

        m_Record = record;

        NN_RESULT_SUCCESS;
    }

    void DownloadTaskListManager::Finalize() NN_NOEXCEPT
    {
        fs::Unmount(MountName);
    }

    Result DownloadTaskListManager::Push(const void* data, size_t size) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        return PushImpl(reinterpret_cast<const char*>(data), size, m_Buffer, sizeof(m_Buffer));
    }

    Result DownloadTaskListManager::Push(nim::AsyncData* asyncDownloadTask) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        NN_RESULT_DO(asyncDownloadTask->Get());

        detail::json::AsyncDataInputStreamForRapidJson<> inputStreamForCheck(asyncDownloadTask, m_Buffer, sizeof(m_Buffer));
        detail::json::AsyncDataInputStreamForRapidJson<> inputStreamForParse(asyncDownloadTask, m_Buffer, sizeof(m_Buffer));
        NN_RESULT_DO(PushImpl(inputStreamForCheck, inputStreamForParse));

        NN_RESULT_SUCCESS;
    }

    Result DownloadTaskListManager::PushImpl(const char* jsonBuffer, size_t jsonBufferSize, char* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        detail::json::MemoryInputStreamForRapidJson inputStreamForCheck(buffer, bufferSize);
        inputStreamForCheck.Set(jsonBuffer, jsonBufferSize);
        detail::json::EncodedMemoryInputStreamForRapidJson encodedInputStreamForCheck(inputStreamForCheck);

        detail::json::MemoryInputStreamForRapidJson inputStreamForParse(buffer, bufferSize);
        inputStreamForParse.Set(jsonBuffer, jsonBufferSize);
        detail::json::EncodedMemoryInputStreamForRapidJson encodedInputStreamForParse(inputStreamForParse);

        return PushImpl(encodedInputStreamForCheck, encodedInputStreamForParse);
    }

    template<typename InputStreamType>
    Result DownloadTaskListManager::PushImpl(InputStreamType& inputStreamForCheck, InputStreamType& inputStreamForParse) NN_NOEXCEPT
    {
        DownloadTaskCallback parseTaskCallback = [this](DownloadTask* downloadTask) -> Result
        {
            NN_RESULT_DO(this->CreateInstallTask(downloadTask));
            NN_RESULT_SUCCESS;
        };

        DownloadTaskListParser parser;

        NN_RESULT_DO(parser.Validate(inputStreamForCheck));

        MarkCleanupAll();
        NN_RESULT_DO(parser.ParseTask(&m_DownloadTask, inputStreamForParse, parseTaskCallback));
        Cleanup();
        NN_RESULT_DO(Flush());

        NN_RESULT_SUCCESS;
    }


    Result DownloadTaskListManager::Clear() NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        std::memset(m_TaskStatusList, 0, sizeof(m_TaskStatusList));
        NN_RESULT_DO(Flush());
        NN_RESULT_SUCCESS;
    }

    int DownloadTaskListManager::ListDownloadTaskStatus(DownloadTaskStatus outList[], int count) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        int outCount = 0;
        for (int i = 0; i < MaxTaskStatusCount && outCount < count; i++)
        {
            if (m_TaskStatusList[i].IsValid())
            {
                outList[outCount] = m_TaskStatusList[i];
                outCount++;
            }
        }

        return outCount;
    }

    Result DownloadTaskListManager::RequestEnsureDownloadTask(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, RequestServer::ManagedStop&& stopper) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncEnsureDownloadTaskImpl>(this, std::move(stopper));
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());
        NN_RESULT_TRY(emplacedRef.GetImpl().Initialize())
            NN_RESULT_CATCH(nim::ResultDeviceAccountNotRegistered)
            {
                NN_DETAIL_NS_TRACE("[DownloadTaskListManager] Ignore to ensure download task since device account not registered\n");

                emplacedRef.Reset();
                auto nullRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, NullAsyncResult>();
                NN_RESULT_THROW_UNLESS(nullRef, ResultOutOfMaxRunningTask());
                *outHandle = sf::NativeHandle(nullRef.GetImpl().GetEvent().GetReadableHandle(), false);
                *outAsync = nullRef;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;
        NN_RESULT_SUCCESS;
    }

    Result DownloadTaskListManager::RequestDownloadTaskListData(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncValue>> outAsync, RequestServer::ManagedStop&& stopper) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncValue, AsyncDownloadTaskListDataImpl>(std::move(stopper));
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize());

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;
        NN_RESULT_SUCCESS;
    }

    bool DownloadTaskListManager::Has(util::Uuid uuid) NN_NOEXCEPT
    {
        for (auto& taskStatus : m_TaskStatusList)
        {
            if (taskStatus.uuid == uuid)
            {
                return true;
            }
        }

        return false;
    }

    void DownloadTaskListManager::Add(const DownloadTaskStatus& status) NN_NOEXCEPT
    {
        for (auto& taskStatus : m_TaskStatusList)
        {
            if (taskStatus.IsEmpty())
            {
                taskStatus = status;
                return;
            }
        }

        NN_DETAIL_NS_TRACE("[DownloadTaskListManager] No empty task status for Add()\n");
    }

    void DownloadTaskListManager::MarkCleanupAll() NN_NOEXCEPT
    {
        for (auto& taskStatus : m_TaskStatusList)
        {
            if (taskStatus.IsValid())
            {
                taskStatus.needsCleanup = true;
            }
        }
    }

    void DownloadTaskListManager::UnmarkCleanup(util::Uuid uuid) NN_NOEXCEPT
    {
        for (auto& taskStatus : m_TaskStatusList)
        {
            if (taskStatus.uuid == uuid)
            {
                taskStatus.needsCleanup = false;
            }
        }
    }

    void DownloadTaskListManager::Cleanup() NN_NOEXCEPT
    {
        for (auto& taskStatus : m_TaskStatusList)
        {
            if (taskStatus.needsCleanup)
            {
                DownloadTaskStatus status = {};
                taskStatus = status;
            }
        }
    }

    Result DownloadTaskListManager::CreateInstallTask(DownloadTask* task) NN_NOEXCEPT
    {
        if (Has(task->uuid))
        {
            UnmarkCleanup(task->uuid);
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_DO(FilterKeys(task, m_Record));
        if (task->keyCount == 0)
        {
            DownloadTaskStatus status = {};
            status.uuid = task->uuid;
            status.applicationId = task->applicationId;
            status.detail = DownloadTaskStatusDetail::AlreadyExists;
            Add(status);

            NN_RESULT_SUCCESS;
        }

        nim::NetworkInstallTaskId taskId;
        auto count = nim::ListApplicationNetworkInstallTask(&taskId, 1, task->applicationId);
        Result taskResult;
        DownloadTaskStatusDetail detail;
        if (count > 0)
        {
            NN_DETAIL_NS_TRACE("[DownloadTaskListManager] Add to task: appId 0x%016llx keyCount %d\n", task->applicationId.value, task->keyCount);
            detail = DownloadTaskStatusDetail::Added;
            taskResult = ResetTaskAttribute(taskId);
            if (taskResult.IsSuccess())
            {
                taskResult = nim::AddNetworkInstallTaskContentMeta(task->keyList, task->keyCount, taskId);
            }

        }
        else
        {
            NN_DETAIL_NS_TRACE("[DownloadTaskListManager] Create task: appId 0x%016llx keyCount %d\n", task->applicationId.value, task->keyCount);
            detail = DownloadTaskStatusDetail::Created;
            taskResult = nim::CreateNetworkInstallTask(&taskId, task->applicationId, task->keyList, task->keyCount, ncm::StorageId::Any, task->installConfig);
        }
        if (taskResult.IsSuccess())
        {
            for (int j = 0; j < task->keyCount; j++)
            {
                auto& key = task->keyList[j];

                NN_DETAIL_NS_TRACE("[DownloadTaskListManager]     key id 0x%016llx version %u type %u\n", key.id, key.version, key.type);

                if (key.type == ncm::ContentMetaType::Application)
                {
#if !defined NN_BUILD_CONFIG_OS_WIN
                    pctl::NotifyApplicationDownloadStarted(task->applicationId);
#endif
                }

                // ダウンロード開始時に作成する記録は便宜的に BuiltInUser とする
                // ダウンロード完了時に実際にダウンロードしたストレージの記録に上書きされる
                ncm::StorageContentMetaKey storageKey = { key, ncm::StorageId::BuiltInUser };
                m_PushKeyBuffer[j] = storageKey;
            }
            if (!(m_Record->Has(task->applicationId)))
            {
                // 新規追加の時のみ、記録を push する
                NN_RESULT_DO(m_Record->Push(task->applicationId, ApplicationEvent::DownloadStarted, m_PushKeyBuffer, task->keyCount, false));
            }
            else
            {
                // それ以外の場合は、記録の順番を入れ替えるために空 push をする
                NN_RESULT_DO(m_Record->Push(task->applicationId, ApplicationEvent::DownloadStarted, nullptr, 0, false));
            }
        }
        else
        {
            NN_DETAIL_NS_TRACE("[DownloadTaskListManager]     Failed to create a task 0x%08x\n", taskResult.GetInnerValueForDebug());
        }
        {
            DownloadTaskStatus status = {};
            status.uuid = task->uuid;
            status.applicationId = task->applicationId;
            status.detail = taskResult.IsSuccess() ? detail : DownloadTaskStatusDetail::Failed;
            status.SetResult(taskResult);

            Add(status);
        }

        NN_RESULT_SUCCESS;
    }

    Result DownloadTaskListManager::Load() NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, MakeTaskStatusListPath(), fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        NN_RESULT_DO(fs::ReadFile(file, 0, m_TaskStatusList, sizeof(m_TaskStatusList)));
        NN_RESULT_SUCCESS;
    }

    Result DownloadTaskListManager::Flush() NN_NOEXCEPT
    {
        {
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, MakeTaskStatusListPath(), fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
            NN_RESULT_DO(fs::WriteFile(file, 0, m_TaskStatusList, sizeof(m_TaskStatusList), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        }
        NN_RESULT_DO(fs::CommitSaveData(MountName));
        NN_RESULT_SUCCESS;
    }
}}}
