﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>

#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/ns/detail/ns_Log.h>
#include "ns_LogUtil.h"
#include "ns_CleanupUtil.h"

namespace nn { namespace ns { namespace srv {

namespace
{
const int MaxContents = 256;
const int MaxSaveDataInfoNum = 128;
const int MaxUsingCacheStorageNum = 2048;

Result CleanupOrphanCacheStorageImpl(fs::SaveDataSpaceId spaceId, ApplicationRecordDatabase* recordDb, nn::fs::SaveDataInfo* pWorkSaveDataInfo, int saveDataInfoNum) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        nn::util::optional<nn::fs::SaveDataInfo> info;
        NN_RESULT_DO(nn::fssystem::FindSaveData(&info, pWorkSaveDataInfo, saveDataInfoNum, spaceId, [&recordDb](const nn::fs::SaveDataInfo& i)
        {
            return i.saveDataType == nn::fs::SaveDataType::Cache && !recordDb->Has(i.applicationId);
        }));
        if(info == nn::util::nullopt)
        {
            break;
        }
        NN_RESULT_DO(fs::DeleteSaveData(spaceId, info->saveDataId));
    }
    NN_RESULT_SUCCESS;
}

bool IsCheckedApplicationId(nn::ncm::ApplicationId* pCheckedApplicationIdList, int16_t lintNum, nn::ncm::ApplicationId applicationId)
{
    for(int16_t i = 0; i < lintNum; i++)
    {
        if(applicationId == pCheckedApplicationIdList[i])
        {
            return true;
        }
    }
    return false;
}

// SD と NAND 両方に同一アプリのキャッシュストレージが存在した場合、 NAND 側のキャッシュストレージを削除する
Result CleanupDuplicateCacheStorageImpl(LockedBufferManager* lockedBuffer) NN_NOEXCEPT
{
    constexpr auto SaveDataInfoListSize = sizeof(fs::SaveDataInfo) * MaxSaveDataInfoNum;
    constexpr auto CheckedApplicationIdListSize = sizeof(ncm::ApplicationId) * MaxUsingCacheStorageNum;

    auto buffer = lockedBuffer->Allocate(SaveDataInfoListSize + CheckedApplicationIdListSize);
    auto ptr = buffer.Get<Bit8*>();
    auto pWorkSaveDataInfo = reinterpret_cast<fs::SaveDataInfo*>(ptr);
    auto checkedApplicationId = reinterpret_cast<ncm::ApplicationId*>(ptr + SaveDataInfoListSize);
    int16_t checkedApplicationNum = 0;

    std::unique_ptr<fs::SaveDataIterator> iter;
    NN_RESULT_DO(fs::OpenSaveDataIterator(&iter, fs::SaveDataSpaceId::SdUser));
    while (NN_STATIC_CONDITION(true))
    {
        int64_t count;
        NN_RESULT_DO(iter->ReadSaveDataInfo(&count, pWorkSaveDataInfo, MaxSaveDataInfoNum));
        if (count == 0)
        {
            break;
        }
        for(int64_t i = 0; i < count; i++)
        {
            if (pWorkSaveDataInfo[i].saveDataType == fs::SaveDataType::Cache)
            {
                if(!IsCheckedApplicationId(checkedApplicationId, checkedApplicationNum, pWorkSaveDataInfo[i].applicationId))
                {
                    checkedApplicationId[checkedApplicationNum++] = pWorkSaveDataInfo[i].applicationId;
                }
            }
        }
    }
    while (NN_STATIC_CONDITION(true))
    {
        nn::util::optional<nn::fs::SaveDataInfo> info;
        NN_RESULT_DO(nn::fssystem::FindSaveData(&info, pWorkSaveDataInfo, MaxSaveDataInfoNum, fs::SaveDataSpaceId::User, [&](const nn::fs::SaveDataInfo& i)
        {
            return i.saveDataType == nn::fs::SaveDataType::Cache && IsCheckedApplicationId(checkedApplicationId, checkedApplicationNum, i.applicationId);
        }));
        if(info == nn::util::nullopt)
        {
            break;
        }
        NN_RESULT_DO(fs::DeleteSaveData(fs::SaveDataSpaceId::User, info->saveDataId));
    }

    NN_RESULT_SUCCESS;
}
}

Result CleanupOrphanContents(ncm::StorageId storageId) NN_NOEXCEPT
{
    static NonRecursiveMutex s_Mutex;
    static ncm::ContentId s_ContentList[MaxContents];
    static bool s_OrphanList[MaxContents];
    std::lock_guard<NonRecursiveMutex> guard(s_Mutex);

    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, storageId));

    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));

    int offset = 0;
    while (NN_STATIC_CONDITION(true))
    {
        int count;
        NN_RESULT_DO(storage.ListContentId(&count, s_ContentList, MaxContents, offset));
        if (count == 0)
        {
            break;
        }

        NN_RESULT_DO(db.LookupOrphanContent(s_OrphanList, s_ContentList, count));
        for (int i = 0; i < count; i++)
        {
            if (s_OrphanList[i])
            {
                NN_DETAIL_NS_TRACE("[CleanupUtil] Found orphan content %s\n", ncm::GetContentIdString(s_ContentList[i]).data);;
                NN_RESULT_DO(storage.Delete(s_ContentList[i]));
            }
        }

        offset += count;
    }
    NN_RESULT_SUCCESS;
}

Result CleanupAllNetworkInstallTask() NN_NOEXCEPT
{
    nim::NetworkInstallTaskId idList[64];
    auto count = nim::ListNetworkInstallTask(idList, static_cast<int>(NN_ARRAY_SIZE(idList)));
    for (int i = 0; i < count; i++)
    {
        auto result = nim::DestroyNetworkInstallTask(idList[i]);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[CleanupUtil] DestroyNetworkInstallTask failed: %08x\n", result.GetInnerValueForDebug());
        }
    }

    NN_RESULT_SUCCESS;
}

Result CleanupAllApplyDeltaTask() NN_NOEXCEPT
{
    nim::ApplyDeltaTaskId idList[64];
    auto count = nim::ListApplyDeltaTask(idList, static_cast<int>(NN_ARRAY_SIZE(idList)));
    for (int i = 0; i < count; i++)
    {
        auto result = nim::DestroyApplyDeltaTask(idList[i]);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[CleanupUtil] DestroyApplyDeltaTask failed: %08x\n", result.GetInnerValueForDebug());
        }
    }

    NN_RESULT_SUCCESS;
}

Result CleanupApplyDeltaTask(ncm::StorageId storageId) NN_NOEXCEPT
{
    nim::ApplyDeltaTaskId idList[64];
    auto count = nim::ListApplyDeltaTask(idList, static_cast<int>(NN_ARRAY_SIZE(idList)));
    for (int i = 0; i < count; i++)
    {
        // ApplyDeltaTask の ContentMeta はすべて同じストレージにあるので、先頭を取得してそのストレージを見る
        int contentMetaCount;
        ncm::StorageContentMetaKey key;
        bool success = true;
        NN_RESULT_TRY(nim::ListApplyDeltaTaskContentMeta(&contentMetaCount, &key, 1, 0, idList[i]))
            NN_RESULT_CATCH_ALL
            {
                success = false;
            }
        NN_RESULT_END_TRY
        if (!success || (contentMetaCount == 1 && key.storageId == storageId))
        {
            auto result = nim::DestroyApplyDeltaTask(idList[i]);
            if (result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[CleanupUtil] DestroyApplyDeltaTask failed: %08x\n", result.GetInnerValueForDebug());
            }
        }
    }

    NN_RESULT_SUCCESS;
}

Result CleanupFragments(ncm::StorageId storageId) NN_NOEXCEPT
{
    ncm::ContentMetaDatabase db;
    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, storageId));

    ncm::ContentManagerAccessor accessor(&db, &storage);

    for(;;)
    {
        const int CountKey = 16;
        ncm::ContentMetaKey keys[CountKey];

        auto count = db.ListContentMeta(keys, CountKey, ncm::ContentMetaType::Patch, ncm::ContentMetaDatabase::AnyApplicationId, 0x0, 0xffffffffffffffff, ncm::ContentInstallType::FragmentOnly);

        for (int i = 0; i < count.listed; ++i)
        {
            NN_RESULT_DO(accessor.DeleteRedundant(keys[i], nullptr));
        }
        if (count.listed == count.total)
        {
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

Result CleanupAllPlaceHolderAndFragmentsIfNoTask(IntegratedContentManager* integrated) NN_NOEXCEPT
{
    nim::NetworkInstallTaskId installId;
    nim::ApplyDeltaTaskId applyId;
    if (nim::ListNetworkInstallTask(&installId, 1) == 0 && nim::ListApplyDeltaTask(&applyId, 1) == 0)
    {
        NN_RESULT_DO(integrated->CleanupAllPlaceHolder());
        NN_RESULT_DO(integrated->CleanupFragments());
    }

    NN_RESULT_SUCCESS;
}

Result CleanupCacheStorage(ApplicationRecordDatabase* recordDb, IntegratedContentManager* integrated, LockedBufferManager* lockedBuffer, bool isCleanupOrphanCacheStorageEnabledForDebug) NN_NOEXCEPT
{
    // そもそも SD カードが登録されていなければクリーンアップは走らない
    if (integrated->IsRegisteredStorage(ncm::StorageId::SdCard))
    {
        if(isCleanupOrphanCacheStorageEnabledForDebug)
        {
            static fs::SaveDataInfo s_SaveDataInfo[MaxSaveDataInfoNum];
            static NonRecursiveMutex s_Mutex;
            std::lock_guard<NonRecursiveMutex> guard(s_Mutex);
            NN_NS_TRACE_RESULT_IF_FAILED(CleanupOrphanCacheStorageImpl(nn::fs::SaveDataSpaceId::SdUser, recordDb, s_SaveDataInfo, MaxSaveDataInfoNum),
                                        "[ShutdownManager] Failed to cleanup orphan cache storage.\n");
        }
        NN_NS_TRACE_RESULT_IF_FAILED(CleanupDuplicateCacheStorageImpl(lockedBuffer),
                                    "[ShutdownManager] Failed to cleanup duplicate cache storage.\n");
    }
    NN_RESULT_SUCCESS;
}

}}}
