﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_Result.h>
#include <nn/ec/ec_Result.h>
#include <nn/ec/system/ec_DeviceAccountSystemApi.h>
#include <nn/ec/system/ec_DeviceLinkSystemApi.h>
#include <nn/es.h>
#include <nn/fs/fs_Mount.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_GameCardManager.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/os/os_Thread.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/util/util_BitArray.h>
#include <nn/util/util_ScopeExit.h>
#include "ns_AccountUtil.h"
#include "ns_AsyncImpl.h"
#include "ns_AsyncThreadAllocator.h"
#include "ns_SystemUpdateUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        Result ConvertEcResult(Result result) NN_NOEXCEPT
        {
            if (result <= ec::ResultCanceled()) return ResultCanceled();
            return result;
        }

        Result RequestSystemUpdateMeta(nim::AsyncContentMetaKey* outValue) NN_NOEXCEPT
        {
            NN_RESULT_TRY(nim::RequestSystemUpdateMeta(outValue))
                NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxTask, ResultOutOfMaxRunningTask())
                NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxRunningTask, ResultOutOfMaxRunningTask())
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        Result RequestLatestVersion(nim::AsyncLatestKeyList* outValue, const ncm::ApplicationId applicationId) NN_NOEXCEPT
        {
            NN_RESULT_TRY(nim::RequestLatestVersion(outValue, applicationId))
                NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxTask, ResultOutOfMaxRunningTask())
                NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxRunningTask, ResultOutOfMaxRunningTask())
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        Result GetLatestKeyList(int* outValue, nim::AsyncLatestKeyList* async, ncm::ContentMetaKey outList[], int count) NN_NOEXCEPT
        {
            NN_RESULT_TRY(async->Get(outValue, outList, count))
                NN_RESULT_CATCH_CONVERT(nim::ResultSystemUpdateRequiredForContentDelivery, ResultSystemUpdateRequiredForContentDelivery())
                NN_RESULT_CATCH(nim::ResultLatestVersionNotFound)
                {
                    *outValue = 0;
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        void FilterUpdateContentMetaKey(ncm::ContentMetaKey target[], int* outCount, int targetCount, const ncm::ContentMetaKey recordedList[], util::BitArray* forceUpdateList, int recordedListCount) NN_NOEXCEPT
        {
            auto targetBegin = &target[0];
            auto targetLast = &target[targetCount];

            auto recordedBegin = &recordedList[0];
            auto recordedLast = &recordedList[recordedListCount];

            auto removedTargetLast = std::remove_if(targetBegin, targetLast, [&recordedBegin, &recordedLast, &forceUpdateList](ncm::ContentMetaKey& targetKey) {
                auto recordedTitle = std::find_if(recordedBegin, recordedLast, [&targetKey](const ncm::ContentMetaKey& installedKey) {
                    return targetKey.id == installedKey.id && targetKey.type == installedKey.type;
                });

                auto recorded = recordedTitle != recordedLast;
                auto forceUpdate = recorded && forceUpdateList ? (*forceUpdateList)[static_cast<util::BitArray::size_type>(std::distance(recordedBegin, recordedTitle))] : false;

                switch (targetKey.type)
                {
                    // アプリのリマスターはサポートしていない。再ダウンロードでなければ更新から削除する
                    case ncm::ContentMetaType::Application:
                    {
                        return !forceUpdate;
                    }

                    case ncm::ContentMetaType::Patch:
                    {
                        // 再ダウンロード、または記録がない（一度もダウンロードされてない）ならダウンロード
                        if (forceUpdate || !recorded)
                        {
                            return false;
                        }

                        // インストールされてるパッチより新しければダウンロード
                        if (targetKey.version > recordedTitle->version)
                        {
                            return false;
                        }

                        return true;
                    }

                    case ncm::ContentMetaType::AddOnContent:
                    {
                        // 記録がある追加コンテンツだけを更新対象にする
                        if (recorded)
                        {
                            if (forceUpdate)
                            {
                                return false;
                            }

                            // インストールされている追加コンテンツより新しければダウンロード
                            if (targetKey.version > recordedTitle->version)
                            {
                                return false;
                            }
                        }

                        return true;
                    }

                    default:
                    {
                        // 基本的にはここにはこないはずだが、来てしまった場合は情報だけ出力して不要とする
                        NN_DETAIL_NS_TRACE("Unexpected content meta: Id: 0x%016llx, Version: %d, Type: %d\n", targetKey.id, targetKey.version, targetKey.type);
                        return true;
                    }
                }
            });

            *outCount = static_cast<int>(removedTargetLast - targetBegin);
        }

        void FilterDownloadApplicationContentMetaKey(ncm::ContentMetaKey target[], int* outCount, int targetCount, const ncm::ContentMetaKey needDownloadList[], int needDownloadListCount) NN_NOEXCEPT
        {
            auto targetBegin = &target[0];
            auto targetLast = &target[targetCount];

            auto needDownloadBegin = &needDownloadList[0];
            auto needDownloadLast = &needDownloadList[needDownloadListCount];

            auto removedTargetLast = std::remove_if(targetBegin, targetLast, [&needDownloadBegin, &needDownloadLast](ncm::ContentMetaKey& targetKey) {
                auto recordedTitle = std::find_if(needDownloadBegin, needDownloadLast, [&targetKey](const ncm::ContentMetaKey& needDownloadKey) {
                    return targetKey.id == needDownloadKey.id && targetKey.type == needDownloadKey.type;
                });

                auto recorded = recordedTitle != needDownloadLast;

                switch (targetKey.type)
                {
                    // 記録がないならダウンロード
                    case ncm::ContentMetaType::Application:
                    {
                        if (!recorded)
                        {
                            return false;
                        }

                        return true;
                    }

                    case ncm::ContentMetaType::Patch:
                    {
                        // 記録がないか、インストールされてるパッチより新しければダウンロード
                        if (!recorded || targetKey.version > recordedTitle->version)
                        {
                            return false;
                        }

                        return true;
                    }

                    default:
                    {
                        return true;
                    }
                }
            });

            *outCount = static_cast<int>(removedTargetLast - targetBegin);
        }

        void FilterDownloadAddOnContentContentMetaKey(ncm::ContentMetaKey target[], int* outCount, int targetCount, const ncm::ContentMetaKey needDownloadList[], util::BitArray* forceUpdateList, int needDownloadListCount) NN_NOEXCEPT
        {
            auto targetBegin = &target[0];
            auto targetLast = &target[targetCount];

            auto needDownloadBegin = &needDownloadList[0];
            auto needDownloadLast = &needDownloadList[needDownloadListCount];

            auto removedTargetLast = std::remove_if(targetBegin, targetLast, [&needDownloadBegin, &needDownloadLast, &forceUpdateList](ncm::ContentMetaKey& targetKey) {
                auto recordedTitle = std::find_if(needDownloadBegin, needDownloadLast, [&targetKey](const ncm::ContentMetaKey& needDownloadKey) {
                    return targetKey.id == needDownloadKey.id && targetKey.type == needDownloadKey.type;
                });

                auto recorded = recordedTitle != needDownloadLast;
                auto forceUpdate = recorded && forceUpdateList ? (*forceUpdateList)[static_cast<util::BitArray::size_type>(std::distance(needDownloadBegin, recordedTitle))] : false;

                switch (targetKey.type)
                {
                    case ncm::ContentMetaType::AddOnContent:
                    {
                        if (forceUpdate)
                        {
                            return false;
                        }

                        // インストールされてる追加コンテンツより新しければダウンロード
                        if (recorded && targetKey.version > recordedTitle->version)
                        {
                            return false;
                        }

                        return true;
                    }

                    default:
                    {
                        return true;
                    }
                }
            });

            *outCount = static_cast<int>(removedTargetLast - targetBegin);
        }

        Result ConvertToNsGameCardRegistrationResult(Result rawResult) NN_NOEXCEPT
        {
            NN_RESULT_TRY(rawResult)
                NN_RESULT_CATCH_CONVERT(nim::ResultGameCardRegistrationAlreadyRegistered, ns::ResultGameCardAlreadyRegistered())
                NN_RESULT_CATCH_CONVERT(nim::ResultGameCardRegistrationUnexpectedGoldPoint, ns::ResultGameCardUnexpectedGoldPoint())
                NN_RESULT_CATCH_CONVERT(nim::ResultGameCardRegistrationInvalidSignature, ns::ResultGameCardRegistrationInvalidSignature())
                NN_RESULT_CATCH_CONVERT(nim::ResultShogunPointExpired, ns::ResultGameCardRegistrationExpired())
                NN_RESULT_CATCH_CONVERT(nim::ResultShogunPointNotFound, ns::ResultGameCardGoldPointNotExist())
                NN_RESULT_CATCH_CONVERT(fs::ResultGameCardAccessFailed, ns::ResultGameCardAccessFailed())
                NN_RESULT_CATCH_CONVERT(nim::ResultShogunPointUnreleased, ns::ResultGameCardNotReleased())
                NN_RESULT_CATCH_CONVERT(nim::ResultShogunPointInvalidCountry, ns::ResultGameCardInvalidCountry())
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY;

            return rawResult;
        }

        uint32_t GetNextAvailableTimeOfUnexpectedError() NN_NOEXCEPT
        {
            uint32_t value = 0;
            size_t valueSize = settings::fwdbg::GetSettingsItemValue(&value, sizeof(value), "ns.ticket", "next_available_time_of_unexpected_error");
            NN_SDK_ASSERT(valueSize == sizeof(value), "[AsyncImpl] Faild to get system settings. (next_available_time_of_unexpected_error)\n");
            NN_UNUSED(valueSize);

            return value;
        }

        NN_STATIC_ASSERT(MaxApplicationCountOnGameCard <= nim::MaxRegisterableApplicationCount);
    }

    Result AsyncBase::ToAsyncResult(Result result) NN_NOEXCEPT
    {
        if (result <= nim::ResultHttpConnectionCanceled()) return ResultCanceled();
        if (result <= ncm::ResultInstallTaskCancelled()) return ResultCanceled();
        return result;
    }

    AsyncLatestSystemUpdateImpl::~AsyncLatestSystemUpdateImpl() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[AsyncLatestSystemUpdateImpl] Destructor\n");

        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
    }

    Result AsyncLatestSystemUpdateImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncLatestSystemUpdateTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
        {
            auto t = reinterpret_cast<AsyncLatestSystemUpdateImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncLatestSystemUpdateTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    }

    Result AsyncLatestSystemUpdateImpl::GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        LatestSystemUpdate latest;
        NN_RESULT_DO(GetImpl(&latest));

        NN_RESULT_THROW_UNLESS(sizeof(latest) <= buffer.GetSize(), ResultBufferNotEnough());
        std::memcpy(buffer.GetPointerUnsafe(), &latest, sizeof(latest));

        NN_RESULT_SUCCESS;
    }

    Result AsyncLatestSystemUpdateImpl::Execute() NN_NOEXCEPT
    {
        ncm::ContentMetaKey latestKey;
        {
            NN_UTIL_SCOPE_EXIT
            {
                std::lock_guard<os::Mutex> guard(m_CancelMutex);
                m_AsyncKey = util::nullopt;
            };
            {
                std::lock_guard<os::Mutex> guard(m_CancelMutex);
                NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultCanceled());
                m_AsyncKey.emplace();

                NN_RESULT_DO(RequestSystemUpdateMeta(&(*m_AsyncKey)));
            }

            NN_RESULT_DO(SaveErrorContextIfFailed(*m_AsyncKey, GetLatestSystemUpdateMeta(&latestKey, &(*m_AsyncKey))));
        }
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultCanceled());
        }

        nim::SystemUpdateTaskId id;
        auto count = nim::ListSystemUpdateTask(&id, 1);
        if (count > 0)
        {
            nim::SystemUpdateTaskInfo info;
            NN_RESULT_DO(nn::nim::GetSystemUpdateTaskInfo(&info, id));
            if (info.key == latestKey)
            {
                NN_RESULT_DO(GetLatestSystemUpdateState(&m_LatestSystemUpdate, id));
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_RESULT_DO(nn::nim::DestroySystemUpdateTask(id));
            }
        }

        bool needsUpdate;
        NN_RESULT_DO(NeedsUpdate(&needsUpdate, latestKey, m_RequiresExFatDriver));

        // 本体更新が必要な場合は、それが再起動不要で適用可能かどうかを判定する
        if (needsUpdate)
        {
            NN_RESULT_DO(nim::CreateSystemUpdateTask(&id, latestKey, m_RequiresExFatDriver));
            NN_RESULT_DO(GetLatestSystemUpdateState(&m_LatestSystemUpdate, id));
        }
        else
        {
            m_LatestSystemUpdate = LatestSystemUpdate::UpToDate;
        }

        NN_RESULT_SUCCESS;
    }

    Result AsyncLatestSystemUpdateImpl::GetImpl(LatestSystemUpdate* outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_Event.TryWait(), ResultNotImplemented());

        *outValue = m_LatestSystemUpdate;
        return m_Result;
    }

    void AsyncLatestSystemUpdateImpl::CancelImpl() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        m_IsCanceled = true;

        if (m_AsyncKey)
        {
            m_AsyncKey->Cancel();
        }
        if (m_AsyncResult)
        {
            m_AsyncResult->Cancel();
        }
    }

    Result AsyncLatestSystemUpdateImpl::GetLatestSystemUpdateState(LatestSystemUpdate* outValue, nim::SystemUpdateTaskId id) NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            m_AsyncResult = util::nullopt;
        };

        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultCanceled());
            m_AsyncResult.emplace();
            NN_RESULT_DO(nim::RequestSystemUpdateTaskRun(&(*m_AsyncResult), id));
        }

        nim::SystemUpdateTaskInfo info;
        NN_RESULT_DO(nim::GetSystemUpdateTaskInfo(&info, id));

        // CannotJudgeYet の場合は再起動不要で更新が適用できるか判定できる段階までまだタスクが進行していないので、
        // 判断できるまで再試行する
        while (info.applyInfo == ncm::SystemUpdateTaskApplyInfo::CannotJudgeYet && info.progress.GetLastResult().IsSuccess())
        {
            {
                std::lock_guard<os::Mutex> guard(m_CancelMutex);
                NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultCanceled());
            }
            os::SleepThread(TimeSpan::FromMilliSeconds(200));
            NN_RESULT_DO(nim::GetSystemUpdateTaskInfo(&info, id));
        }
        NN_RESULT_DO(info.progress.GetLastResult());

        // 再起動不要で適用可能な場合は上位には UpToDate を返す
        if (info.applyInfo == ncm::SystemUpdateTaskApplyInfo::RequireNoReboot)
        {
            *outValue = LatestSystemUpdate::UpToDate;
        }
        else
        {
            *outValue = info.progress.state == ncm::InstallProgressState::Downloaded ? LatestSystemUpdate::Downloaded : LatestSystemUpdate::NeedsDownload;
        }
        NN_RESULT_SUCCESS;
    }

    AsyncDownloadLatestUpdateImpl::~AsyncDownloadLatestUpdateImpl() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[AsyncDownloadLatestUpdateImpl] Destructor\n");

        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
    }

    Result AsyncDownloadLatestUpdateImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncDownloadLatestUpdateTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
        {
            auto t = reinterpret_cast<AsyncDownloadLatestUpdateImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncDownloadLatestUpdateTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    }

    Result AsyncDownloadLatestUpdateImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

    void AsyncDownloadLatestUpdateImpl::CancelImpl() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        m_IsCanceled = true;

        if (m_AsyncKey)
        {
            m_AsyncKey->Cancel();
        }
        if (m_AsyncResult)
        {
            m_AsyncResult->Cancel();
        }
    }

    Result AsyncDownloadLatestUpdateImpl::Execute() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            m_AsyncKey = util::nullopt;
        };
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultCanceled());
            m_AsyncKey.emplace();

            NN_RESULT_DO(RequestSystemUpdateMeta(&(*m_AsyncKey)));
        }

        ncm::ContentMetaKey latestKey;
        NN_RESULT_DO(SaveErrorContextIfFailed(*m_AsyncKey, GetLatestSystemUpdateMeta(&latestKey, &(*m_AsyncKey))));

        bool needsUpdate;
        NN_RESULT_DO(NeedsUpdate(&needsUpdate, latestKey, m_RequiresExFatDriver));
        NN_RESULT_THROW_UNLESS(needsUpdate, ResultAlreadyUpToDate());

        nim::SystemUpdateTaskId id;
        NN_RESULT_DO(CreateOrReuseSystemUpdateTask(&id, latestKey, m_RequiresExFatDriver));

        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            m_AsyncResult = util::nullopt;
        };
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultCanceled());
            m_AsyncResult.emplace();
            NN_RESULT_DO(RequestSystemUpdateTaskRun(&(*m_AsyncResult), id));
        }

        NN_RESULT_DO(GetAndSaveErrorContext(*m_AsyncResult));


        NN_RESULT_SUCCESS;
    }

    AsyncPrepareCardUpdateImpl::~AsyncPrepareCardUpdateImpl() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
    }

    void AsyncPrepareCardUpdateImpl::CancelImpl() NN_NOEXCEPT
    {
        m_Task->Cancel();
    }

    Result AsyncPrepareCardUpdateImpl::Execute() NN_NOEXCEPT
    {
        return m_Task->PrepareAndExecute();
    }

    Result AsyncPrepareCardUpdateImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncPrepareCardUpdateTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
        {
            auto t = reinterpret_cast<AsyncPrepareCardUpdateImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncPrepareCardUpdateTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationUpdateInfoImpl::Initialize(const ncm::ApplicationId applicationId, ncm::ContentMetaKey keyList[], int count) NN_NOEXCEPT
    {
        std::memcpy(m_InstalledContentMetaKeyList, keyList, sizeof(keyList[0]) * count);
        m_InstalledContentMetaKeyCount = count;
        return ns::srv::RequestLatestVersion(&m_AsyncLatestKeyList, applicationId);
    }

    Result AsyncApplicationUpdateInfoImpl::Immediate(ApplicationUpdateInfo immediate) NN_NOEXCEPT
    {
        m_Immediate = immediate;
        m_ImmediateEvent.Signal();

        NN_RESULT_SUCCESS;
    }

    Result AsyncApplicationUpdateInfoImpl::GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(sizeof(ApplicationUpdateInfo) <= buffer.GetSize(), ResultBufferNotEnough());

        if (m_Immediate)
        {
            auto info = *m_Immediate;
            std::memcpy(buffer.GetPointerUnsafe(), &info, sizeof(info));
            NN_RESULT_SUCCESS;
        }

        int count;
        NN_RESULT_DO(SaveErrorContextIfFailed(m_AsyncLatestKeyList, GetLatestKeyList(&count, &m_AsyncLatestKeyList, m_LatestKeyList, MaxContentMetaCountPerApplication)));

        FilterUpdateContentMetaKey(m_LatestKeyList, &count, count, m_InstalledContentMetaKeyList, nullptr, m_InstalledContentMetaKeyCount);

        auto info = count > 0 ? ApplicationUpdateInfo::Updatable : ApplicationUpdateInfo::UpToDate;
        std::memcpy(buffer.GetPointerUnsafe(), &info, sizeof(info));
        NN_RESULT_SUCCESS;
    }

    Result AsyncDownloadApplicationImpl::Add(const ncm::ContentMetaKey keyList[], int count) NN_NOEXCEPT
    {
        NN_RESULT_DO(Add(keyList, count, false));

        NN_RESULT_SUCCESS;
    }

    Result AsyncDownloadApplicationImpl::Add(const ncm::ContentMetaKey keyList[], int count, bool forceUpdate) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_InstalledContentMetaKeyCount + count <= MaxContentMetaCountPerApplication, ResultBufferNotEnough());
        std::memcpy(&m_RequestedContentMetaKeyList[m_InstalledContentMetaKeyCount], keyList, sizeof(keyList[0]) * count);

        for (int i = 0; i < count; i++)
        {
            m_ForceUpdateList[m_InstalledContentMetaKeyCount + i] = forceUpdate;
        }

        m_InstalledContentMetaKeyCount += count;
        NN_RESULT_SUCCESS;
    }

    Result AsyncDownloadApplicationImpl::Run() NN_NOEXCEPT
    {
        return ns::srv::RequestLatestVersion(&m_AsyncLatestKeyList, m_AppId);
    }

    Result AsyncDownloadApplicationImpl::GetImpl() NN_NOEXCEPT
    {
        int latestCount = 0;
        {
            int count;
            NN_RESULT_DO(SaveErrorContextIfFailed(m_AsyncLatestKeyList, GetLatestKeyList(&count, &m_AsyncLatestKeyList, m_LatestKeyList, MaxContentMetaCountPerApplication)));

            if (m_Mode == DownloadApplicationMode::Update)
            {
                FilterUpdateContentMetaKey(m_LatestKeyList, &latestCount, count, m_RequestedContentMetaKeyList, &m_ForceUpdateList, m_InstalledContentMetaKeyCount);
            }
            else if (m_Mode == DownloadApplicationMode::DownloadApplication)
            {
                FilterDownloadApplicationContentMetaKey(m_LatestKeyList, &latestCount, count, m_RequestedContentMetaKeyList, m_InstalledContentMetaKeyCount);
            }
            else
            {
                FilterDownloadAddOnContentContentMetaKey(m_LatestKeyList, &latestCount, count, m_RequestedContentMetaKeyList, &m_ForceUpdateList, m_InstalledContentMetaKeyCount);
            }
        }

        int updateCount = latestCount;
        NN_RESULT_THROW_UNLESS(updateCount > 0, ResultAlreadyUpToDate());

        nim::NetworkInstallTaskId taskId;
        if (!m_ForceDirectUpdate)
        {
            NN_RESULT_DO(nim::CreateNetworkInstallTask(&taskId, m_AppId, m_LatestKeyList, updateCount, m_StorageId));
        }
        else
        {
            // 必ず直接更新を行うフラグを立てる(InstallConfig_LatestAddOnContentsAuto は標準で立てているのでそれも立てる)
            Bit32 config = ncm::InstallConfig_LatestAddOnContentsAuto | ncm::InstallConfig_SkipIndirectUpdateCheck;
            NN_RESULT_DO(nim::CreateNetworkInstallTask(&taskId, m_AppId, m_LatestKeyList, updateCount, m_StorageId, config));
        }

        NN_RESULT_SUCCESS;
    }

    template<class AsyncT>
    AsyncGameCardBase<AsyncT>::~AsyncGameCardBase() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
    }

    template<class AsyncT>
    Result AsyncGameCardBase<AsyncT>::Initialize(const account::Uid& uid, ncm::ApplicationId appId, GameCardManager* gameCardManager) NN_NOEXCEPT
    {
        m_Uid = uid;
        m_AppId = appId;

        NN_RESULT_DO(gameCardManager->GetGameCardInfo(&m_GameCardInfo));
        NN_RESULT_THROW_UNLESS(
            std::any_of(m_GameCardInfo.idList, m_GameCardInfo.idList + m_GameCardInfo.idCount, [&appId](ncm::ApplicationId id) NN_NOEXCEPT { return appId == id; }),
            ns::ResultGameCardApplicationIdMismatch());

        NN_RESULT_SUCCESS;
    }

    template<class AsyncT>
    Result AsyncGameCardBase<AsyncT>::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncGameCardRegistrationTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
            {
                auto t = reinterpret_cast<AsyncGameCardBase<AsyncT>*>(p);
                t->m_Result = ConvertToNsGameCardRegistrationResult(t->Execute());
                t->m_Event.Signal();
            }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncGameCardRegistrationTask));
        os::StartThread(info.thread);
        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    };

    template<class AsyncT>
    void AsyncGameCardBase<AsyncT>::CancelBase() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        if (m_AuthRequest)
        {
            m_AuthRequest->Cancel();
        }
        if (m_Async)
        {
            m_Async->Cancel();
        }
    }

    template<class AsyncT>
    Result AsyncGameCardBase<AsyncT>::GetToken() NN_NOEXCEPT
    {
        if (m_Uid == account::InvalidUid)
        {
            m_IdToken[0] = '\0';
            NN_RESULT_SUCCESS;
        }

        account::NetworkServiceAccountManager accountManager;
        NN_RESULT_DO(account::GetNetworkServiceAccountManager(&accountManager, m_Uid));

        nsd::NasServiceSetting shopSetting;
        NN_RESULT_DO(nsd::GetNasServiceSetting(&shopSetting, nsd::NasServiceNameOfNxShop));

        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            m_AuthRequest = util::nullopt;
        };
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);

            m_AuthRequest.emplace();
            account::NintendoAccountAuthorizationRequestParameters params = {
                "nxeshop",
                " ",
                " "
            };

            NN_ALIGNAS(4096) static char g_AuthRequestWork[4096];
            NN_RESULT_DO(accountManager.CreateNintendoAccountAuthorizationRequest(
                &(*m_AuthRequest),
                shopSetting.clientId,
                shopSetting.redirectUri.value,
                nsd::NasServiceSetting::RedirectUri::Size,
                params,
                g_AuthRequestWork,
                sizeof(g_AuthRequestWork)));
        }

        os::SystemEvent event;
        NN_RESULT_DO(m_AuthRequest->GetSystemEvent(&event));
        event.Wait();

        NN_RESULT_DO(m_AuthRequest->GetResult());
        size_t tokenSize;
        NN_RESULT_DO(m_AuthRequest->GetIdToken(&tokenSize, m_IdToken, sizeof(m_IdToken)));
        m_IdToken[tokenSize] = '\0';

        NN_RESULT_SUCCESS;
    }

    template<class AsyncT>
    Result AsyncGameCardBase<AsyncT>::Execute() NN_NOEXCEPT
    {
        NN_RESULT_DO(GetToken());

        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);

            m_Async.emplace();
            auto result = RequestAsync(&(*m_Async), m_IdToken, m_GameCardInfo.cert, sizeof(m_GameCardInfo.cert));
            if (result.IsFailure())
            {
                m_Async = util::nullopt;
            }
            NN_RESULT_DO(result);
        }
        m_Async->Wait();

        NN_RESULT_SUCCESS;
    };

    template<class AsyncT>
    const ncm::ApplicationId* AsyncGameCardBase<AsyncT>::GetApplicationIdList() const NN_NOEXCEPT
    {
        return m_GameCardInfo.idList;
    };

    template<class AsyncT>
    int AsyncGameCardBase<AsyncT>::GetApplicationIdListCount() const NN_NOEXCEPT
    {
        return m_GameCardInfo.idCount;
    };

// ------------------------------------------

    AsyncCheckGameCardRegistrationImpl::~AsyncCheckGameCardRegistrationImpl() NN_NOEXCEPT {}

    Result AsyncCheckGameCardRegistrationImpl::Initialize(ncm::ApplicationId appId, GameCardManager* gameCardManager) NN_NOEXCEPT
    {
        NN_RESULT_DO(AsyncGameCardBase::Initialize(account::InvalidUid, appId, gameCardManager));
        NN_RESULT_SUCCESS;
    }

    Result AsyncCheckGameCardRegistrationImpl::Run() NN_NOEXCEPT
    {
        return AsyncGameCardBase::Run();
    }

    Result AsyncCheckGameCardRegistrationImpl::GetImpl() NN_NOEXCEPT
    {
        NN_RESULT_DO(GetResult());

        nim::GameCardRegistrationStatus statusLow;
        NN_RESULT_DO(SaveErrorContextIfFailed(*GetAsync(), ConvertToNsGameCardRegistrationResult(GetAsync()->Get(&statusLow))));

        NN_RESULT_THROW_UNLESS(!statusLow.isRegistered, ResultGameCardAlreadyRegistered());
        NN_RESULT_SUCCESS;
    }

    void AsyncCheckGameCardRegistrationImpl::CancelImpl() NN_NOEXCEPT
    {
        AsyncGameCardBase::CancelBase();
    }

    Result AsyncCheckGameCardRegistrationImpl::RequestAsync(nim::AsyncGameCardRegistrationStatus* outValue, const char* token, const void* cert, size_t certSize) NN_NOEXCEPT
    {
        return nn::nim::RequestGameCardRegistrationStatus(outValue, GetApplicationIdList(), GetApplicationIdListCount(), token, cert, certSize);
    }

// ------------------------------------------

    AsyncGameCardRegistrationGoldPointImpl::~AsyncGameCardRegistrationGoldPointImpl() NN_NOEXCEPT {}

    Result AsyncGameCardRegistrationGoldPointImpl::Initialize(const account::Uid& uid, ncm::ApplicationId appId, GameCardManager* gameCardManager) NN_NOEXCEPT
    {
        NN_RESULT_DO(AsyncGameCardBase::Initialize(uid, appId, gameCardManager));
        NN_RESULT_SUCCESS;
    }

    Result AsyncGameCardRegistrationGoldPointImpl::Run() NN_NOEXCEPT
    {
        return AsyncGameCardBase::Run();
    }

    size_t AsyncGameCardRegistrationGoldPointImpl::GetSizeImpl() NN_NOEXCEPT
    {
        return sizeof(int);
    }

    Result AsyncGameCardRegistrationGoldPointImpl::GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        NN_RESULT_DO(GetResult());

        nim::GameCardRegistrationStatus statusLow;
        NN_RESULT_DO(SaveErrorContextIfFailed(*GetAsync(), ConvertToNsGameCardRegistrationResult(GetAsync()->Get(&statusLow))));

        std::memcpy(buffer.GetPointerUnsafe(), &statusLow.goldPoint, sizeof(statusLow.goldPoint));

        NN_RESULT_SUCCESS;
    }

    void AsyncGameCardRegistrationGoldPointImpl::CancelImpl() NN_NOEXCEPT
    {
        AsyncGameCardBase::CancelBase();
    }

    Result AsyncGameCardRegistrationGoldPointImpl::RequestAsync(nim::AsyncGameCardRegistrationStatus* outValue, const char* token, const void* cert, size_t certSize) NN_NOEXCEPT
    {
        return nn::nim::RequestGameCardRegistrationStatus(outValue, GetApplicationIdList(), GetApplicationIdListCount(), token, cert, certSize);
    }

// ------------------------------------------

    AsyncRegisterGameCardImpl::~AsyncRegisterGameCardImpl() NN_NOEXCEPT {}

    Result AsyncRegisterGameCardImpl::Initialize(const account::Uid& uid, ncm::ApplicationId appId, int goldPoint, GameCardManager* gameCardManager) NN_NOEXCEPT
    {
        NN_RESULT_DO(AsyncGameCardBase::Initialize(uid, appId, gameCardManager));
        m_GoldPoint = goldPoint;

        NN_RESULT_SUCCESS;
    }

    Result AsyncRegisterGameCardImpl::Run() NN_NOEXCEPT
    {
        return AsyncGameCardBase::Run();
    }

    Result AsyncRegisterGameCardImpl::GetImpl() NN_NOEXCEPT
    {
        NN_RESULT_DO(AsyncGameCardBase::GetResult());
        NN_RESULT_DO(SaveErrorContextIfFailed(*AsyncGameCardBase::GetAsync(), ConvertToNsGameCardRegistrationResult(AsyncGameCardBase::GetAsync()->Get())));
        NN_RESULT_SUCCESS;
    }

    void AsyncRegisterGameCardImpl::CancelImpl() NN_NOEXCEPT
    {
        AsyncGameCardBase::CancelBase();
    }

    Result AsyncRegisterGameCardImpl::RequestAsync(nim::AsyncResult* outValue, const char* token, const void* cert, size_t certSize) NN_NOEXCEPT
    {
        nim::GameCardRegistrationStatus status = { m_GoldPoint, false };
        return nim::RequestRegisterGameCard(outValue, GetApplicationIdList(), GetApplicationIdListCount(), token, cert, certSize, status);
    }

// ------------------------------------------

    AsyncDeviceLinkBase::~AsyncDeviceLinkBase() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
    }

    Result AsyncDeviceLinkBase::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncDeviceLinkTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
        {
            auto t = reinterpret_cast<AsyncDeviceLinkBase*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncDeviceLinkTask));
        os::StartThread(info.thread);
        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    };

    void AsyncDeviceLinkBase::CancelBase() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        m_IsCanceled = true;

        if (m_Async)
        {
            m_Async->Cancel();
        }

        if (m_AsyncInfo)
        {
            m_AsyncInfo->Cancel();
        }
    }

    Result AsyncDeviceLinkBase::Execute() NN_NOEXCEPT
    {
        return ConvertEcResult(ExecuteImpl());
    }

    Result AsyncDeviceLinkBase::ExecuteImpl() NN_NOEXCEPT
    {
        util::optional<ec::system::DeviceAccountInfo> deviceAccountInfo;
        {
            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

            deviceAccountInfo.emplace();
            NN_RESULT_TRY(ec::system::GetDeviceAccountInfo(&(*deviceAccountInfo)))
                NN_RESULT_CATCH(ec::ResultDeviceAccountNotRegistered)
                {
                    deviceAccountInfo = util::nullopt;
                }
            NN_RESULT_END_TRY
        }

        if (!deviceAccountInfo)
        {
            util::optional<ec::system::DeviceRegistrationInfo> deviceRegistrationInfo;
            {
                m_AsyncInfo.emplace();
                NN_UTIL_SCOPE_EXIT{ m_AsyncInfo = util::nullopt; };
                NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

                NN_RESULT_DO(ec::system::RequestDeviceRegistrationInfoPrivate(&(*m_AsyncInfo)));
                deviceRegistrationInfo.emplace();
                NN_RESULT_TRY(SaveErrorContextIfFailed(*m_AsyncInfo, m_AsyncInfo->Get(&(*deviceRegistrationInfo))))
                    NN_RESULT_CATCH(ec::ResultDeviceNotRegistered)
                    {
                        deviceRegistrationInfo = util::nullopt;
                    }
                NN_RESULT_END_TRY
            }

            if(deviceRegistrationInfo)
            {
                {
                    m_Async.emplace();
                    NN_UTIL_SCOPE_EXIT{ m_Async = util::nullopt; };
                    NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

                    NN_RESULT_DO(ec::system::RequestUnlinkDeviceAllPrivate(&(*m_Async)));
                    NN_RESULT_DO(GetAndSaveErrorContext(*m_Async));
                }

                {
                    m_Async.emplace();
                    NN_UTIL_SCOPE_EXIT{ m_Async = util::nullopt; };
                    NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

                    NN_RESULT_DO(ec::system::RequestUnregisterDeviceAccountPrivate(&(*m_Async)));
                    NN_RESULT_DO(GetAndSaveErrorContext(*m_Async));
                }
            }

            {
                m_Async.emplace();
                NN_UTIL_SCOPE_EXIT{ m_Async = util::nullopt; };
                NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

                NN_RESULT_DO(ec::system::RequestRegisterDeviceAccountPrivate(&(*m_Async)));
                NN_RESULT_DO(GetAndSaveErrorContext(*m_Async));
            }

            {
                NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

                deviceAccountInfo.emplace();
                NN_RESULT_DO(ec::system::GetDeviceAccountInfo(&(*deviceAccountInfo)));
            }
        }

        {
            m_Async.emplace();
            NN_UTIL_SCOPE_EXIT{ m_Async = util::nullopt; };
            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

            NN_RESULT_DO(nn::ec::system::RequestCreateVirtualAccountPrivate(&(*m_Async), m_Uid));
            NN_RESULT_DO(GetAndSaveErrorContext(*m_Async));
        }

        {
            m_Async.emplace();
            NN_UTIL_SCOPE_EXIT{ m_Async = util::nullopt; };
            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

            NN_RESULT_DO(ec::system::RequestLinkDevicePrivate(&(*m_Async), m_Uid));
            // 他のデバイスで既にリンク済みの場合、ec::ResultAccountAlreadyDeviceLinked が返る
            NN_RESULT_DO(GetAndSaveErrorContext(*m_Async));
        }

        NN_RESULT_SUCCESS;
    }

    bool AsyncDeviceLinkBase::IsCanceled() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        return m_IsCanceled;
    }

// ------------------------------------------

    AsyncDeviceLinkImpl::~AsyncDeviceLinkImpl() NN_NOEXCEPT {}

    Result AsyncDeviceLinkImpl::Run() NN_NOEXCEPT
    {
        return AsyncDeviceLinkBase::Run();
    }

    Result AsyncDeviceLinkImpl::GetImpl() NN_NOEXCEPT
    {
        AsyncDeviceLinkBase::GetEvent().Wait();
        return AsyncDeviceLinkBase::GetResult();
    }

    void AsyncDeviceLinkImpl::CancelImpl() NN_NOEXCEPT
    {
        AsyncDeviceLinkBase::CancelBase();
    }

// ------------------------------------------

    AsyncDownloadApplicationPrepurchasedRightsBase::~AsyncDownloadApplicationPrepurchasedRightsBase() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
    }

    Result AsyncDownloadApplicationPrepurchasedRightsBase::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncDeviceLinkTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
        {
            auto t = reinterpret_cast<AsyncDownloadApplicationPrepurchasedRightsBase*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncDownloadApplicationPrepurchasedRightsTask));
        os::StartThread(info.thread);
        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    };

    void AsyncDownloadApplicationPrepurchasedRightsBase::CancelBase() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        m_IsCanceled = true;

        if (m_Async)
        {
            m_Async->Cancel();
        }
    }

    Result AsyncDownloadApplicationPrepurchasedRightsBase::Execute() NN_NOEXCEPT
    {
        return ConvertEcResult(ExecuteImpl());
    }

    Result AsyncDownloadApplicationPrepurchasedRightsBase::ExecuteImpl() NN_NOEXCEPT
    {
        {
            // まだアクセスできる時間ではない場合
            if (m_pTicketManager->GetNextAvailableTimeForDownloadApplicationPrepurchasedRights() > os::GetSystemTick().ToTimeSpan() && m_pTicketManager->GetNextAvailableTimeForDownloadApplicationPrepurchasedRights() != TimeSpan::FromSeconds(0))
            {
                NN_RESULT_THROW(ResultTryLater());
            }
        }

        {
            auto stopper = m_RequestServer->Stop();

            // チケットのダウンロード
            bool hasReleasedContent = false;
            bool hasUnavailableContent = false;

            for(int i = 0; i < m_RightsIdCount; i++)
            {
                m_Async.emplace();
                NN_UTIL_SCOPE_EXIT { m_Async = util::nullopt; };
                NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

                NN_RESULT_DO(ec::system::RequestDownloadTicketForPrepurchasedContents(&(*m_Async), m_RightsIds[i]));

                ec::system::TicketDownloadStatusForPrepurchasedContents status;
                NN_RESULT_TRY(SaveErrorContextIfFailed(*m_Async, m_Async->Get(&status)))
                    // サーバ側の権利が失われていた場合、予約情報を削除して ResultPrepurchasedRightsLost を返す
                    NN_RESULT_CATCH(ec::ResultTicketNotFound)
                    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
                        es::DeleteTicket(&m_RightsIds[i], 1);
#endif
                        NN_RESULT_THROW(ResultPrepurchasedRightsLost());
                    }
                    NN_RESULT_CATCH(ec::ResultCanceled)
                    {
                        NN_RESULT_RETHROW;
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        // サーバ要因で予期せぬエラーが起きた場合、
                        // 連続してサーバにアクセスされないように次回アクセス可能な時間を設定する
                        m_pTicketManager->SetNextAvailableTimeForDownloadApplicationPrepurchasedRights(os::GetSystemTick().ToTimeSpan() + TimeSpan::FromSeconds(GetNextAvailableTimeOfUnexpectedError()));
                        NN_DETAIL_NS_INFO("[AsyncDownloadApplicationPrepurchasedRights] Failed to download application ticket for unexpected reason 0x%08x.\n", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                        NN_RESULT_RETHROW;
                    }
                NN_RESULT_END_TRY

                // サーバの負荷分散のために次回アクセス可能な時間が指定された場合
                if (status.delayTime != 0)
                {
                    m_pTicketManager->SetNextAvailableTimeForDownloadApplicationPrepurchasedRights(os::GetSystemTick().ToTimeSpan() + TimeSpan::FromSeconds(status.delayTime));
                }
                // 予約購入コンテンツがまだ利用できない場合
                if (status.isPrepurchaseRecordDownloaded)
                {
                    hasUnavailableContent = true;
                }
                else
                {
                    hasReleasedContent = true;
                }
            }

            // 全ての予約購入コンテンツのチケットの取得に成功した場合
            if (hasReleasedContent && !hasUnavailableContent)
            {
                NN_RESULT_SUCCESS;
            }
            // 一部の予約購入コンテンツのチケットの取得に成功した場合
            else if(hasReleasedContent && hasUnavailableContent)
            {
                NN_RESULT_THROW(ResultPrepurchasedContentPartiallyStillUnavailable());
            }
            // 全ての予約購入コンテンツのチケットの取得に失敗した場合
            else
            {
                NN_RESULT_THROW(ResultPrepurchasedContentStillUnavailable());
            }
        }
    }

    bool AsyncDownloadApplicationPrepurchasedRightsBase::IsCanceled() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        return m_IsCanceled;
    }

    // ------------------------------------------

    AsyncDownloadApplicationPrepurchasedRightsImpl::~AsyncDownloadApplicationPrepurchasedRightsImpl() NN_NOEXCEPT {}

    Result AsyncDownloadApplicationPrepurchasedRightsImpl::Run() NN_NOEXCEPT
    {
        return AsyncDownloadApplicationPrepurchasedRightsBase::Run();
    }

    Result AsyncDownloadApplicationPrepurchasedRightsImpl::GetImpl() NN_NOEXCEPT
    {
        AsyncDownloadApplicationPrepurchasedRightsBase::GetEvent().Wait();
        return AsyncDownloadApplicationPrepurchasedRightsBase::GetResult();
    }

    void AsyncDownloadApplicationPrepurchasedRightsImpl::CancelImpl() NN_NOEXCEPT
    {
        AsyncDownloadApplicationPrepurchasedRightsBase::CancelBase();
    }

    // ------------------------------------------

    AsyncSyncRightsBase::~AsyncSyncRightsBase() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
    }

    Result AsyncSyncRightsBase::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncDeviceLinkTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p) NN_NOEXCEPT
        {
            auto t = reinterpret_cast<AsyncSyncRightsBase*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncSyncRightsTask));
        os::StartThread(info.thread);
        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    };

    void AsyncSyncRightsBase::CancelBase() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        m_IsCanceled = true;

        if (m_Async)
        {
            m_Async->Cancel();
        }

        if (m_AsyncAssignedELicenses)
        {
            m_AsyncAssignedELicenses->Cancel();
        }

        if (m_AsyncProgress)
        {
            m_AsyncProgress->Cancel();
        }
    }

    Result AsyncSyncRightsBase::Execute() NN_NOEXCEPT
    {
        return ConvertEcResult(ExecuteImpl());
    }

    Result AsyncSyncRightsBase::ExecuteImpl() NN_NOEXCEPT
    {
        // 機器認証の権利の発行依頼を出す
        {
            m_AsyncAssignedELicenses.emplace();
            NN_UTIL_SCOPE_EXIT{ m_AsyncAssignedELicenses = util::nullopt; };
            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

            NN_RESULT_DO(nim::RequestAssignAllDeviceLinkedELicenses(&(*m_AsyncAssignedELicenses)));
            int count;
            NN_RESULT_DO(SaveErrorContextIfFailed(*m_AsyncAssignedELicenses, m_AsyncAssignedELicenses->GetSize(&count)));
        }

        // eLicense を同期する
        {
            account::Uid userList[account::UserCountMax];
            int userCount;
            NN_RESULT_DO(account::ListAllUsers(&userCount, userList, account::UserCountMax));

            for (const auto& user : util::MakeSpan(userList, userCount))
            {
                m_Async.emplace();
                NN_UTIL_SCOPE_EXIT{ m_Async = util::nullopt; };
                NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

                account::NintendoAccountId naId;
                NN_RESULT_TRY(GetNintendoAccountId(&naId, user))
                    NN_RESULT_CATCH(account::ResultNetworkServiceAccountUnavailable)    // NA 連携されていないアカウントは eLicense 同期をスキップする
                    {
                        continue;
                    }
                    NN_RESULT_CATCH_ALL                                                 // 想定外のエラーが起きた場合も同期可能なアカウントで同期をする
                    {
                        NN_DETAIL_NS_TRACE("[AsyncSyncRightsBase] Failed get NintendoAccountId %08x\n", NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug());
                        continue;
                    }
                NN_RESULT_END_TRY
                NN_RESULT_DO(nim::RequestSyncELicenses(&(*m_Async), naId));
                NN_RESULT_DO(GetAndSaveErrorContext(*m_Async));
            }
        }

        // チケットを同期する
        {
            m_AsyncProgress.emplace();
            NN_UTIL_SCOPE_EXIT{ m_AsyncProgress = util::nullopt; };
            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

            NN_RESULT_DO(ec::system::RequestSyncTicketForSystem(&(*m_AsyncProgress)));
            NN_RESULT_DO(GetAndSaveErrorContext(*m_AsyncProgress));
        }

        NN_RESULT_SUCCESS;
    }

    bool AsyncSyncRightsBase::IsCanceled() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        return m_IsCanceled;
    }

    // ------------------------------------------

    AsyncSyncRightsImpl::~AsyncSyncRightsImpl() NN_NOEXCEPT {}

    Result AsyncSyncRightsImpl::Run() NN_NOEXCEPT
    {
        return AsyncSyncRightsBase::Run();
    }

    Result AsyncSyncRightsImpl::GetImpl() NN_NOEXCEPT
    {
        AsyncSyncRightsBase::GetEvent().Wait();
        return AsyncSyncRightsBase::GetResult();
    }

    void AsyncSyncRightsImpl::CancelImpl() NN_NOEXCEPT
    {
        AsyncSyncRightsBase::CancelBase();
    }
}}}
