﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/arp/arp_Api.h>
#include <nn/fs/fs_Result.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_ApplicationViewSystemApi.h>
#include <nn/ns/ns_RetailInteractiveDisplayApi.h>
#include <nn/ns/srv/ns_ApplicationViewManager.h>
#include <nn/ns/srv/ns_LockedBufferManager.h>
#include <nn/ns/srv/ns_Shell.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include "ns_ApplicationInstallTask.h"
#include "ns_ApplicationApplyDeltaTask.h"
#include "ns_Config.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        bool IsMaybeCorrupted(Result result) NN_NOEXCEPT
        {
            return
                result <= fs::ResultDataCorrupted()
             || result <= ns::ResultApplicationLogoCorrupted();
        }
    }

Result ApplicationViewManager::CommitDownloadTaskIfNeeded(ncm::ApplicationId id) NN_NOEXCEPT
{
    bool isCommittable;
    NN_RESULT_DO(m_CommitManager->IsDownloadTaskCommittable(&isCommittable, id));
    if (isCommittable)
    {
        auto managedStop = m_RequestServer->Stop();
        NN_RESULT_DO(m_CommitManager->CommitDownloadTask(id));
    }
    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::CommitApplyDeltaTaskIfNeeded(ncm::ApplicationId id) NN_NOEXCEPT
{
    bool isCommittable;
    NN_RESULT_DO(m_CommitManager->IsApplyDeltaTaskCommittable(&isCommittable, id));
    if (isCommittable)
    {
        auto managedStop = m_RequestServer->Stop();
        NN_RESULT_DO(m_CommitManager->CommitApplyDeltaTask(id));
    }
    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::UpdateViewFromDownloadTask(ns::ApplicationView* view, ncm::ApplicationId id) NN_NOEXCEPT
{
    auto stopper = m_CommitManager->GetCommitStopper();

    ApplicationInstallTask task(id);
    if (!task.IsValid())
    {
        NN_RESULT_SUCCESS;
    }
    bool isDownloaded;
    NN_RESULT_DO(task.IsDownloaded(&isDownloaded));
    if (isDownloaded)
    {
        auto buffer = m_LockedBufferManager->Allocate(sizeof(ncm::StorageContentMetaKey) * MaxContentMetaCountPerApplication);
        auto keyList = buffer.Get<ncm::StorageContentMetaKey*>();

        int count;
        NN_RESULT_DO(task.ListNotCommittedKey(&count, keyList, MaxContentMetaCountPerApplication, 0));
        for (int i = 0; i < count; i++)
        {
            if (keyList[i].key.type == ncm::ContentMetaType::Application)
            {
                view->flag.Set<ApplicationViewFlag_WaitingApplicationCommit>(true);
            }
            else if (keyList[i].key.type == ncm::ContentMetaType::AddOnContent)
            {
                view->flag.Set<ApplicationViewFlag_WaitingAocCommit>(true);
            }
            else if (keyList[i].key.type == ncm::ContentMetaType::Patch)
            {
                view->flag.Set<ApplicationViewFlag_WaitingPatchInstall>(true);
            }
        }
    }
    else
    {
        view->flag.Set<ApplicationViewFlag_Downloading>(true);
    }
    NN_RESULT_DO(task.GetProgress(&(view->progress)));

    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::UpdateViewFromApplyDeltaTask(ns::ApplicationView* view, ncm::ApplicationId id) NN_NOEXCEPT
{
    auto stopper = m_CommitManager->GetCommitStopper();

    ApplicationApplyDeltaTask task(id);
    if (!task.IsValid())
    {
        NN_RESULT_SUCCESS;
    }
    view->flag.Set<ApplicationViewFlag_ApplyingDelta>(true);
    view->flag.Set<ApplicationViewFlag_IsLaunchable>(false);
    NN_RESULT_DO(task.GetProgress(&(view->applyProgress)));

    NN_RESULT_SUCCESS;
}

void ApplicationViewManager::Initialize(ApplicationRecordDatabase* record, ApplicationEntityManager* entityManager, IntegratedContentManager* integrated, RequestServer* requestServer, SystemReportManager* reportManager, CommitManager* commitManager, LockedBufferManager* LockedBufferManager) NN_NOEXCEPT
{
    m_RecordDb = record;
    m_EntityManager = entityManager;
    m_Integrated = integrated;
    m_RequestServer = requestServer;
    m_SystemReportManager = reportManager;
    m_CommitManager = commitManager;
    m_LockedBufferManager = LockedBufferManager;
    m_EntityViewCache.Initialize(m_RecordDb);
}

Result ApplicationViewManager::GetApplicationViewDeprecated(const sf::OutArray<ns::ApplicationViewDeprecated>& outList, const sf::InArray<ncm::ApplicationId>& idList) NN_NOEXCEPT
{
    for (size_t i = 0; i < outList.GetLength(); i++)
    {
        ApplicationView view;
        NN_RESULT_DO(GetApplicationViewOne(&view, idList[i]));

        ApplicationViewDeprecated deprecated = {};
        deprecated.id = view.id;
        deprecated.version = view.version;
        deprecated.flag = view.flag;
        deprecated.progress = {
            view.progress.downloaded,
            view.progress.total,
            view.progress.lastResult,
            view.progress.state,
            view.progress.isRunning
        };
        deprecated.applyProgress = {
            view.applyProgress.applied,
            view.applyProgress.total,
            view.applyProgress.lastResult,
            view.applyProgress.state,
        };
        outList[i] = deprecated;
    }

    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::GetApplicationView(const sf::OutArray<ns::ApplicationView>& outList, const sf::InArray<ncm::ApplicationId>& idList) NN_NOEXCEPT
{
    for (size_t i = 0; i < outList.GetLength(); i++)
    {
        NN_RESULT_DO(GetApplicationViewOne(&outList[i], idList[i]));
    }

    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::GetApplicationViewDownloadErrorContext(nn::sf::Out<nn::err::ErrorContext> outValue, nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_RecordDb->Has(id), ResultApplicationRecordNotFound());
    auto stopper = m_CommitManager->GetCommitStopper();

    ApplicationInstallTask task(id);
    if (!task.IsValid())
    {
        *outValue = {};
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(task.GetErrorContext(outValue.GetPointer()));
    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::ListApplicationDownloadingContentMeta(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::StorageContentMetaKey>& outList, ncm::ApplicationId id, std::int32_t offset) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_RecordDb->Has(id), ResultApplicationRecordNotFound());
    auto stopper = m_CommitManager->GetCommitStopper();

    ApplicationInstallTask task(id);
    if (!task.IsValid())
    {
        *outCount = 0;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(task.ListKey(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()), offset));
    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::GetApplicationViewOne(ns::ApplicationView* outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    // view の取得に先立ち、必要ならコミット処理を行う
    NN_RESULT_DO(CommitDownloadTaskIfNeeded(id));
    NN_RESULT_DO(CommitApplyDeltaTaskIfNeeded(id));

    ApplicationView view = {};
    bool cacheFound = true;
    NN_RESULT_TRY(m_EntityViewCache.LoadEntityViewCache(&view, id))
        NN_RESULT_CATCH(ResultEntityViewCacheNotFound)
        {
            cacheFound = false;
        }
        NN_RESULT_CATCH(ResultEntityViewCacheOutDated)
        {
            cacheFound = false;
        }
    NN_RESULT_END_TRY;
    if (!cacheFound)
    {
        ApplicationRecord record;
        NN_RESULT_DO(GetApplicationEntityView(&record, &view, id));
        m_EntityViewCache.StoreEntityViewCache(id, record.lastUpdated, view);
    }
    NN_RESULT_DO(UpdateViewFromDownloadTask(&view, id));
    NN_RESULT_DO(UpdateViewFromApplyDeltaTask(&view, id));

    *outValue = view;

    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::GetApplicationEntityView(ns::ApplicationRecord* outRecord, ns::ApplicationView* outView, ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationView view = {};
    view.id = id;

    view.flag.Set<ApplicationViewFlag_HasAllEntity>();
    view.flag.Set<ApplicationViewFlag_HasAllAddOnContentEntity>();
    {
        ApplicationRecordAccessor accessor;
        NN_RESULT_DO(m_RecordDb->Open(&accessor, id));
        for (int i = 0; i < accessor.Count(); i++)
        {
            auto& record = accessor[i];
            auto& key = record.key;

            view.flag.Set<ApplicationViewFlag_HasRecord>();

            switch (key.type)
            {
            case ncm::ContentMetaType::Application:
                {
                    if (record.storageId == ncm::StorageId::Card)
                    {
                        view.flag.Set<ApplicationViewFlag_IsGameCard>();
                    }

                    view.flag.Set<ApplicationViewFlag_HasMainRecord>();
                    if (view.version < key.version)
                    {
                        view.version = key.version;
                    }
                }
                break;
            case ncm::ContentMetaType::Patch:
                {
                    view.flag.Set<ApplicationViewFlag_HasPatchRecord>();
                    if (view.version < key.version)
                    {
                        view.version = key.version;
                    }
                }
                break;
            case ncm::ContentMetaType::AddOnContent:
                view.flag.Set<ApplicationViewFlag_HasAddOnContentRecord>(); break;
            default: break;
            }

            bool hasKey;
            NN_RESULT_DO(m_Integrated->HasContentMetaKey(&hasKey, key));
            if (hasKey)
            {
                if (record.storageId == ncm::StorageId::Card)
                {
                    view.flag.Set<ApplicationViewFlag_HasGameCardEntity>();
                }

                switch(key.type)
                {
                case ncm::ContentMetaType::Application:
                    {
                        view.flag.Set<ApplicationViewFlag_HasMainEntity>();
                        view.flag.Set<ApplicationViewFlag_IsLaunchable>();
                    }
                    break;
                case ncm::ContentMetaType::Patch:
                    view.flag.Set<ApplicationViewFlag_HasPatchEntity>(); break;
                case ncm::ContentMetaType::AddOnContent:
                    view.flag.Set<ApplicationViewFlag_HasAnyAddOnContentEntity>(); break;
                default: break;
                }
            }
            else
            {
                view.flag.Set<ApplicationViewFlag_HasAllEntity>(false);

                switch (key.type)
                {
                case ncm::ContentMetaType::AddOnContent:
                    view.flag.Set<ApplicationViewFlag_HasAllAddOnContentEntity>(false); break;
                default: break;
                }
            }
        }

        auto key = accessor.GetByIdFromData(id.value);
        view.flag.Set<ApplicationViewFlag_HasMainInstallRecord>(key && key->storageId != ncm::StorageId::Card);
        view.flag.Set<ApplicationViewFlag_MaybeCorrupted>(IsMaybeCorrupted(accessor.GetProperty().GetTerminateResult()));
        view.flag.Set<ApplicationViewFlag_IsCleanupAddOnContentWithNoRightsRecommended>(accessor.GetProperty().needsCleanupAoc);
        view.flag.Set<ApplicationViewFlag_IsPreInstalledApplication>(accessor.GetProperty().IsPreInstalledApplication());

        // Accessor でロックをしている間に、参照している Record を取得しておく
        NN_RESULT_DO(m_RecordDb->Get(outRecord, id));
    }

    bool isAutoDeleteDisabled;
    NN_RESULT_DO(m_RecordDb->IsAutoDeleteDisabled(&isAutoDeleteDisabled, id));
    view.flag.Set<ApplicationViewFlag_AutoDeleteDisabled>(isAutoDeleteDisabled);

    *outView = view;
    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::ReportThroughput(ApplicationInstallTask* task, ncm::ApplicationId applicationId, ncm::StorageContentMetaKey* keys, int count) NN_NOEXCEPT
{
    int64_t throughput;
    NN_RESULT_DO(task->GetThroughput(&throughput));

    for (int i = 0; i < count; ++i)
    {
        auto skey = keys[i];
        int64_t contentSize;
        NN_RESULT_DO(task->CalculateContentsSize(&contentSize, skey.key, skey.storageId));
        m_SystemReportManager->ReportApplicationInstallPerformance(applicationId, skey.key, skey.storageId, contentSize, throughput);
    }
    NN_RESULT_SUCCESS;
}

Result ApplicationViewManager::GetApplicationDownloadTaskStatus(sf::Out<ApplicationDownloadTaskStatus> outValue, ncm::ApplicationId id)
{
    auto stopper = m_CommitManager->GetCommitStopper();

    ApplicationInstallTask task(id);
    if (!task.IsValid())
    {
        *outValue = ApplicationDownloadTaskStatus::NotRunning;
        NN_RESULT_SUCCESS;
    }

    ApplicationDownloadProgress progress;
    NN_RESULT_DO(task.GetProgress(&progress));
    if (progress.state == ns::ApplicationDownloadState::Finished)
    {
        *outValue = ApplicationDownloadTaskStatus::Downloaded;
    }
    else if (progress.state == ns::ApplicationDownloadState::Runnable)
    {
        *outValue = ApplicationDownloadTaskStatus::Running;
    }
    else
    {
        *outValue = ApplicationDownloadTaskStatus::Fatal;
    }

    NN_RESULT_SUCCESS;
}

void ApplicationViewManager::InvalidateCache() NN_NOEXCEPT
{
    m_EntityViewCache.Clear();
}

}}}
