﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_StaticAssert.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/lr/lr_RegisteredLocationResolver.h>
#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_Service.h>
#include <nn/ncm/ncm_ContentInfoUtil.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_ApplicationManagerInterfaceServer.h>
#include <nn/ns/srv/ns_SaveDataInfo.h>
#include <nn/ns/detail/ns_ApplicationLanguage.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>

#include "ns_AsyncImpl.h"
#include "ns_ApplicationPathRedirectionUtil.h"
#include "ns_ApplicationApplyDeltaTask.h"
#include "ns_Config.h"
#include "ns_LogoFileUtil.h"
#include "ns_ApplicationInstallTask.h"
#include "ns_ApplicationLanguage.h"
#include "ns_GameCardStopperImpl.h"
#include "ns_ProgramIndexUtil.h"
#include "ns_RequestServerStopperImpl.h"
#include "ns_ShellUtil.h"
#include "ns_SystemUpdateUtil.h"

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(disable : 4344)
#endif

namespace nn { namespace ns { namespace srv {

namespace {
    const ncm::StorageId SupportedInstallStorages[] =
    {
        ncm::StorageId::BuildInUser,
        ncm::StorageId::SdCard
    };
    NN_STATIC_ASSERT(sizeof(SupportedInstallStorages) / sizeof(SupportedInstallStorages[0]) <= ApplicationOccupiedSize::MaxStorageCount);

    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_RequestServerStack[8192];
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_GameCardManagerStack[12 * 1024];
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_SdCardManagerStack[12 * 1024];
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_PushNotificationDispatcherStack[8 * 1024];
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_NotificationTaskManagerStack[8 * 1024];
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_RequestServerManagerStack[8 * 1024];

    pm::BootMode GetBootMode() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_HORIZON
        return pm::GetBootMode();
#else
        return pm::BootMode_Normal;
#endif
    }

    Result RedirectApplicationContentPath(
        bool* outHasRedirectPath,
        lr::Path* outPath,
        ncm::ProgramId id,
        ncm::ContentType type) NN_NOEXCEPT
    {
        lr::LocationResolver hostLocationResolver;
        NN_RESULT_DO(lr::OpenLocationResolver(&hostLocationResolver, ncm::StorageId::Host));

        *outHasRedirectPath = false;
        switch (type)
        {
        case ncm::ContentType::HtmlDocument:
            NN_RESULT_TRY(hostLocationResolver.ResolveApplicationHtmlDocumentPath(outPath, id))
                NN_RESULT_CATCH(lr::ResultHtmlDocumentNotFound){ break; }
            NN_RESULT_END_TRY
            *outHasRedirectPath = true;
            break;
        case ncm::ContentType::LegalInformation:
            NN_RESULT_TRY(hostLocationResolver.ResolveApplicationLegalInformationPath(outPath, id))
                NN_RESULT_CATCH(lr::ResultLegalInformationNotFound){ break; }
            NN_RESULT_END_TRY
            *outHasRedirectPath = true;
            break;
        default:
            break;
        }

        NN_RESULT_SUCCESS;
    }

    void AddOccupiedSize(ApplicationOccupiedSizeEntity* pRefEntity, ncm::ContentMetaType type, int64_t size) NN_NOEXCEPT
    {
        switch (type)
        {
        case ncm::ContentMetaType::Application:
            pRefEntity->appSize += size;
            break;
        case ncm::ContentMetaType::Patch:
            pRefEntity->patchSize += size;
            break;
        case ncm::ContentMetaType::AddOnContent:
            pRefEntity->aocSize += size;
            break;
        default:
            break;
        }
    }

    Result AddInstallTaskSize(ApplicationOccupiedSize* pRefValue, ApplicationInstallTask* pTask) NN_NOEXCEPT
    {
        if (!pTask->IsValid())
        {
            NN_RESULT_SUCCESS;
        }

        int count = 0;
        int offset = 0;
        ncm::InstallTaskOccupiedSize list[16];
        constexpr int ListCount = static_cast<int>(NN_ARRAY_SIZE(list));
        do
        {
            NN_RESULT_DO(pTask->ListOccupiedSize(&count, list, ListCount, offset));
            for (int i = 0; i < count; i++)
            {
                const auto& entry = list[i];
                const auto begin = pRefValue->storage;
                const auto end = pRefValue->storage + NN_ARRAY_SIZE(pRefValue->storage);
                auto entity = std::find_if(begin, end, [&entry](const ApplicationOccupiedSizeEntity& e) NN_NOEXCEPT { return e.storageId == entry.storageId; });
                if (entity == end)
                {
                    NN_DETAIL_NS_WARN("Install storage is not supported: %d\n", entry.storageId);
                    continue;
                }

                AddOccupiedSize(entity, entry.key.type, entry.size);
            }
            offset += count;
        } while (count == ListCount);

        NN_RESULT_SUCCESS;
    }

    Result AddApplyDeltaTaskSize(ApplicationOccupiedSizeEntity* pEntity, ApplicationApplyDeltaTask* pTask) NN_NOEXCEPT
    {
        if (!pTask->IsValid())
        {
            NN_RESULT_SUCCESS;
        }

        int64_t occupied;
        NN_RESULT_DO(pTask->CalculateOccupiedSize(&occupied, pEntity->storageId));
        pEntity->patchSize += occupied;

        NN_RESULT_SUCCESS;
    }
}

ApplicationManagerInterfaceServer::ApplicationManagerInterfaceServer() NN_NOEXCEPT :
    m_RequestList(&m_ControlDataManager, &m_DownloadTaskListManager, &m_ApplicationVersionManager, &m_CommitManager, &m_TicketManager, &m_RecordDb, &m_Sender),
    m_RequestServer(&m_RequestList, g_RequestServerStack, sizeof(g_RequestServerStack)),
    m_GameCardManager(g_GameCardManagerStack, sizeof(g_GameCardManagerStack)),
    m_SdCardManager(&m_Integrated, &m_RequestServer, &m_RecordDb, &m_ViewManager, g_SdCardManagerStack, sizeof(g_SdCardManagerStack)),
    m_PushNotificationDispatcher(g_PushNotificationDispatcherStack, sizeof(g_PushNotificationDispatcherStack), g_NotificationTaskManagerStack, sizeof(g_NotificationTaskManagerStack)),
    m_GameCardRegistrationManger(&m_GameCardManager),
    m_RequestServerManager(g_RequestServerManagerStack, sizeof(g_RequestServerManagerStack)),
    m_TicketManager(&m_Integrated, &m_RecordDb, &m_RequestServer),
    m_CommitManager(),
    m_ECommerceManager(&m_RequestServer),
    m_LockedBufferManager(m_LockedBufferPool, ApplicationManagerInterfaceServer::LockedBufferSize),
    m_PreInstallManager(&m_RecordDb),
    m_DynamicRightsManager(&m_RecordDb, &m_Integrated, &m_RequestServer, &m_RightsEnvironmentManager),
    m_IsContentsDeliveryDebugApiEnabled(false)
    {}

Result ApplicationManagerInterfaceServer::Initialize(RequestServer* requestServerForSystemUpdate, SystemReportManager* systemReportManager) NN_NOEXCEPT
{
    m_BootMode = GetBootMode();
    m_AppletLauncher.Initialize(m_BootMode);
    NN_RESULT_DO(ovln::InitializeSenderWithQueue(&m_Sender, MaxOverlayQueueCount));

    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        auto readSize = nn::settings::fwdbg::GetSettingsItemValue(&m_IsContentsDeliveryDebugApiEnabled, sizeof(m_IsContentsDeliveryDebugApiEnabled), "contents_delivery", "enable_debug_api");
        if (readSize != sizeof(m_IsContentsDeliveryDebugApiEnabled))
        {
            NN_DETAIL_NS_TRACE("[ApplicationManagerInterfaceServer] Failed to read contents_delivery/enable_debug_api flag from fwdbg setting.\n");
        }
#elif defined( NN_BUILD_CONFIG_OS_WIN )
        m_IsContentsDeliveryDebugApiEnabled = true;
#else
#error "Unsupported platform"
#endif
    }

    // メンテナンスモードでは AppletLauncher および ShutdownManager のみ初期化する
    if (IsMaintenanceMode() || IsSafeMode())
    {
        // ContentMetaDatabase や ContentStorage が壊れていたとしても、
        // メンテナンスモードが起動するように Result を無視する
        // ContentMetaDatabase や ContentStorage は起動時に InitializeBuildInContentManager() で作り直される
        auto storageId = ncm::StorageId::BuildInUser;
        if (ncm::VerifyContentMetaDatabase(storageId).IsSuccess() &&
            ncm::VerifyContentStorage(storageId).IsSuccess())
        {
            auto dbResult = ncm::ActivateContentMetaDatabase(storageId);
            auto storageResult = ncm::ActivateContentStorage(storageId);
            if (dbResult.IsFailure() || storageResult.IsFailure())
            {
                NN_DETAIL_NS_WARN("[ApplicationManagerInterfaceServer] Activate failed, db: 0x%08x, storage: 0x%08x\n", dbResult.GetInnerValueForDebug(), storageResult.GetInnerValueForDebug());
                ncm::InactivateContentMetaDatabase(storageId);
                ncm::InactivateContentStorage(storageId);
            }
        }

        m_ShutdownManager.InitializeForMaintenanceMode();
        m_FactoryResetManager.InitializeForMaintenanceMode();
        m_AppletLauncher.SetLaunchOverlayAppletEnabled(false);
        NN_RESULT_SUCCESS;
    }

    SaveDataInfo saveInfo = { ApplicationDataStoreMountName, ApplicationManagerSaveDataId, ApplicationManagerSaveDataSize, ApplicationManagerSaveDataJournalSize, ApplicationManagerSaveDataFlags };
    NN_RESULT_DO(m_DataStore.Initialize(saveInfo));
    NN_RESULT_DO(InitializeBuildInContentManager());

    m_SystemReportManager = systemReportManager;
    NN_RESULT_DO(m_RecordDb.Initialize(ApplicationRecordSaveDataId, &m_Integrated));
    m_ApplicationEntityManager.Initialize(&m_RecordDb, &m_Integrated, &m_RequestServer, m_SystemReportManager, &m_TicketManager);
    NN_RESULT_DO(m_SdCardManager.Initialize(saveInfo));

    NN_RESULT_DO(m_ShutdownManager.Initialize(&m_ApplicationEntityManager, &m_DataStore, &m_Integrated, &m_RequestServer, &m_RecordDb, &m_PushNotificationDispatcher, &m_LockedBufferManager, m_SdCardManager.NeedsSystemCleanup()));

    NN_RESULT_DO(m_ControlDataManager.Initialize(&m_RecordDb, &m_Integrated, &m_RequestServer));
    NN_RESULT_DO(m_DownloadTaskListManager.Initialize(&m_RecordDb));
    NN_RESULT_DO(m_BlackListManager.Initialize());
    NN_RESULT_DO(m_ApplicationVersionManager.Initialize(&m_RecordDb, &m_BlackListManager));
    m_ViewManager.Initialize(&m_RecordDb, &m_ApplicationEntityManager, &m_Integrated, &m_RequestServer, m_SystemReportManager, &m_CommitManager, &m_LockedBufferManager);
    m_RequestServer.Initialize(TimeSpan::FromSeconds(60));
    m_DownloadManager.Initialize(&m_RecordDb, &m_Integrated, &m_RequestServer, &m_LockedBufferManager);
    m_FactoryResetManager.Initialize(&m_RequestServer);
    NN_RESULT_DO(m_PushNotificationDispatcher.Initialize(NN_NS_PROGRAM_ID_OF_NS_PROCESS, &m_RequestList, &m_RequestServer, &m_VulnerabilityManager, &m_ApplicationVersionManager));
    m_ApplicationLaunchManager.Initialize(&m_RecordDb, &m_Integrated, &m_ApplicationVersionManager, &m_GameCardManager, &m_ControlDataManager, &m_TicketManager, &m_CommitManager);
    m_VulnerabilityManager.Initialize();
    NN_RESULT_DO(m_PseudoDeviceIdManager.Initialize());
    m_RequestServerManager.Initialize(&m_RequestServer, requestServerForSystemUpdate);

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    NN_RESULT_DO(m_GameCardManager.Initialize(&m_Integrated, &m_RecordDb, &m_ControlDataManager, &m_AppletLauncher));
    if (m_GameCardManager.NeedsRebootOnAutoBoot())
    {
        m_AppletLauncher.SetLaunchSystemAppletEnabled(false);
        m_AppletLauncher.SetLaunchOverlayAppletEnabled(false);
    }
    auto autoBoot = m_GameCardManager.GetApplicationIdForAutoBoot();
    if (autoBoot)
    {
        m_AppletLauncher.SetLaunchSystemAppletEnabled(false);
        m_AppletLauncher.SetLaunchOverlayAppletEnabled(false);

        os::ProcessId processId;
        auto result = m_ApplicationLaunchManager.LaunchApplicationForAutoBoot(&processId, *autoBoot);
        NN_UNUSED(processId);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[ApplicationManagerInterfaceServer] Faild to auto boot application\n");
        }
    }
#endif

    NN_RESULT_DO(m_ApplicationVerificationManager.Initialize(&m_Integrated, &m_RecordDb, &m_RequestServer, &m_TicketManager));

    m_CommitManager.Initialize(&m_RecordDb, &m_ApplicationEntityManager, &m_Integrated, m_SystemReportManager, &m_Sender);

    NN_RESULT_DO(m_ContentDeliveryManager.Initialize(&m_RecordDb, &m_ApplicationVersionManager, &m_BlackListManager, &m_CommitManager, &m_GameCardManager, &m_Integrated, &m_RequestServer));

    m_ApplicationCopyIdentifierManager.Initialize(&m_GameCardManager);

    m_SystemReportManager->SetSdCardReportEnabled(m_SdCardManager.IsAttached());
    m_SystemReportManager->ReportSdCardInfo();
    m_SystemReportManager->ReportSdCardFreeSpace();

    NN_RESULT_SUCCESS;
}

ApplicationManagerInterfaceServer::~ApplicationManagerInterfaceServer() NN_NOEXCEPT
{
    if (!IsMaintenanceMode() && !IsSafeMode())
    {
        m_RequestServerManager.Finalize();
        m_VulnerabilityManager.Finalize();
        m_PushNotificationDispatcher.Finalize();
        m_ApplicationVersionManager.Finalize();
        m_RequestServer.Finalize();
        m_ControlDataManager.Finalize();
        m_RecordDb.Finalize();
        m_SdCardManager.Finalize();
    }

    m_AppletLauncher.Finalize();
}

Result ApplicationManagerInterfaceServer::CalculateApplicationOccupiedSize(sf::Out<ApplicationOccupiedSize> outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationOccupiedSize occupiedSize = {};
    int storageCount = 0;

    {
        ApplicationRecordAccessor list;
        NN_RESULT_DO(m_RecordDb.Open(&list, id));

        for (auto& storageId : SupportedInstallStorages)
        {
            if (m_Integrated.IsRegisteredStorage(storageId))
            {
                occupiedSize.storage[storageCount++].storageId = storageId;
            }
        }

        // インストールされている ContentMetaKey が必要なので、一時記録ではなく、記録の本体を見に行く
        for (int i = 0; i < list.GetDataCount(); i++)
        {
            const auto& key = list.GetDataAt(i).key;
            for (int j = 0; j < storageCount; j++)
            {
                int64_t occupiedSizeOne;
                NN_RESULT_DO(m_Integrated.CalculateOccupiedSize(&occupiedSizeOne, key, occupiedSize.storage[j].storageId));
                AddOccupiedSize(&occupiedSize.storage[j], key.type, occupiedSizeOne);
            }
        }
    }

    // 各タスクの消費サイズを計上する
    ApplicationInstallTask installTask(id);
    ApplicationApplyDeltaTask applyDeltaTask(id);

    if (installTask.IsValid() || applyDeltaTask.IsValid())
    {
        auto stopper = m_RequestServer.Stop();
        {
            // 止めた後にタスクは取得し直す
            ApplicationInstallTask task(id);
            if (task.IsValid())
            {
                NN_RESULT_DO(AddInstallTaskSize(&occupiedSize, &task));
            }
        }
        {
            ApplicationApplyDeltaTask task(id);
            if (task.IsValid())
            {
                for (int i = 0; i < storageCount; i++)
                {
                    NN_RESULT_DO(AddApplyDeltaTaskSize(&occupiedSize.storage[i], &task));
                }
            }
        }
    }

    outValue.Set(occupiedSize);

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::IsSystemProgramInstalled(sf::Out<bool> outValue, ncm::SystemProgramId id) NN_NOEXCEPT
{
    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));
    ncm::ContentId contentId;
    NN_RESULT_TRY(db.GetLatestProgram(&contentId, id))
        NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH(ncm::ResultContentNotFound)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetSystemUpdateContentPath(ncm::Path* outValue) NN_NOEXCEPT
{
    auto systemUpdateMetaKey = GetSystemUpdateMetaKey();
    NN_RESULT_THROW_UNLESS(systemUpdateMetaKey, ResultSystemUpdateMetaKeyNotFound());

    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));

    ncm::ContentId contentId;
    db.GetContentIdByType(&contentId, *systemUpdateMetaKey, ncm::ContentType::Data);

    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::BuildInSystem));

    bool exist;
    NN_RESULT_DO(storage.Has(&exist, contentId));
    NN_RESULT_THROW_UNLESS(exist, ResultSystemUpdateMetaContentNotFound());

    storage.GetPath(outValue, contentId);

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetApplicationContentPath(sf::Out<ncm::Path> outValue, ncm::ApplicationId id, ncm::ContentType type) NN_NOEXCEPT
{
    // TORIAEZU: DeveloperInterface 経由の場合は Redirect されているはずなので、Redirect を優先して利用するようにする
    //           この関数は LegalInformation の解決のために呼び出されるので、 DeveloperInterface 経由の場合は、 ほぼ使われないので、 ProgramIndex == 0 の時のみをサポートするとする
    bool hasHostRedirection;
    lr::Path path;
    NN_RESULT_DO(RedirectApplicationContentPath(&hasHostRedirection, &path, GetProgramId(id, 0), type));
    if (hasHostRedirection)
    {
        util::Strlcpy(outValue.GetPointer()->string, path.string, sizeof(outValue.GetPointer()->string));
        NN_RESULT_SUCCESS;
    }

    util::optional<ncm::ContentMetaKey> found;
    NN_RESULT_DO(m_RecordDb.FindLaunchableMain(&found, id));
    NN_RESULT_THROW_UNLESS(found, ResultApplicationContentNotFound());

    uint8_t programIndex;
    NN_RESULT_DO(m_Integrated.GetMinimumProgramIndex(&programIndex, *found));
    ncm::ContentId contentId;
    NN_RESULT_TRY(m_Integrated.GetContentId(&contentId, *found, type, programIndex))
        NN_RESULT_CATCH(ncm::ResultContentNotFound) { NN_RESULT_THROW(ResultApplicationContentNotFound()); }
    NN_RESULT_END_TRY

    m_Integrated.GetContentPath(outValue.GetPointer(), contentId);

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::ResolveApplicationContentPath(ncm::ProgramId id, ncm::ContentType type) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(type == ncm::ContentType::HtmlDocument, ResultUnsupportedContentType());

    arp::ApplicationLaunchProperty property;
    NN_RESULT_DO(GetCurrentApplicationLaunchProperty(&property));
    NN_RESULT_THROW_UNLESS(GetProgramId(property.id, property.programIndex) == id, ResultUnexpectedProgramIndex());

    bool hasHostRedirection;
    {
        lr::Path path;
        NN_RESULT_DO(RedirectApplicationContentPath(&hasHostRedirection, &path, id, type));
        NN_UNUSED(path);
    }
    if (hasHostRedirection)
    {
        // TORIAEZU: Host 起動なので、バージョンは 0 と仮定
        ncm::ContentMetaKey key = ncm::ContentMetaKey::Make(id.value, 0, ncm::ContentMetaType::Application);
        NN_RESULT_DO(RegisterHtmlDocumentPatch(key, id, property.programIndex));

        NN_RESULT_SUCCESS;
    }
    else
    {
        util::optional<ncm::ContentMetaKey> appKey;
        util::optional<ncm::ContentMetaKey> patchKey;
        NN_RESULT_DO(m_RecordDb.FindLaunchableApplicationAndPatch(&appKey, &patchKey, property.id));
        NN_RESULT_THROW_UNLESS(appKey, ResultApplicationContentNotFound());

        // アプリケーションのコンテンツパスを lr に登録
        // パッチにあって、アプリにはないというケースにも対応する
        bool found = true;
        ncm::ContentId contentId;
        NN_RESULT_TRY(m_Integrated.GetContentId(&contentId, *appKey, type, property.programIndex))
            NN_RESULT_CATCH(ncm::ResultContentNotFound) { found = false; }
        NN_RESULT_END_TRY

        ncm::StorageId storageId;
        NN_RESULT_DO(m_Integrated.GetContentMetaStorage(&storageId, *appKey));

        if (found)
        {
            ncm::Path tmpPath;
            lr::Path path;
            NN_RESULT_DO(m_Integrated.GetContentPath(&tmpPath, contentId));
            std::memcpy(path.string, tmpPath.string, sizeof(path.string));

            {
                lr::LocationResolver locationResolver;
                NN_RESULT_DO(lr::OpenLocationResolver(&locationResolver, storageId));
                locationResolver.RedirectApplicationHtmlDocumentPath(id, path);
            }
        }
        else
        {
            lr::LocationResolver locationResolver;
            NN_RESULT_DO(lr::OpenLocationResolver(&locationResolver, storageId));
            NN_RESULT_DO(locationResolver.EraseApplicationHtmlDocumentRedirection(id));
        }

        auto mainKey = patchKey ? patchKey : appKey;

        // コンテンツの存在確認
        // アプリにあって、パッチにないというケースの確認が必要なのでここで行う
        NN_RESULT_TRY(m_Integrated.GetContentId(&contentId, *mainKey, type, property.programIndex))
            NN_RESULT_CATCH_CONVERT(ncm::ResultContentNotFound, ResultApplicationContentNotFound())
        NN_RESULT_END_TRY

        // パッチの登録
        NN_RESULT_DO(RegisterHtmlDocumentPatch(*mainKey, id, property.programIndex));

        NN_RESULT_SUCCESS;
    }
}

Result ApplicationManagerInterfaceServer::TerminateApplication(os::ProcessId pid) NN_NOEXCEPT
{
    return srv::TerminateProcess(pid);
}

Result ApplicationManagerInterfaceServer::BeginInstallApplication(ncm::ApplicationId id, std::uint32_t version, ncm::StorageId storage) NN_NOEXCEPT
{
    auto managedStop = m_RequestServer.Stop();

    auto key = ncm::ContentMetaKey::Make(id, version);

    nim::NetworkInstallTaskId taskId;
    NN_RESULT_DO(nim::CreateNetworkInstallTask(&taskId, id, &key, 1, storage));

    NN_RESULT_DO(m_RecordDb.Push(id, ApplicationEvent::DownloadStarted, nullptr, 0));

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::PushDownloadTaskList(const sf::InBuffer& buffer) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsContentsDeliveryDebugApiEnabled(), ResultNotPermittedOnProduction());

    auto managedStop = m_RequestServer.Stop();

    NN_RESULT_DO(m_DownloadTaskListManager.Push(buffer.GetPointerUnsafe(), buffer.GetSize()));

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::InitializeBuildInContentManager() NN_NOEXCEPT
{
    auto storageId = ncm::StorageId::BuildInUser;

    auto result = ncm::VerifyContentMetaDatabase(storageId);
    if (result.IsFailure())
    {
        NN_RESULT_DO(ncm::CreateContentMetaDatabase(storageId));
    }

    result = ncm::VerifyContentStorage(storageId);
    if (result.IsFailure())
    {
        NN_RESULT_DO(ncm::CreateContentStorage(storageId));
    }

    bool isSuccess = false;
    NN_RESULT_DO(ncm::ActivateContentStorage(storageId));
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            // Result を無視
            ncm::InactivateContentStorage(storageId);
        }
    };
    NN_RESULT_DO(ncm::ActivateContentMetaDatabase(storageId));
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            // Result を無視
            ncm::InactivateContentMetaDatabase(storageId);
        }
    };

    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));
    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, storageId));

    m_Integrated.Register(storageId, std::move(db), std::move(storage));

    isSuccess = true;
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetApplicationLogoData(nn::sf::Out<std::int64_t> outSize, const nn::sf::OutBuffer& outBuffer, nn::ncm::ApplicationId id, const nn::ncm::Path& logoPath) NN_NOEXCEPT
{
    util::optional<ncm::ContentMetaKey> mainMeta, patchMeta;
    NN_RESULT_DO(m_RecordDb.FindLaunchableApplicationAndPatch(&mainMeta, &patchMeta, id));
    NN_RESULT_THROW_UNLESS(mainMeta, ResultApplicationContentNotFound());

    // LOGO はどのサブプログラムであっても同じデータを持っているはずなので、最初に見つかるプログラムのものを利用する
    uint8_t programIndex;
    NN_RESULT_DO(m_Integrated.GetMinimumProgramIndex(&programIndex, *mainMeta));

    ncm::ContentId contentId;
    NN_RESULT_TRY(m_Integrated.GetContentId(&contentId, *mainMeta, ncm::ContentType::Program, programIndex))
        NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
        {
            NN_RESULT_THROW(ResultApplicationContentNotFound());
        }
    NN_RESULT_END_TRY;

    ncm::Path contentPath;
    NN_RESULT_TRY(m_Integrated.GetContentPath(&contentPath, contentId))
        NN_RESULT_CATCH(ncm::ResultContentNotFound)
        {
            NN_RESULT_THROW(ResultApplicationContentNotFound());
        }
    NN_RESULT_END_TRY;

    // GetApplicationLogoData が LaunchApplication よりも先に呼び出される可能性があるので、
    // こちらでも外部鍵の登録をしておく
    NN_RESULT_DO(m_ApplicationLaunchManager.RegisterLogoDataExternalKey(*mainMeta, programIndex));

    size_t readSize;
    NN_RESULT_DO(ReadLogoData(&readSize, outBuffer.GetPointerUnsafe(), outBuffer.GetSize(), GetProgramId(id, programIndex), contentPath, logoPath));

    *outSize = readSize;
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetContentPath(lr::Path* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex) NN_NOEXCEPT
{
    ncm::ContentId contentId;
    ncm::Path path;
    NN_RESULT_TRY(m_Integrated.GetContentId(&contentId, key, type, programIndex))
        NN_RESULT_CATCH(ncm::ResultContentNotFound) { NN_RESULT_THROW(ResultApplicationContentNotFound()); }
    NN_RESULT_END_TRY
    NN_RESULT_DO(m_Integrated.GetContentPath(&path, contentId));

    // ncm と lr のパスは同じなのだがコピーが要るのが残念
    std::memcpy(outValue->string, path.string, sizeof(lr::Path));

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::RegisterHtmlDocumentPatch(const ncm::ContentMetaKey& key, ncm::ProgramId id, uint8_t programIndex) NN_NOEXCEPT
{
    // 毎回取得せずに、メンバ変数として持ってもいいが、ひとまず
    lr::RegisteredLocationResolver locationResolver;
    NN_RESULT_DO(lr::OpenRegisteredLocationResolver(&locationResolver));

    // lr へパッチ情報の登録
    if (key.type == ncm::ContentMetaType::Patch)
    {
        lr::Path lrPath;
        NN_RESULT_DO(GetContentPath(&lrPath, key, ncm::ContentType::HtmlDocument, programIndex));
        NN_RESULT_DO(locationResolver.RegisterHtmlDocumentPath(id, lrPath));
        NN_DETAIL_NS_TRACE("Register Html Document Patch of %s\n", lrPath.string);
    }
    else
    {
        NN_RESULT_DO(locationResolver.UnregisterHtmlDocumentPath(id));
    }
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::StartApplyDeltaTask(ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationApplyDeltaTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), ResultApplyDeltaTaskNotFound());

    ApplicationApplyDeltaProgress progress;
    NN_RESULT_DO(task.GetProgress(&progress));
    NN_RESULT_THROW_UNLESS(progress.state == ApplicationApplyDeltaState::WaitApply, ResultApplyDeltaTaskAlreadyRunning());

    NN_RESULT_DO(task.StartApplyDeltaTask());

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetTotalSpaceSize(sf::Out<std::int64_t> outSize, ncm::StorageId storageId) NN_NOEXCEPT
{
    if (storageId == ncm::StorageId::SdCard && !m_SdCardManager.IsAttached())
    {
        NN_RESULT_THROW(ResultStorageAccessFailed());
    }
    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, storageId));

    int64_t size;
    NN_RESULT_DO(storage.GetTotalSpaceSize(&size));
    outSize.Set(size);

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetFreeSpaceSize(sf::Out<std::int64_t> outSize, ncm::StorageId storageId) NN_NOEXCEPT
{
    if (storageId == ncm::StorageId::SdCard && !m_SdCardManager.IsAttached())
    {
        NN_RESULT_THROW(ResultStorageAccessFailed());
    }
    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, storageId));

    int64_t size;
    NN_RESULT_DO(storage.GetFreeSpaceSize(&size));
    outSize.Set(size);

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetApplicationDesiredLanguage(nn::sf::Out<nn::Bit8> outValue, Bit32 supportedLanguageFlag) NN_NOEXCEPT
{
    settings::LanguageCode code;
    settings::GetLanguageCode(&code);

    auto language = ToApplicationLanguage(code);
    NN_RESULT_THROW_UNLESS(language, ResultApplicationLanguageNotFound());
    auto languageList = GetLanguagePriorityList(*language);
    NN_RESULT_THROW_UNLESS(languageList, ResultApplicationLanguageNotFound());

    for (int i = 0; i < LanguagePriorityListLength; i++)
    {
        auto flag = ns::detail::ToSupportedLanguageFlag(languageList[i]);
        if (supportedLanguageFlag == 0 || (supportedLanguageFlag & flag) == flag)
        {
            outValue.Set(static_cast<Bit8>(languageList[i]));
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultApplicationLanguageNotFound());
}

Result ApplicationManagerInterfaceServer::SetApplicationTerminateResult(ncm::ApplicationId id, uint32_t result) NN_NOEXCEPT
{
    return m_RecordDb.SetApplicationTerminateResult(id, result::detail::ConstructResult(result));
}

Result ApplicationManagerInterfaceServer::ClearApplicationTerminateResult(ncm::ApplicationId id) NN_NOEXCEPT
{
    return m_RecordDb.ClearApplicationTerminateResult(id);
}

Result ApplicationManagerInterfaceServer::GetApplicationTerminateResult(sf::Out<uint32_t> outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    Result result;
    NN_RESULT_DO(m_RecordDb.GetApplicationTerminateResult(&result, id, false));
    *outValue = result.GetInnerValueForDebug();
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetRawApplicationTerminateResult(sf::Out<uint32_t> outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    Result result;
    NN_RESULT_DO(m_RecordDb.GetApplicationTerminateResult(&result, id, true));
    *outValue = result.GetInnerValueForDebug();
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::ConvertApplicationLanguageToLanguageCode(nn::sf::Out<settings::LanguageCode> outValue, Bit8 language) NN_NOEXCEPT
{
    auto code = MakeLanguageCode(static_cast<ns::detail::ApplicationLanguage>(language));
    NN_RESULT_THROW_UNLESS(code, ResultApplicationLanguageNotFound());

    outValue.Set(*code);
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::ConvertLanguageCodeToApplicationLanguage(nn::sf::Out<Bit8> outValue, settings::LanguageCode code) NN_NOEXCEPT
{
    auto language = ToApplicationLanguage(code);
    NN_RESULT_THROW_UNLESS(language, ResultApplicationLanguageNotFound());

    outValue.Set(static_cast<Bit8>(*language));

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetBackgroundDownloadStressTaskInfo(nn::sf::Out<nim::BackgroundDownloadStressTaskInfo> outValue) NN_NOEXCEPT
{
    nim::BackgroundDownloadStressTaskInfo info;
    NN_RESULT_DO(nim::GetBackgroundDownloadStressTaskInfo(&info));
    outValue.Set(info);
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetBackgroundApplyDeltaStressTaskInfo(nn::sf::Out<nim::BackgroundDownloadStressTaskInfo> outValue) NN_NOEXCEPT
{
    nim::BackgroundDownloadStressTaskInfo info;
    NN_RESULT_DO(nim::GetBackgroundApplyDeltaStressTaskInfo(&info));
    outValue.Set(info);
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::CountApplicationContentMeta(sf::Out<std::int32_t> outValue, ncm::ApplicationId appId) NN_NOEXCEPT
{
    ApplicationRecordAccessor list;
    NN_RESULT_DO(m_RecordDb.Open(&list, appId));
    *outValue = list.Count();
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetOwnedApplicationContentMetaStatus(sf::Out<ns::ApplicationContentMetaStatus> outValue, ncm::ApplicationId appId, nn::Bit64 contentMetaId) NN_NOEXCEPT
{
    ApplicationRecordAccessor list;
    NN_RESULT_DO(m_RecordDb.Open(&list, appId));

    // 非永続化対象( GameCard etc )レコードも含めて探す.
    auto meta = list.GetById(contentMetaId);
    NN_RESULT_THROW_UNLESS(meta, ns::ResultNoContentMetaInApplicationRecord());

    // 非永続化コンテンツのストレージも検索対象に含める.
    ncm::StorageId installed;
    NN_RESULT_DO(m_Integrated.GetContentMetaStorage(&installed, meta->key));

    NN_RESULT_DO(m_ApplicationLaunchManager.CheckContentsRights(meta->key, installed));

    ApplicationContentMetaStatus status = {};
    status.id = meta->key.id;
    status.type = meta->key.type;
    status.version = meta->key.version;
    status.installedStorage = installed;

    outValue.Set(status);
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::RegisterContentsExternalKey(ncm::ApplicationId appId, nn::Bit64 contentMetaId) NN_NOEXCEPT
{
    ApplicationRecordAccessor list;
    NN_RESULT_DO(m_RecordDb.Open(&list, appId));

    // 非永続化対象( GameCard etc )レコードも含めて探す.
    auto meta = list.GetById(contentMetaId);
    if (meta && ncm::StorageId::GameCard == meta->storageId)
    {
        // OnGameCardコンテンツはパッチを除いて内部鍵想定のため外部鍵登録シーケンスに移行せず成功扱い.
        // 尚、OnCardPatch の外部鍵登録はこのシーケンスを経由しない.
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(meta, ns::ResultNoContentMetaInApplicationRecord());

    ncm::StorageId installed;
    NN_RESULT_DO(m_Integrated.GetInstalledContentMetaStorage(&installed, meta->key));

    NN_RESULT_DO(m_ApplicationLaunchManager.RegisterContentsExternalKey(meta->key, installed));

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetRequestServerStopper(sf::Out<sf::SharedPointer<ns::detail::IRequestServerStopper>> outValue) NN_NOEXCEPT
{
    auto emplacedRef = sf::CreateSharedObjectEmplaced<ns::detail::IRequestServerStopper, RequestServerStopperImpl>(m_RequestServer.Stop());
    *outValue = emplacedRef;

    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetStorageSize(sf::Out<std::int64_t> outTotalSize, sf::Out<std::int64_t> outFreeSpaceSize, ncm::StorageId storageId) NN_NOEXCEPT
{
    fs::ContentStorageId id;
    switch(storageId)
    {
    case ncm::StorageId::BuiltInUser:
        id = fs::ContentStorageId::User;
        break;
    case ncm::StorageId::SdCard:
        id = fs::ContentStorageId::SdCard;
        break;
    default:
        NN_RESULT_THROW(ResultUnsupportedStorage());
    }

    const char* mountName = "nssize";
    const char* rootPath = "nssize:/";
    NN_RESULT_DO(fs::MountContentStorage(mountName, id));
    NN_UTIL_SCOPE_EXIT
    {
        fs::Unmount(mountName);
    };

    int64_t totalSize;
    NN_RESULT_DO(fs::GetTotalSpaceSize(&totalSize, rootPath));
    int64_t freeSize;
    NN_RESULT_DO(fs::GetFreeSpaceSize(&freeSize, rootPath));

    *outTotalSize = totalSize;
    *outFreeSpaceSize = freeSize;
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetRunningApplicationProgramId(sf::Out<ncm::ProgramId> outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    return srv::GetRunningApplicationProgramId(outValue.GetPointer(), id);
}

Result ApplicationManagerInterfaceServer::GetMainApplicationProgramIndex(sf::Out<uint8_t> outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    ApplicationLaunchInfo info;
    NN_RESULT_DO(m_ApplicationLaunchManager.GetApplicationLaunchInfo(&info, id));
    NN_RESULT_DO(GetMainApplicationProgramIndexByApplicationLaunchInfo(outValue, info));
    NN_RESULT_SUCCESS;
}

Result ApplicationManagerInterfaceServer::GetMainApplicationProgramIndexByApplicationLaunchInfo(sf::Out<uint8_t> outValue, const ApplicationLaunchInfo& info) NN_NOEXCEPT
{
    if ((info.launchInfoFlags & static_cast<Bit8>(ApplicationLaunchInfoFlag::AutoBootByGameCard)) == static_cast<Bit8>(ApplicationLaunchInfoFlag::AutoBootByGameCard))
    {
        NN_RESULT_THROW_UNLESS(info.applicationStorageId == ncm::StorageId::GameCard, ResultApplicationContentNotFound());
        *outValue = 0; // AutoBoot のアプリはサブプログラムを入れる想定はない
    }
    else if (info.applicationStorageId == ncm::StorageId::Host)
    {
        NN_RESULT_THROW_UNLESS(info.patchStorageId == ncm::StorageId::None, ResultApplicationContentNotFound());
        NN_RESULT_DO(m_ApplicationLaunchManager.GetProgramIndexForDevelop(outValue.GetPointer(), info));
    }
    else
    {
        util::optional<ncm::ContentMetaKey> found;
        NN_RESULT_DO(m_RecordDb.FindLaunchableMain(&found, info.id));
        NN_RESULT_THROW_UNLESS(found, ResultApplicationContentNotFound());

        NN_RESULT_DO(m_Integrated.GetMinimumProgramIndex(outValue.GetPointer(), *found));
    }

    NN_RESULT_SUCCESS;
}
}}}
