﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <limits>
#include <mutex>

#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_ApplicationEntityViewCache.h>

namespace nn { namespace ns { namespace srv {

Result ApplicationEntityViewCacheHolder::LoadEntityViewCache(ApplicationView* outValue, ncm::ApplicationId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CacheMutex);

    ApplicationRecord record;
    NN_RESULT_DO(m_RecordDatabase->Get(&record, id));

    auto entry = std::find_if(m_Cache, m_Cache + CacheEntryCount, [&id] (const ApplicationEntityViewCache& cache) NN_NOEXCEPT { return cache.applicationId == id; } );

    NN_RESULT_THROW_UNLESS(entry != m_Cache + CacheEntryCount, ResultEntityViewCacheNotFound());
    NN_RESULT_THROW_UNLESS(entry->lastUpdated == record.lastUpdated, ResultEntityViewCacheOutDated());

    *outValue = entry->entityView;
    entry->cacheUsedCounter = m_CacheUsedCounter;
    m_CacheUsedCounter++;

    NN_RESULT_SUCCESS;
}
void ApplicationEntityViewCacheHolder::StoreEntityViewCache(ncm::ApplicationId id, int64_t lastUpdated, const ApplicationView& view) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CacheMutex);

    // まずは同一 ID のエントリを探す
    auto entry = std::find_if(m_Cache, m_Cache + CacheEntryCount, [&id] (const ApplicationEntityViewCache& cache) NN_NOEXCEPT {return cache.applicationId == id;} );

    // 同一 ID のエントリが見つからなかった場合、cacheUsedCounter を用いて LLU で追い出す
    // 初期化時に cacheUsedCounter は 0 にしてあるため、未利用のエントリもこれで見つかる
    if (entry == m_Cache + CacheEntryCount)
    {
        int64_t minimumCounter = std::numeric_limits<int64_t>::max();
        for (auto& cache : m_Cache)
        {
            if (cache.cacheUsedCounter < minimumCounter)
            {
                minimumCounter = cache.cacheUsedCounter;
                entry = &cache;
            }
        }
    }

    entry->applicationId = id;
    entry->lastUpdated = lastUpdated;
    entry->cacheUsedCounter = m_CacheUsedCounter;
    entry->entityView = view;
    m_CacheUsedCounter++;
}

}}}

