﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nsd/detail/server/nsd_ServerManager.h>
#include <nn/nsd/detail/server/nsd_ManagerImpl.h>
#include <nn/nsd/detail/nsd_ServiceObjectDfc.h>
#include <nn/nsd/detail/nsd_FqdnResolver.h>
#include <nn/nsd/detail/nsd_Log.h>
#include <nn/nsd/detail/fs/nsd_Fs.h>
#include <nn/nsd/nsd_ServiceName.h>
#include <nn/nsd/detail/nsd_DetailApi.h>
#include <nn/nsd/detail/device/nsd_Device.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/os/os_Thread.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_Assert.h>

namespace {

    // 管理する最大セッション数
    const int SessionCountForUser  = 20;
    const int SessionCountForAdmin = 5;
    const int SessionCountMax      = SessionCountForUser + SessionCountForAdmin;

    // NSD サーバマネージャ
    class NsdServerManager : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax * 2, 2> // 最大sessionカウント多めにしています
    {
    };
    NsdServerManager g_NsdServerManager;

    // スタック
    const size_t NsdStackSize = 16 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_NsdServerStack[NsdStackSize];

    // スレッド
    nn::os::ThreadType g_NsdServerThread;

    // システムセーブデータ から読み込む NSD設定
    nn::nsd::SaveData g_NsdSaveData;
}

namespace nn { namespace nsd { namespace detail { namespace server {

// IPC サーバの開始
void InitializeIpcServer() NN_NOEXCEPT
{
    // サービス名の登録とポートの初期化
    // for nsd:a
    nn::sf::SharedPointer<IManager> serviceAdmin;
    nn::nsd::detail::GetServiceObjectByDfc( &serviceAdmin, nn::nsd::ServiceType_Admin );
    g_NsdServerManager.RegisterObjectForPort(serviceAdmin, SessionCountForAdmin, nn::nsd::ServiceNameForAdmin);
    // for nsd:u
    nn::sf::SharedPointer<IManager> serviceUser;
    nn::nsd::detail::GetServiceObjectByDfc( &serviceUser, nn::nsd::ServiceType_User );
    g_NsdServerManager.RegisterObjectForPort(serviceUser, SessionCountForUser, nn::nsd::ServiceNameForUser);

    // 開始
    g_NsdServerManager.Start();
}

// IPC サーバのループ
void LoopIpcServer(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    // セーブデータ設定の読み込み
    auto result = nn::nsd::detail::fs::ReadSaveData( &g_NsdSaveData );

    // ファイルが存在しない場合、デフォルト設定とする
    if ( nn::nsd::ResultNotFound::Includes( result ) )
    {
        NN_DETAIL_NSD_INFO("[NSD] No system save data.\n");
        DeviceId ownDeviceId;
        detail::device::LoadOwnDeviceId(&ownDeviceId);
        const EnvironmentIdentifier env = nn::nsd::detail::device::GetEnvironmentIdentifierFromSettings();
        detail::GetDefaultSettings( &g_NsdSaveData, ownDeviceId, env );

        // FqdnResolver へセット
        GetFqdnResolverPtr()->SetSaveData( &g_NsdSaveData );
        GetFqdnResolverPtr()->SetFatalError( nn::ResultSuccess() );
        GetFqdnResolverPtr()->SetSettingPresent(false);
    }
    else if ( result.IsFailure() )
    {
        NN_DETAIL_NSD_ERROR("[NSD] Fatal error occurred.\n");
        GetFqdnResolverPtr()->SetSaveData( &g_NsdSaveData );

        // FqdvResolver に以降の失敗を伝える
        GetFqdnResolverPtr()->SetFatalError( result );
        GetFqdnResolverPtr()->SetSettingPresent(false);
    }
    else
    {
        NN_DETAIL_NSD_INFO("[NSD] Read system save data. (isAvailable=%d)\n", g_NsdSaveData.isAvailable );

        GetFqdnResolverPtr()->SetSaveData( &g_NsdSaveData );
        GetFqdnResolverPtr()->SetFatalError( nn::ResultSuccess() );
        GetFqdnResolverPtr()->SetSettingPresent(true);
    }

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    //  ポートへの接続要求 / セッションへの処理要求 を行う
    g_NsdServerManager.LoopAuto();
#endif
}

//----------------------------------------------------------------
// NSD の IPC サーバ起動
nn::Result StartServer() NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    // IPC サーバの開始
    InitializeIpcServer();

    // 処理スレッドの作成・開始
    auto createThreadResult = nn::os::CreateThread( &g_NsdServerThread,
                                                    LoopIpcServer,
                                                    nullptr,
                                                    g_NsdServerStack,
                                                    sizeof(g_NsdServerStack),
                                                    NN_SYSTEM_THREAD_PRIORITY( nsd, IpcServer ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( createThreadResult );

    // スレッド名
    nn::os::SetThreadNamePointer( &g_NsdServerThread,
                                  NN_SYSTEM_THREAD_NAME( nsd, IpcServer ) );

    nn::os::StartThread( &g_NsdServerThread );
    NN_DETAIL_NSD_INFO( "[NSD] Start NSD Server\n" );
#else
    LoopIpcServer(NULL);
#endif

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// NSD の IPC サーバ停止
nn::Result StopServer() NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    // 処理ループの停止リクエスト送信
    g_NsdServerManager.RequestStop();

    nn::os::WaitThread(&g_NsdServerThread);
    nn::os::DestroyThread(&g_NsdServerThread);
#endif

    NN_RESULT_SUCCESS;
}

}}}} // nn::nsd::detail::server
